/*
 * xtl_logger_stdio.c
 *
 * log message consumer that writes to stdio
 *
 * Copyright (c) 2010 Citrix
 * Part of a generic logging interface used by various dom0 userland libraries.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "xentoollog.h"

#include <time.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <syslog.h>
#include <stdio.h>

#ifndef __UCLIBC__
extern char *program_invocation_name;
#endif

struct xentoollog_logger_stdiostream {
    xentoollog_logger vtable;
    FILE *f;
    xentoollog_level min_level;
    unsigned flags;
    int progress_erase_len, progress_last_percent;
};

static void progress_erase(xentoollog_logger_stdiostream *lg) {
    if (lg->progress_erase_len)
        fprintf(lg->f, "\r%*s\r", lg->progress_erase_len, "");
}

#define SL_BUF_SIZE 200
static char sl_buf[SL_BUF_SIZE];
static int sl_index=0;

static void slflush(FILE *stream)
{
    if ( sl_index )
    {
        syslog(LOG_DAEMON | LOG_INFO, "%s", sl_buf);
        sl_index = 0;
    }
    if ( stream )
        fflush(stream);
}

static int vslprintf(FILE *stream, const char *fmt, va_list ap)
{
    int old_sl_index = sl_index;

    sl_index += vsnprintf(sl_buf + sl_index, SL_BUF_SIZE-sl_index, fmt, ap);

    if ( sl_index > SL_BUF_SIZE )
        slflush(NULL);

    return fwrite(sl_buf + old_sl_index, sl_index - old_sl_index, 1, stream);
}

static int slprintf(FILE *stream, const char *fmt, ...)
{
    va_list ap;
    int ret;
    va_start(ap, fmt);
    ret = vslprintf(stream, fmt, ap);
    va_end(ap);
    return ret;
}

static void stdiostream_vmessage(xentoollog_logger *logger_in,
                                 xentoollog_level level,
                                 int errnoval,
                                 const char *context,
                                 const char *format,
                                 va_list al) {
    xentoollog_logger_stdiostream *lg = (void*)logger_in;

    if (level < lg->min_level)
        return;

    progress_erase(lg);

    if (lg->flags & XTL_STDIOSTREAM_SHOW_DATE) {
        struct tm lt_buf;
        time_t now = time(0);
        struct tm *lt= localtime_r(&now, &lt_buf);
        slprintf(lg->f, "%04d-%02d-%02d %02d:%02d:%02d %s ",
                lt->tm_year+1900, lt->tm_mon+1, lt->tm_mday,
                lt->tm_hour, lt->tm_min, lt->tm_sec,
                tzname[daylight ? !!lt->tm_isdst : 0]);
    }
    if (lg->flags & XTL_STDIOSTREAM_SHOW_PID)
        slprintf(lg->f, "[%lu] ", (unsigned long)getpid());

    if (context)
        slprintf(lg->f, "%s: ", context);

    slprintf(lg->f, "%s: ", xtl_level_to_string(level));

    vslprintf(lg->f, format, al);

    if (errnoval >= 0)
        slprintf(lg->f, ": %s", strerror(errnoval));

    slprintf(lg->f, "\n");
    slflush(lg->f);
}

static void stdiostream_progress(struct xentoollog_logger *logger_in,
                                 const char *context,
                                 const char *doing_what, int percent,
                                 unsigned long done, unsigned long total) {
    xentoollog_logger_stdiostream *lg = (void*)logger_in;
    int newpel, extra_erase;
    xentoollog_level this_level;

    if (!(lg->flags & XTL_STDIOSTREAM_HIDE_PROGRESS))
        return;

    if (percent < lg->progress_last_percent) {
        this_level = XTL_PROGRESS;
    } else if (percent == lg->progress_last_percent) {
        return;
    } else if (percent < lg->progress_last_percent + 5) {
        this_level = XTL_DETAIL;
    } else {
        this_level = XTL_PROGRESS;
    }

    if (this_level < lg->min_level)
        return;

    if (lg->progress_erase_len)
        putc('\r', lg->f);

    lg->progress_last_percent = percent;

    newpel = fprintf(lg->f, "%s%s" "%s: %lu/%lu  %3d%%%s",
                     context?context:"", context?": ":"",
                     doing_what, done, total, percent,
		     done == total ? "\n" : "");

    extra_erase = lg->progress_erase_len - newpel;
    if (extra_erase > 0)
        fprintf(lg->f, "%*s\r", extra_erase, "");

    lg->progress_erase_len = newpel;
}

static void stdiostream_destroy(struct xentoollog_logger *logger_in) {
    xentoollog_logger_stdiostream *lg = (void*)logger_in;
    progress_erase(lg);
    free(lg);
    closelog();
}

void xtl_stdiostream_set_minlevel(xentoollog_logger_stdiostream *lg,
                                  xentoollog_level min_level) {
    lg->min_level = min_level;
}

void xtl_stdiostream_adjust_flags(xentoollog_logger_stdiostream *lg,
                                  unsigned set_flags, unsigned clear_flags) {
    unsigned new_flags = (lg->flags & ~clear_flags) | set_flags;
    if (new_flags & XTL_STDIOSTREAM_HIDE_PROGRESS)
        progress_erase(lg);
    lg->flags = new_flags;
}

xentoollog_logger_stdiostream *xtl_createlogger_stdiostream
        (FILE *f, xentoollog_level min_level, unsigned flags) {
    xentoollog_logger_stdiostream newlogger;

    newlogger.f = f;
    newlogger.min_level = min_level;
    newlogger.flags = flags;

    if (newlogger.flags & XTL_STDIOSTREAM_SHOW_DATE) tzset();

    newlogger.progress_erase_len = 0;
#ifndef __UCLIBC__
    openlog(program_invocation_name, LOG_NOWAIT, LOG_DAEMON);
#endif
    return XTL_NEW_LOGGER(stdiostream, newlogger);
}
