/**
 * Simple command-line tool to make simple hypercalls to retrieve config/host
 * information when the full toolstack is not available.
 *
 * Usage: xeninfo <host-total-mem|host-total-cpus|xen-caps|xen-commandline>
 *
 * Based on code by James Bulpin <james@xensource.com>.
 * Modifications and additions by Andrew Peace <andrew@xensource.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include "xenctrl.h"

enum mode {
    MODE_PRINT_CAPS,
    MODE_PRINT_HOST_TOTAL_MEMORY,
    MODE_PRINT_HOST_TOTAL_CPUS,
    MODE_PRINT_COMMANDLINE,
    MODE_PRINT_DMESG,
};

int print_caps(xc_interface *xc)
{
    int ret;
    xen_capabilities_info_t xen_caps;

    ret = xc_version(xc, XENVER_capabilities, &xen_caps);
    if (ret == 0) {
        printf("%s\n", xen_caps);
    }

    return ret;
}

int print_host_total_memory(xc_interface *xc) {
    int ret;
    xc_physinfo_t info;

    memset(&info, 0, sizeof (info));
    ret = xc_physinfo(xc, &info);
    if (ret == 0) {
	printf("%"PRId64"\n", info.total_pages * XC_PAGE_SIZE / 1024);
    }

    return ret;
}

int print_host_total_cpus(xc_interface *xc) {
    int ret;
    xc_physinfo_t info;

    memset(&info, 0, sizeof (info));
    ret = xc_physinfo(xc, &info);
    if (ret == 0) {
      printf("%"PRIu32"\n", info.nr_cpus);
    }

    return ret;
}

int print_commandline(xc_interface *xc) {
    int ret;
    xen_commandline_t xen_commandline;

    ret = xc_version(xc, XENVER_commandline, &xen_commandline);
    if (ret == 0) {
        printf("%s\n", xen_commandline);
    }

    return ret;

}

int print_dmesg(xc_interface *xc) {
    int ret;
    unsigned int nr_chars = 4 * 1024 * 1024;
    char * buff = malloc(nr_chars);

    if ( ! buff ){
        fprintf(stderr, "Failed to allocate buffer");
        return 1;
    }

    ret = xc_readconsolering(xc, buff, &nr_chars, 0, 0, NULL);

    if ( ret ){
        fprintf(stderr, "readconsole hypercall failed %d\n", ret);
        goto out;
    }

    if ( nr_chars > 4 * 1024 * 1024 ){
        fprintf(stderr, "Huh - nr_chars is weird\n");
        ret = 1;
        goto out;
    }

    fwrite(buff, nr_chars, sizeof (char), stdout);

out:
    free(buff);
    return ret;
}

int main(int argc, char **argv)
{
    xc_interface *xc;
    int ret = 0;
    enum mode mode;

    if (argc < 2) {
        fprintf(stderr, "Incorrect number of arguments.\n"
                         "Specify either 'host-total-mem', 'host-total-cpus', 'xen-caps' or 'xen-commandline' or 'dmesg'.\n");
        return 1;
    }

    if (strcmp(argv[1], "host-total-mem") == 0) {
        mode = MODE_PRINT_HOST_TOTAL_MEMORY;
    } else if (strcmp(argv[1], "host-total-cpus") == 0) {
        mode = MODE_PRINT_HOST_TOTAL_CPUS;
    } else if (strcmp(argv[1], "xen-caps") == 0) {
        mode = MODE_PRINT_CAPS;
    } else if (strcmp(argv[1], "xen-commandline") == 0) {
        mode = MODE_PRINT_COMMANDLINE;
    } else if (strcmp(argv[1], "dmesg") == 0) {
        mode = MODE_PRINT_DMESG;
    } else {
        fprintf(stderr, "Unkown argument.\n"
                        "Specify either 'host-total-mem', 'host-total-cpus', 'xen-caps' or 'xen-commandline'.\n");
	return 1;
    }

    xc = xc_interface_open(0,0,0);
    if (NULL == xc) {
        fprintf(stderr, "Unable to open libxc.\n");
        ret = 1;
    } else {
        switch (mode) {
        case MODE_PRINT_CAPS:
            if (print_caps(xc) != 0) {
                perror("Unable to read xen_caps.\n");
                ret = 1;
            }
            break;

	case MODE_PRINT_HOST_TOTAL_MEMORY:
	    if (print_host_total_memory(xc) != 0) {
		perror("Unable to read physinfo.\n");
		ret = 1;
	    }
	    break;

	case MODE_PRINT_HOST_TOTAL_CPUS:
	    if (print_host_total_cpus(xc) != 0) {
		perror("Unable to read physinfo.\n");
		ret = 1;
	    }
	    break;

	case MODE_PRINT_DMESG:
	    if (print_dmesg(xc) != 0) {
		perror("Unable to read xen_dmesg.\n");
		ret = 1;
	    }
	    break;

	case MODE_PRINT_COMMANDLINE:
	    if (print_commandline(xc) != 0) {
		perror("Unable to read xen_commandline.\n");
		ret = 1;
	    }
	    break;
        }

        xc_interface_close(xc);
    }

    return ret;
}
