/*
  libreiserfs - a library for manipulating reiserfs partitions
  Copyright (C) 2001-2004 Yury Umanets <torque@ukrpost.net>.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.
                                                                                                 
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.
                                                                                                 
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/stat.h>

#include <reiserfs/exception.h>
#include <reiserfs/debug.h>

#if ENABLE_NLS
# include <locale.h>
# include <libintl.h>
# define _(String) dgettext (PACKAGE, String)
#else
# define _(String) (String)
#endif

static int check_choose(const char *valid_chooses, int choose) {
	unsigned int i;
    
	if (!valid_chooses)
		return 0;
	
	for (i = 0; i < strlen(valid_chooses); i++)
		if (valid_chooses[i] == choose) return 1;
	
	return 0;
}

int progs_user_choose(const char *valid_chooses, const char *error_msg, 
		      const char *format, ...) 
{
	char mess[4096], buf[255];
	int choose, prompt_count = 0;
	va_list args;
    
	if (!valid_chooses || !format || !error_msg)
		return 0;
	
	memset(mess, 0, 4096);
    
	va_start(args, format);
	vsprintf(mess, format, args);
	va_end(args);
	
	fprintf(stderr, mess);
	fflush(stderr);
	
	do {
		memset(buf, 0, 255);
		
		fgets(buf, 255, stdin);
		choose = buf[0];
			
		if (!check_choose(valid_chooses, choose)) {
			if (prompt_count < 2) {
				fprintf(stderr, error_msg);
				fflush(stderr);
			}	
		} else
			break;
	} while (prompt_count++ < 2);
    
	if (!check_choose(valid_chooses, choose))
		choose = 0;
	
	return choose;
}

int progs_check_dev(const char *dev_name) {
	struct stat st;
    
	if (!dev_name)
		return 0;
	
	if (stat(dev_name, &st) == -1)
		return 0;
	
	if (!S_ISBLK(st.st_mode)) {
		libreiserfs_exception_throw(EXCEPTION_WARNING,
					    EXCEPTION_IGNORE, 
					    _("Device %s isn't "
					      "a block device."),
					    dev_name);
	}
	
	return 1;
}

long progs_str2long(const char *str, int error) {
	long result = 0;
	char *err;

	if (!str) return 0;
    
	result = strtol(str, &err, 10);
    
	if (errno == ERANGE || *err)
		return error;
	
	return result;
}

#define KB 1024
#define MB (1024 * KB)
#define GB (1024 * MB)

long progs_parse_size(const char *str, 
		      size_t block_size) 
{
	long long size;
	char number[255], label = 0;
     
	if (!str || strlen(str) == 0)
		return -1;
	
	memset(number, 0, 255);
    
	strncpy(number, str, strlen(str));
	label = number[strlen(number) - 1];

	if (label == 'K' || label == 'M' || label == 'G')
		number[strlen(number) - 1] = '\0';
	else
		label = 0;	

	if ((size = progs_str2long(number, 0)) == 0)
		return 0;
    
	if (label == 0 || label == 'M')
		size = size * MB;
	if (label == 'K')
		size = size * KB;
	if (label == 'G')
		size = size * GB;

	return (size / block_size);
}
