/* $Id: banner.c,v 1.5 2002/03/02 22:01:40 sverrehu Exp $ */
/*------------------------------------------------------------------------
 |  FILE            banner.c
 |  MODULE OF       banner
 |
 |  DESCRIPTION     The main program.
 |
 |  WRITTEN BY      Sverre H. Huseby
 +----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <shhmsg.h>
#include <shhopt.h>

#include "banner.h"

/*-----------------------------------------------------------------------+
|  PRIVATE DATA                                                          |
+-----------------------------------------------------------------------*/

static int width = 75, height;
static int useLetterChar = 0;
static int doCenter = 0;
static char dotChar = '\0';
static char **buf;

/*-----------------------------------------------------------------------+
|  PRIVATE FUNCTIONS                                                     |
+-----------------------------------------------------------------------*/

/*------------------------------------------------------------------------
 |
 |  NAME          bufAlloc
 |
 |  FUNCTION      Allocate the buffer.
 */
static void bufAlloc(void)
{
    int q;

    height = getLetterHeight();
    if ((buf = (char **) malloc(height * sizeof(char *))) == NULL)
	msgFatal("out of memory\n");
    for (q = 0; q < height; q++) {
	if ((buf[q] = (char *) malloc((width + 1) * sizeof(char))) == NULL)
	    msgFatal("out of memory\n");
	*buf[q] = '\0';
    }
}

/*------------------------------------------------------------------------
 |  NAME          bufFlush
 |
 |  FUNCTION      Ouput whatever is currently in the buffer.
 */
static void bufFlush(void)
{
    int q, w, indent;

    if (!strlen(buf[0]))
	return;
    indent = (width - strlen(buf[0])) / 2;
    printf("\n");
    for (q = 0; q < height; q++) {
	if (doCenter)
	    for (w = 0; w < indent; w++)
		printf(" ");
	printf("%s\n", buf[q]);
	*buf[q] = '\0';
    }
}

/*------------------------------------------------------------------------
 |  NAME          bufAddLetter
 |
 |  FUNCTION      Add a letter to the buffer. There _must_ be room for it.
 |
 |  INPUT         ascii   the letter to add.
 */
static void bufAddLetter(int ascii)
{
    int q, w;
    char s[81], *p, c;

    for (q = 0; q < height; q++) {
	strncpy(s, getLetterLine(ascii, q), 80);
	if (!strlen(s)) /* no definition for this character */
	    break;
	c = useLetterChar ? ascii : dotChar;
	if (c) {
	    p = s;
	    while (*p) {
		if (*p != ' ')
		    *p = c;
		++p;
	    }
	}
	strcat(buf[q], s);
	for (w = 0; w < getInterCharWidth(); w++)
	    if (strlen(buf[q]) < width)
		strcat(buf[q], " ");
    }
}

/*------------------------------------------------------------------------
 |  NAME          bufAddWord
 |
 |  FUNCTION      Add a word to the buffer, possibly breaking lines.
 |
 |  INPUT         s       the word to add. should contain no white-space.
 */
static void bufAddWord(const unsigned char *s)
{
    int addlen;
    
    addlen = getLetterWidth(' ') + getInterCharWidth() + getTextWidth(s);
    if (strlen(buf[0]) + addlen > width)
	bufFlush();
    else if (strlen(buf[0]))
	bufAddLetter(' ');
    while (*s) {
	if (strlen(buf[0]) + getLetterWidth(*s) > width)
	    bufFlush();
	bufAddLetter(*s);
	++s;
    }
}

/*------------------------------------------------------------------------
 |  NAME          bufAddLetters
 |
 |  FUNCTION      Add several letters to the buffer, possibly breaking lines.
 |
 |  INPUT         s       the letters to add.
 */
static void bufAddLetters(const unsigned char *s)
{
    unsigned char *copy, *word, *p;
    int newline, done = 0;

    if ((copy = (char *) malloc((strlen(s) + 1) * sizeof(char))) == NULL)
	msgFatal("out of memory\n");
    strcpy(copy, s);
    p = copy;
    while (!done) {
	/* skip space before the next word */
	while (*p && (*p == ' ' || *p == '\t'))
	    ++p;
	if (!*p)
	    break;

	/* start of a word found. search for the end. */
	word = p;
	while (*p && *p != ' ' && *p != '\t' && *p != '\n')
	    ++p;
	newline = (*p == '\n');
	done = (*p == '\0');
	*p++ = '\0';

	if (strlen(word))
	    bufAddWord(word);

	if (newline)
	    bufFlush();
    }
    free(copy);
}

/*------------------------------------------------------------------------
 |  NAME          setWidth
 |
 |  FUNCTION      Argument handler: set width of lines.
 |
 |  INPUT         w       the width.
 */
static void setWidth(int w)
{
    if (w < 5 * getLetterWidth('m'))
	msgFatal("width %d too narrow\n", w);
    width = w;
}

/*------------------------------------------------------------------------
 |  NAME          useChar
 |
 |  FUNCTION      Argument handler: set character used to draw letters.
 |
 |  INPUT         s       string, of wich the first character is used.
 */
static void useChar(char *s)
{
    dotChar = *(unsigned char *)s;
}

/*------------------------------------------------------------------------
 |  NAME          usage
 |
 |  FUNCTION      Display program usage, and exit.
 |
 |  RETURNS       Never returns.
 */
static void usage(void)
{
    printf(
      "usage: %s [option] [text ...]\n"
      "\n"
      "  -c, --use-char=CHARACTER   "
	   "use CHARACTER to draw the letters\n"
      "  -C, --center               "
           "center the text on each line\n"
      "  -f, --font=FONT            "
	   "1=default font, 2=smaller font\n"
      "  -h, --help                 "
	   "display this help and exit\n"
      "  -l, --use-letters          "
	   "use individual characters to draw letters\n"
      "  -V, --version              "
	   "output version information and exit\n"
      "  -w, --width=COLS           "
	   "set max column width to COLS\n"
      "\n"
      "Reads from stdin if no text is given.\n",
      msgGetName());
    exit(0);
}

/*------------------------------------------------------------------------
 |  NAME          version
 |
 |  FUNCTION      Display program version, and exit.
 |
 |  RETURNS       Never returns.
 */
static void version(void)
{
    printf("banner %s, by Sverre H. Huseby\n", VERSION);
    exit(0);
}

/*-----------------------------------------------------------------------+
|  PUBLIC FUNCTIONS                                                      |
+-----------------------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int q;
    
    optStruct opt[] = {
      /* short long           type        var/func        special       */
        { 'h', "help",        OPT_FLAG,   usage,          OPT_CALLFUNC },
        { 'V', "version",     OPT_FLAG,   version,        OPT_CALLFUNC },
        { 'w', "width",       OPT_INT,    setWidth,       OPT_CALLFUNC },
        { 'l', "use-letters", OPT_FLAG,   &useLetterChar, 0 },
        { 'c', "use-char",    OPT_STRING, useChar,        OPT_CALLFUNC },
        { 'f', "font",        OPT_INT,    setFont,        OPT_CALLFUNC },
        { 'C', "center",      OPT_FLAG,   &doCenter,      0 },
	{ 0, 0, OPT_END, 0, 0 }  /* no more options */
    };
    
    msgSetName(argv[0]);
    setFont(1);
    optParseOptions(&argc, argv, opt, 0);
    bufAlloc();
    if (argc > 1) {
	/* letters given as arguments */
	for (q = 1; q < argc; q++) {
	    bufAddLetters(argv[q]);
	    if (q < argc - 1)
		bufAddLetters(" ");
	}
    } else {
	/* read letters from stdin */
	char line[81];

	while (fgets(line, 80, stdin))
	    bufAddLetters(line);
    }
    bufFlush();
    return 0;
}
