/*
  ==============================================================================

   This file is part of the JUCE library.
   Copyright (c) 2020 - Raw Material Software Limited

   JUCE is an open source library subject to commercial or open-source
   licensing.

   By using JUCE, you agree to the terms of both the JUCE 6 End-User License
   Agreement and JUCE Privacy Policy (both effective as of the 16th June 2020).

   End User License Agreement: www.juce.com/juce-6-licence
   Privacy Policy: www.juce.com/juce-privacy-policy

   Or: You may also use this code under the terms of the GPL v3 (see
   www.gnu.org/licenses).

   JUCE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL WARRANTIES, WHETHER
   EXPRESSED OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR PURPOSE, ARE
   DISCLAIMED.

  ==============================================================================
*/

namespace juce
{
// This byte-code is generated from native/java/com/rmsl/juce/ComponentPeerView.java with min sdk version 16
// See juce_core/native/java/README.txt on how to generate this byte-code.
static const uint8 javaComponentPeerView[] =
{31,139,8,8,139,79,248,96,0,3,74,97,118,97,68,101,120,66,121,116,101,67,111,100,101,46,100,101,120,0,165,155,11,124,220,69,181,
199,207,204,127,31,201,102,147,108,54,105,146,182,73,179,233,51,109,243,216,52,125,154,0,125,211,180,9,45,77,90,104,130,192,
38,249,39,217,118,243,223,237,238,230,5,168,229,161,45,40,80,164,64,161,69,138,60,20,244,34,23,193,203,5,46,162,34,2,22,229,
125,43,2,82,47,87,30,23,43,32,104,47,40,189,191,51,255,217,100,211,135,85,111,250,249,238,57,51,115,102,254,243,56,115,102,254,
155,166,203,28,242,4,235,230,209,29,101,15,100,92,253,135,105,151,221,117,240,202,238,155,26,239,243,103,77,251,209,43,231,175,
56,120,91,195,92,162,24,17,13,109,156,235,39,253,51,84,71,84,42,236,252,133,224,77,7,209,82,200,199,157,68,197,144,175,102,
16,93,0,185,55,147,8,69,20,201,38,90,55,157,232,61,200,165,165,68,203,193,233,96,53,88,15,54,2,11,108,5,253,96,8,92,4,182,129,
75,193,118,112,39,248,33,248,49,120,2,60,13,126,1,94,0,7,192,107,224,99,48,123,18,209,74,16,2,23,129,189,224,17,240,22,16,101,
68,19,64,13,88,1,206,5,23,128,59,192,67,224,69,224,13,16,85,129,243,193,46,176,31,252,5,76,41,39,90,11,190,8,126,8,220,147,49,
22,96,130,203,193,93,96,63,248,31,224,156,130,121,0,13,160,3,12,130,155,192,61,224,25,112,16,248,166,18,85,130,101,160,21,116,
131,8,184,0,124,1,92,12,118,128,175,129,93,224,38,176,15,220,1,238,2,247,128,239,131,7,193,163,224,199,224,105,240,12,120,30,
28,0,175,129,55,193,187,224,16,248,35,248,20,200,105,68,46,224,1,147,64,29,56,13,172,1,103,129,78,16,5,23,130,237,224,90,112,
11,248,46,120,24,60,7,126,5,126,7,222,7,89,88,219,66,48,30,84,130,185,160,1,44,7,205,160,21,132,128,5,190,8,174,4,215,131,155,
193,67,224,49,240,19,240,52,251,8,248,51,248,12,120,103,160,77,48,31,180,130,243,192,0,248,50,184,26,236,1,119,129,251,193,83,
224,117,240,14,248,4,100,84,16,229,129,9,96,6,88,4,26,193,185,160,15,124,1,236,1,183,130,111,129,187,193,195,224,25,240,27,112,
8,124,202,109,204,36,154,12,234,192,82,176,22,116,1,11,108,7,215,128,27,192,77,224,78,240,29,240,8,248,57,120,21,188,5,62,6,
71,128,123,22,81,62,40,5,11,193,50,208,4,76,144,4,23,130,43,192,78,176,27,236,3,183,129,187,192,125,224,17,240,52,120,30,28,0,
175,131,15,128,107,54,252,13,204,6,171,64,59,176,192,118,112,53,248,6,184,7,60,2,126,6,94,6,191,1,239,128,195,192,85,73,148,
3,138,64,57,168,0,115,193,50,176,26,132,65,4,244,131,27,193,191,129,71,193,79,192,11,224,32,248,4,120,170,208,15,48,7,52,130,
110,144,0,151,128,171,192,55,192,119,193,195,224,231,224,21,240,17,248,4,80,53,214,12,248,64,33,152,8,202,193,108,80,13,230,129,
83,193,114,176,1,156,15,226,224,98,112,37,216,9,174,5,55,128,61,224,22,112,27,248,22,112,161,121,132,36,202,2,133,160,136,236,
56,53,30,76,0,19,65,9,64,248,32,132,9,66,24,32,108,121,194,54,39,108,103,194,118,37,108,27,130,171,19,220,146,224,94,4,215,32,
44,41,97,250,9,83,72,24,62,233,97,80,13,8,130,90,48,7,32,92,18,194,40,205,3,243,193,2,29,55,23,129,207,129,122,208,0,78,1,167,
130,211,192,98,176,132,236,184,186,12,172,0,171,192,89,32,4,58,64,39,232,34,123,124,169,31,183,150,223,158,104,143,89,232,180,
71,235,156,207,243,32,117,190,87,235,247,34,63,55,173,45,191,158,179,7,117,126,142,206,207,213,101,41,125,156,214,125,122,94,
185,253,124,173,63,166,235,22,167,181,201,115,253,228,68,91,231,57,126,86,219,76,79,107,103,166,110,39,79,235,7,160,23,104,253,
141,137,246,51,121,206,223,214,237,176,254,190,110,167,74,183,51,78,175,195,97,221,31,94,11,94,228,85,122,77,248,167,81,235,
252,172,213,90,207,128,205,26,173,243,115,155,180,94,128,252,102,173,79,133,126,134,214,131,208,215,106,189,1,250,58,173,47,135,
190,94,235,77,105,249,173,105,250,57,208,91,181,222,149,150,31,75,211,135,210,244,109,105,109,238,72,203,223,9,189,69,235,187,
211,242,119,79,28,213,121,158,207,212,58,207,97,170,157,125,105,246,60,135,27,82,115,130,252,141,90,191,55,205,230,177,52,253,
201,18,219,31,235,244,124,158,173,245,103,145,191,73,235,7,210,244,55,161,183,105,253,125,232,237,90,63,12,253,28,173,59,176,
9,207,213,186,15,250,231,181,94,156,150,207,126,117,158,214,3,200,63,95,235,193,52,251,134,82,246,109,65,67,100,203,76,193,123,
125,26,197,201,150,223,87,82,208,15,180,124,80,203,127,215,242,33,45,31,214,246,79,16,199,135,0,185,132,45,115,5,199,138,58,
250,41,177,44,35,183,224,184,97,151,151,233,242,50,148,248,5,251,121,161,186,43,5,176,211,238,87,178,140,30,87,178,148,126,169,
100,29,253,183,146,89,244,182,242,243,233,180,18,210,137,200,243,7,226,61,233,162,171,149,28,71,55,67,102,34,114,25,74,214,
209,135,196,123,123,134,74,103,233,252,44,68,129,63,170,113,219,233,92,60,247,42,37,11,105,167,78,127,131,231,87,231,251,112,75,
187,86,167,119,43,105,208,141,100,239,197,155,180,220,163,164,160,219,180,188,157,120,239,73,218,165,100,5,125,147,120,159,77,
85,207,203,71,100,251,141,146,11,233,13,37,39,210,127,17,239,201,74,74,106,249,62,113,44,158,71,63,34,142,21,94,122,86,201,241,
116,136,215,90,247,123,60,118,54,203,18,68,199,251,148,156,73,127,82,242,84,21,204,74,177,195,13,181,46,89,116,169,146,197,180,
93,167,119,40,185,128,222,85,235,83,165,236,38,161,197,215,136,215,197,174,87,134,124,83,203,110,37,115,169,71,201,241,36,132,
45,165,90,191,105,202,62,128,21,233,85,178,154,194,74,46,166,205,74,230,211,22,45,35,74,158,70,125,74,142,167,65,45,47,82,114,
62,109,83,50,64,87,40,233,167,175,41,153,77,87,42,233,162,175,107,63,217,165,203,175,83,114,46,93,175,100,38,237,85,210,67,183,
104,187,125,74,102,208,173,74,218,235,16,192,12,221,169,229,183,180,252,182,146,147,232,46,45,239,214,249,223,209,126,248,93,
45,255,69,203,123,180,159,126,79,201,207,209,189,74,206,163,199,180,252,177,146,147,233,5,37,203,233,69,45,95,210,242,101,93,
254,159,58,125,64,203,95,41,153,67,175,40,57,157,126,173,228,108,122,85,201,83,232,117,37,109,255,9,104,255,225,244,65,37,231,
208,111,149,180,253,137,237,223,84,178,158,222,83,178,150,62,210,242,99,37,103,209,97,37,39,208,39,74,6,233,83,157,254,139,182,
251,171,150,159,233,242,35,122,94,124,130,207,164,34,186,129,88,46,161,31,170,125,41,233,63,148,204,163,28,193,231,142,237,167,
179,177,35,174,33,82,94,246,5,37,37,61,69,124,254,20,208,163,196,119,129,241,218,79,237,189,148,58,15,241,186,68,235,16,203,
26,244,129,52,73,231,243,121,198,177,155,203,123,33,219,117,121,153,174,63,59,173,254,16,202,45,93,30,32,251,204,228,243,110,
159,174,191,3,242,122,93,206,231,252,91,56,8,15,213,218,119,129,195,181,118,94,54,202,11,128,161,203,39,65,175,152,99,219,212,
170,124,67,233,119,87,219,119,130,118,225,161,152,143,123,211,46,179,176,150,46,180,194,207,122,160,218,190,83,181,46,201,162,
173,8,204,103,98,95,183,46,241,80,235,82,15,109,245,205,129,85,150,220,186,110,46,149,183,120,196,34,225,34,203,199,35,246,171,
62,59,244,51,158,212,247,181,201,152,181,152,111,88,141,136,219,231,190,61,95,109,223,1,252,121,117,227,198,31,219,114,112,30,
149,27,94,248,76,49,90,230,56,239,17,86,96,26,214,192,43,242,196,177,214,243,149,117,197,59,126,81,241,1,199,76,135,122,238,
187,213,246,28,183,110,133,125,204,182,71,169,20,151,137,235,220,119,14,186,22,160,151,30,196,9,3,173,112,127,63,170,230,56,
140,153,240,103,163,191,73,228,249,233,28,159,87,221,151,50,144,226,126,255,181,218,62,247,242,93,5,52,89,122,97,151,129,218,173,
137,108,106,173,203,65,95,151,194,202,139,25,105,24,205,11,172,68,236,31,181,56,13,22,30,99,61,46,161,235,167,9,250,215,214,
107,200,112,204,59,127,2,109,72,96,188,70,33,249,141,115,160,165,236,91,147,217,232,119,53,70,228,117,112,255,29,148,229,216,26,
172,163,101,78,182,95,100,184,105,193,249,78,242,231,115,29,126,214,56,236,75,203,87,192,35,59,234,25,21,191,112,168,145,194,
31,106,236,251,85,187,63,71,143,115,50,252,53,22,152,137,217,106,247,129,252,92,53,102,161,254,225,126,91,99,191,251,199,124,
124,11,203,25,201,95,60,146,31,84,249,82,223,80,87,213,216,254,215,234,203,85,254,192,222,192,207,93,87,99,223,81,91,3,185,232,
35,223,132,49,206,114,31,122,61,1,79,206,25,177,59,235,132,118,19,149,93,14,60,148,71,114,46,236,246,179,15,185,253,133,177,
64,33,226,225,100,71,22,122,227,66,175,218,119,21,34,85,70,177,224,108,218,149,217,190,43,31,169,82,164,166,170,84,1,218,61,21,
107,237,117,76,112,70,209,46,86,16,122,153,163,16,245,242,168,196,225,68,222,86,156,139,30,199,66,199,231,169,204,195,105,62,
63,219,246,250,161,245,209,48,157,179,103,28,180,110,156,29,86,160,71,125,246,226,156,157,140,54,99,62,117,110,58,10,104,202,
146,185,52,93,240,237,204,10,56,208,243,86,238,147,19,251,45,224,36,110,209,141,167,250,13,203,215,175,123,242,103,242,79,159,
210,61,31,227,233,200,40,162,230,12,151,219,95,84,146,145,165,52,43,24,163,86,151,215,88,104,20,144,95,250,167,78,89,190,128,
252,206,173,190,4,230,193,235,106,118,57,156,254,113,126,37,173,224,16,93,235,96,79,99,239,230,81,217,189,223,69,21,183,102,59,
42,62,2,31,130,247,193,43,128,131,185,225,80,119,242,209,159,109,167,209,63,148,62,250,199,46,231,123,93,33,118,35,127,127,196,
247,12,105,84,220,40,102,220,32,170,175,19,178,242,102,49,107,143,32,29,41,240,14,169,125,166,37,224,87,123,214,161,34,21,226,
68,202,199,130,77,216,227,28,159,156,42,62,253,186,198,142,137,177,13,38,149,47,243,40,127,78,213,121,59,85,22,92,172,226,66,
94,90,217,7,35,101,13,35,101,110,117,43,195,187,171,46,27,112,157,169,35,3,167,237,125,159,25,180,247,76,203,98,63,21,74,245,
14,34,23,161,36,71,229,196,130,141,228,16,21,159,140,142,39,47,104,183,197,239,168,109,24,83,142,238,3,255,20,5,237,88,237,247,
229,142,236,153,242,96,106,207,20,168,62,165,242,103,232,252,13,190,113,234,249,82,239,189,170,160,253,30,104,249,150,171,190,
46,66,235,220,30,238,111,159,25,250,57,117,65,59,238,231,168,122,246,91,98,125,90,94,42,22,47,61,201,216,78,215,99,75,217,175,
57,137,253,170,145,185,176,251,209,122,156,126,156,115,156,188,206,180,60,135,126,214,230,160,253,238,238,23,252,118,223,30,
148,100,75,131,54,213,34,238,189,191,41,232,166,77,65,151,206,205,132,31,99,37,125,155,130,14,148,103,96,63,142,71,127,112,71,
19,121,122,188,188,38,195,65,251,108,61,126,255,91,151,226,134,191,126,29,57,206,172,56,204,190,97,40,255,187,248,132,117,234,
86,28,57,162,198,189,184,158,28,33,174,227,197,115,248,108,189,10,117,248,78,233,31,239,167,150,111,162,30,26,170,21,210,177,
16,103,110,235,173,121,88,187,18,94,59,156,22,6,113,164,243,98,175,120,50,44,95,54,100,150,235,165,139,143,208,108,193,223,27,
84,124,216,114,43,199,157,92,46,71,188,240,177,116,88,190,60,150,78,247,190,129,12,126,131,21,59,43,94,63,153,93,179,109,247,
244,201,236,214,218,118,252,66,37,28,234,123,8,158,3,94,15,190,239,240,247,35,153,72,241,92,62,20,180,191,31,241,251,90,58,70,
71,184,8,181,252,24,83,165,155,191,59,224,18,203,151,133,57,241,136,216,153,171,105,122,162,226,143,209,96,17,13,100,120,168,
226,144,229,91,136,149,169,224,203,189,180,231,58,245,125,14,127,31,144,173,241,160,132,159,247,82,208,254,94,192,47,70,158,39,
241,60,212,204,21,139,220,153,120,78,38,250,230,145,254,41,115,230,184,41,186,184,152,6,118,123,68,197,135,150,207,195,81,86,
46,148,191,63,146,210,173,192,0,158,225,81,253,139,173,67,191,12,127,94,197,107,246,154,243,179,62,8,218,223,61,140,93,115,123,
100,156,199,99,114,225,89,177,245,107,224,45,126,95,197,199,164,35,9,209,145,52,95,206,212,49,43,187,214,246,161,232,217,184,
241,12,164,183,186,16,207,180,247,78,20,123,135,235,228,106,31,154,132,226,18,52,212,242,28,172,13,158,93,97,168,217,21,185,194,
82,119,19,79,38,123,76,14,101,185,95,186,134,61,230,79,106,118,42,62,104,121,222,15,201,171,60,73,173,46,159,156,30,39,175,50,
164,203,253,28,123,3,110,56,121,21,175,158,220,114,173,109,249,179,147,91,158,1,75,203,199,103,180,39,203,159,183,96,226,44,
242,151,79,41,153,7,111,91,66,207,144,127,194,188,135,202,137,91,225,54,238,225,54,16,27,33,133,223,185,220,233,117,94,178,225,
137,113,86,32,223,206,41,92,238,242,186,46,233,126,162,48,213,238,1,15,86,242,13,244,227,210,2,193,151,251,7,174,57,232,17,
25,139,60,197,244,247,142,96,38,117,31,73,239,207,63,211,19,110,169,226,215,255,120,15,154,117,15,166,253,191,123,208,172,122,
192,123,211,254,134,144,125,139,227,95,141,150,188,79,249,187,137,132,242,61,169,238,74,215,213,218,223,103,162,143,184,111,
184,225,253,19,140,255,37,127,193,148,229,184,111,56,59,92,184,111,168,187,67,140,230,56,248,254,234,70,191,179,113,150,128,
191,248,253,83,38,227,182,97,240,109,35,3,55,138,102,135,52,248,150,177,85,86,28,202,150,21,239,129,119,193,219,236,239,121,232,
27,127,47,232,229,104,43,43,11,103,21,27,21,21,51,42,171,103,165,197,249,31,213,142,238,13,67,231,62,85,107,223,189,23,74,55,
241,140,108,129,69,14,249,23,87,124,202,227,211,239,12,181,246,247,65,124,119,227,91,246,4,249,136,186,187,149,227,105,101,120,
207,179,223,22,6,177,67,61,226,20,145,143,217,182,130,83,40,136,17,175,80,246,86,112,50,249,28,173,181,249,184,81,87,170,246,
83,37,126,105,5,203,201,39,237,178,42,158,223,15,237,223,113,241,79,234,123,87,254,94,243,112,54,141,126,129,170,127,188,71,165,
75,142,74,115,253,2,178,207,252,60,244,78,232,60,38,160,101,134,46,247,235,243,125,130,206,47,211,82,106,2,122,190,102,211,2,
251,30,160,243,171,104,190,150,66,197,80,161,255,57,105,244,46,33,181,111,164,238,16,182,238,30,249,14,89,98,118,71,243,89,102,
171,180,67,247,193,165,203,92,176,147,58,207,173,101,166,150,169,239,160,115,84,79,72,223,101,120,92,117,186,255,1,149,95,167,
253,182,110,164,167,118,253,185,90,206,211,245,132,190,127,177,204,214,207,103,61,119,164,60,87,245,79,106,139,212,248,138,70,
250,97,183,237,211,207,11,140,88,218,227,10,104,107,65,246,187,184,168,32,188,187,185,26,194,86,56,121,42,201,83,235,169,112,
89,180,47,22,181,76,43,185,206,52,227,27,195,230,96,245,230,208,64,136,196,74,146,43,27,73,52,146,108,156,9,160,174,38,185,186,
137,74,87,247,119,154,75,58,59,205,68,34,220,17,142,132,147,195,103,68,187,204,117,241,232,64,184,203,140,83,209,26,115,184,35,
26,138,119,45,15,39,250,194,137,68,83,56,145,52,45,20,136,38,146,77,104,173,9,205,52,53,145,209,132,4,62,86,243,71,19,21,54,
133,172,174,120,52,220,85,19,138,197,106,150,116,38,195,3,104,185,158,230,142,205,143,197,34,225,206,80,50,28,181,166,166,108,
154,194,221,102,231,112,103,196,92,22,138,68,58,66,157,91,18,245,52,254,68,181,210,139,58,163,22,122,150,172,89,198,114,40,153,
94,212,19,15,197,122,195,157,137,154,101,33,107,32,132,6,39,29,167,40,26,137,198,87,134,35,73,51,126,226,242,230,80,50,30,30,
170,167,153,127,179,124,76,83,197,199,154,174,11,133,45,244,175,232,216,146,245,102,39,10,242,71,10,162,137,154,165,253,86,87,
196,172,167,130,244,204,198,165,97,171,139,91,31,109,99,0,75,93,131,197,90,49,96,114,227,19,198,22,52,71,121,186,116,217,204,
177,101,236,36,83,215,90,43,163,157,253,137,101,189,33,171,199,76,45,114,122,87,70,76,211,135,52,146,121,122,60,218,31,171,167,
249,199,150,180,198,77,115,109,71,194,140,15,152,113,60,229,244,72,180,35,20,105,10,13,71,251,147,163,143,41,251,219,245,234,
169,118,172,65,40,221,95,107,198,120,111,115,200,10,245,112,149,57,127,119,21,118,248,70,171,59,122,76,255,79,82,39,181,73,234,
169,122,108,189,176,21,235,79,246,153,201,222,104,87,205,210,80,2,141,35,13,191,180,176,188,202,107,167,157,216,126,69,87,56,
25,141,219,221,153,117,98,179,99,154,172,58,137,109,179,210,71,102,231,180,166,206,104,95,77,188,47,17,169,217,140,0,80,115,76,
216,152,250,55,227,66,61,173,60,105,3,39,136,28,83,199,174,236,162,127,182,157,122,42,63,89,213,122,154,220,212,21,138,12,132,
183,212,132,44,43,154,84,49,163,102,133,213,25,137,38,194,86,207,178,72,40,161,130,193,177,54,141,152,216,184,46,47,63,78,121,
179,217,215,161,13,76,152,148,30,199,164,37,220,99,133,146,253,113,147,55,12,199,224,154,8,246,86,13,118,88,188,197,220,218,
111,90,157,40,201,75,47,177,31,55,57,45,171,49,18,49,123,66,17,123,25,86,12,117,154,49,123,177,167,30,199,38,222,211,223,135,177,
167,89,229,167,91,33,40,246,216,147,54,154,121,70,180,165,191,179,215,246,140,180,122,254,52,147,181,29,155,85,76,42,77,203,
107,49,59,251,227,112,136,19,84,105,65,12,180,122,216,35,71,243,226,102,119,4,237,160,27,3,81,59,116,183,134,226,61,102,122,111,
39,28,199,220,238,90,61,141,179,203,250,147,225,72,205,146,120,60,52,204,78,80,79,185,105,217,156,67,190,163,50,240,142,219,
186,105,221,10,242,166,251,28,137,141,36,55,54,146,115,99,35,126,160,174,38,215,198,213,141,43,87,174,38,7,100,35,127,242,137,
182,113,117,27,10,89,225,83,109,163,202,106,106,67,105,83,27,206,188,141,109,168,213,166,90,16,109,100,180,113,61,124,52,177,
218,68,206,182,213,172,59,32,112,78,182,113,46,206,71,87,91,147,202,118,178,68,126,59,14,229,246,70,242,183,31,235,5,249,237,
199,89,4,143,29,144,166,6,131,193,17,189,54,77,159,147,166,215,165,233,115,211,244,121,105,250,252,52,125,65,154,190,16,122,
150,173,175,140,132,122,18,148,61,38,18,82,65,232,56,17,151,92,33,21,138,184,38,203,166,80,135,25,161,140,144,62,215,105,124,168,
171,235,248,241,159,50,67,218,121,19,36,58,40,143,79,253,165,253,201,100,212,90,23,199,99,204,46,114,117,68,145,236,131,84,167,
33,185,58,213,65,78,238,78,117,94,117,145,19,23,134,80,156,178,58,57,70,69,113,188,46,73,114,98,228,72,166,28,149,104,141,135,
172,68,119,52,222,71,217,124,93,192,57,156,80,214,104,200,190,53,160,161,104,63,210,19,59,227,102,40,121,108,252,227,184,76,
142,174,112,119,55,9,147,156,38,31,168,20,232,198,105,124,92,211,196,210,225,86,110,53,147,45,212,1,75,206,110,37,188,221,163,
199,109,23,101,169,20,135,172,198,46,154,136,125,49,166,181,149,105,133,37,71,23,142,185,173,101,170,82,181,10,227,88,29,189,
41,233,91,17,101,35,155,67,16,150,197,140,227,5,137,147,236,113,228,97,77,27,121,121,103,242,208,90,195,125,166,106,116,149,25,
238,233,77,82,62,84,117,154,164,119,137,51,155,244,174,94,107,181,96,230,76,75,85,178,119,47,229,66,85,179,140,128,169,38,215,
59,154,129,234,110,164,214,135,6,207,78,41,155,40,139,149,104,52,201,237,147,15,137,150,97,56,74,95,11,118,111,184,211,164,28,
228,108,176,194,188,212,60,18,245,252,163,111,12,106,156,27,195,35,30,203,117,206,194,26,68,7,91,163,91,208,189,146,145,180,50,
138,152,56,92,98,145,208,240,202,120,8,35,118,160,244,108,245,185,137,68,47,229,98,149,224,117,152,205,117,161,126,118,71,223,
72,198,122,51,1,199,29,201,89,58,226,185,148,109,231,224,240,90,30,29,196,166,24,73,110,136,81,193,72,66,29,108,171,194,93,93,
232,147,126,76,115,20,207,80,117,198,100,196,67,61,169,54,85,6,154,209,109,170,235,36,229,235,132,25,103,255,214,78,144,209,
27,74,216,110,87,212,11,7,105,137,118,235,229,139,71,251,236,225,195,4,181,149,147,58,122,163,8,161,34,76,30,172,250,90,21,149,
19,100,132,251,250,40,151,223,47,194,161,200,178,80,44,209,140,105,167,108,157,209,98,70,86,88,93,35,229,72,182,36,67,113,248,
187,186,126,180,14,199,76,242,42,245,60,251,42,66,25,120,216,198,80,164,31,91,56,140,163,96,139,137,135,37,26,173,68,50,132,
227,16,165,137,181,177,16,206,70,26,31,78,180,70,113,54,173,24,138,97,219,42,215,90,97,133,176,78,93,104,59,161,215,140,220,91,
204,225,101,220,159,162,45,39,120,75,201,78,21,180,244,242,140,58,35,42,40,101,99,165,205,56,119,239,12,220,23,200,17,49,187,
147,228,138,152,86,79,178,151,92,186,171,194,34,135,197,222,224,182,204,193,51,88,201,176,82,17,192,107,165,239,58,87,180,131,
67,9,57,162,145,174,94,245,57,72,121,81,43,245,70,179,76,133,18,236,147,209,172,229,102,34,25,143,14,179,227,140,102,106,231,
74,171,153,242,174,137,163,89,45,161,1,51,53,95,152,235,164,153,110,175,38,127,108,19,45,201,104,44,134,172,34,132,0,213,143,
163,46,142,232,188,5,223,26,164,236,104,250,27,0,229,68,199,132,106,242,70,173,85,81,108,44,21,22,40,51,106,165,28,59,91,169,
205,253,145,100,56,198,75,162,146,112,206,12,62,1,84,85,88,180,132,47,48,83,177,14,45,217,75,171,90,114,69,237,5,119,219,242,
60,212,235,71,240,78,34,54,57,99,202,177,61,177,80,28,150,42,24,100,199,198,184,183,51,166,130,119,105,44,26,235,143,156,48,92,
139,56,185,227,246,59,37,77,142,155,61,236,27,241,19,191,110,82,73,220,236,195,80,237,225,175,181,142,58,171,156,113,21,9,141,
132,153,164,156,4,199,204,145,151,61,242,34,173,38,145,61,149,138,210,83,141,246,160,213,238,225,106,105,119,114,85,173,41,229,
145,84,140,212,113,223,198,104,92,34,21,10,55,132,211,98,219,196,227,102,243,101,56,132,67,48,97,7,71,229,192,217,137,49,65,
209,147,74,70,236,62,157,21,142,68,206,136,38,149,59,120,19,216,48,169,0,133,138,72,141,68,15,24,179,163,217,253,194,85,15,197,
240,178,209,100,113,194,238,77,227,232,179,244,72,29,234,120,113,36,123,195,9,114,241,231,212,160,150,181,200,229,195,198,64,
75,80,121,34,50,250,147,221,11,85,132,23,3,228,28,80,81,195,173,196,218,110,114,240,91,14,229,242,103,186,115,101,114,70,107,
116,67,194,36,223,192,49,103,194,64,56,158,236,15,69,244,185,229,25,24,157,10,49,72,98,136,228,80,16,212,130,57,160,14,204,37,
49,76,223,115,72,250,186,116,231,180,85,209,15,28,98,23,180,205,6,253,82,20,255,182,33,224,36,122,220,33,191,44,57,147,158,
118,136,43,132,59,231,67,131,190,38,102,85,237,144,23,150,58,233,101,23,209,203,14,99,151,140,15,192,226,247,82,16,218,105,160,
143,36,185,43,219,13,121,72,100,95,110,136,119,69,126,245,208,159,13,250,178,144,155,27,68,94,94,184,65,38,80,183,65,100,161,
246,171,210,126,186,156,191,73,174,25,172,162,95,73,177,155,159,118,180,252,45,154,204,185,159,222,178,69,131,113,137,120,75,
60,32,220,149,242,57,170,151,59,228,160,252,153,24,28,146,7,47,124,99,135,144,78,207,146,170,134,234,134,134,83,219,13,250,146,
231,34,67,108,19,243,26,238,44,55,140,159,139,160,40,26,23,156,100,200,39,133,20,121,69,78,41,207,68,79,156,194,105,184,60,
114,246,237,78,143,139,92,194,37,93,198,172,89,114,160,210,41,103,201,68,37,45,176,123,176,64,238,150,55,42,197,193,202,77,114,
207,216,220,17,197,153,42,222,203,201,189,220,227,43,74,104,0,98,13,29,50,228,13,242,102,53,32,7,50,104,7,207,24,189,105,240,
231,46,135,76,94,0,121,149,67,36,33,14,27,44,80,243,22,149,254,106,9,237,208,6,55,217,6,151,216,98,143,33,222,22,238,210,53,
107,170,218,214,180,85,211,151,196,133,170,214,67,134,188,82,222,135,5,251,234,196,42,250,157,16,219,121,178,138,63,147,91,74,
215,24,25,113,217,100,184,46,151,134,204,234,21,249,185,114,176,20,159,1,214,134,74,233,14,41,31,103,235,210,237,210,95,42,99,
165,50,183,94,246,223,126,142,28,94,77,79,74,227,113,113,169,42,148,217,95,145,3,165,215,183,111,222,14,127,144,178,141,30,213,
213,114,182,7,74,174,167,239,27,206,139,158,18,87,200,15,197,48,186,120,183,225,184,94,222,41,94,16,207,163,252,148,237,109,
116,131,176,77,229,147,84,42,31,250,18,186,148,201,93,114,115,151,126,66,155,228,226,74,116,38,104,227,107,50,178,246,11,121,74,
131,225,125,79,204,63,69,8,195,243,3,33,171,68,73,246,105,78,143,211,91,235,204,218,236,242,84,139,252,66,121,97,125,131,203,
123,138,40,29,199,249,99,51,229,74,81,154,67,79,25,226,58,204,125,192,16,151,201,160,240,23,74,111,165,236,43,45,51,232,43,162,
106,146,147,148,50,123,138,147,14,6,103,209,79,13,241,11,116,145,222,49,196,160,59,39,92,66,151,75,241,85,84,126,220,160,7,69,
105,245,230,53,67,231,143,223,78,178,92,92,41,139,3,114,178,180,28,254,253,162,40,91,78,69,70,137,187,72,20,45,46,202,44,90,
94,36,139,102,21,57,108,171,50,101,37,97,117,250,136,125,142,156,194,246,162,120,146,173,200,226,210,226,50,146,194,225,217,25,
16,254,169,23,111,115,220,94,50,77,60,94,34,196,142,82,33,110,7,59,38,161,88,120,165,216,25,152,185,109,155,227,193,73,179,
196,171,147,200,112,81,14,215,16,254,217,168,179,47,32,118,4,158,229,143,183,249,99,91,185,144,187,193,99,229,228,42,154,152,
231,135,239,251,237,127,181,48,126,187,28,38,247,79,195,199,126,254,120,147,63,182,77,199,199,94,254,120,112,186,227,18,73,2,
176,116,107,253,239,161,0,120,192,66,241,226,116,33,46,155,33,196,190,25,134,184,127,198,56,177,31,250,155,96,71,133,16,123,
193,189,224,13,176,115,38,134,8,30,4,251,193,190,89,66,188,8,118,204,22,226,189,74,33,94,173,18,226,201,106,225,184,172,70,56,
118,215,192,166,198,41,118,205,133,221,60,41,238,7,7,230,217,223,45,167,255,158,128,101,234,111,106,248,59,232,212,223,213,168,
223,181,147,253,183,53,169,223,115,242,223,215,240,247,210,169,191,177,113,209,232,223,217,24,62,187,140,127,15,33,2,246,239,
119,166,230,192,38,96,219,240,255,55,19,62,251,247,58,252,127,204,100,192,126,46,255,93,142,161,237,249,255,121,57,2,246,239,
23,248,255,131,145,174,171,254,159,154,207,238,43,255,13,208,255,1,98,184,185,187,60,52,0,0,0,0};

//==============================================================================
#if JUCE_PUSH_NOTIFICATIONS && JUCE_MODULE_AVAILABLE_juce_gui_extra
 extern bool juce_handleNotificationIntent (void*);
 extern void juce_firebaseDeviceNotificationsTokenRefreshed (void*);
 extern void juce_firebaseRemoteNotificationReceived (void*);
 extern void juce_firebaseRemoteMessagesDeleted();
 extern void juce_firebaseRemoteMessageSent(void*);
 extern void juce_firebaseRemoteMessageSendError (void*, void*);
#endif

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (create, "<init>", "(II)V")

DECLARE_JNI_CLASS (AndroidLayoutParams, "android/view/ViewGroup$LayoutParams")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (addView,          "addView",             "(Landroid/view/View;Landroid/view/ViewGroup$LayoutParams;)V") \
 METHOD (removeView,       "removeView",          "(Landroid/view/View;)V") \
 METHOD (updateViewLayout, "updateViewLayout",    "(Landroid/view/View;Landroid/view/ViewGroup$LayoutParams;)V")

DECLARE_JNI_CLASS (AndroidViewManager, "android/view/ViewManager")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (create,           "<init>",             "(IIIIIII)V") \
 FIELD  (gravity,          "gravity",            "I") \
 FIELD  (windowAnimations, "windowAnimations",   "I")

DECLARE_JNI_CLASS (AndroidWindowManagerLayoutParams, "android/view/WindowManager$LayoutParams")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (getDisplayCutout,     "getDisplayCutout", "()Landroid/view/DisplayCutout;")

 DECLARE_JNI_CLASS_WITH_MIN_SDK (AndroidWindowInsets, "android/view/WindowInsets", 28)
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (getSafeInsetBottom, "getSafeInsetBottom", "()I") \
 METHOD (getSafeInsetLeft,   "getSafeInsetLeft",   "()I") \
 METHOD (getSafeInsetRight,  "getSafeInsetRight",  "()I") \
 METHOD (getSafeInsetTop,    "getSafeInsetTop",    "()I")

 DECLARE_JNI_CLASS_WITH_MIN_SDK (AndroidDisplayCutout, "android/view/DisplayCutout", 28)
#undef JNI_CLASS_MEMBERS

//==============================================================================
namespace
{
    enum
    {
        SYSTEM_UI_FLAG_VISIBLE = 0,
        SYSTEM_UI_FLAG_LOW_PROFILE = 1,
        SYSTEM_UI_FLAG_HIDE_NAVIGATION = 2,
        SYSTEM_UI_FLAG_FULLSCREEN = 4,
        SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION = 512,
        SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN = 1024,
        SYSTEM_UI_FLAG_IMMERSIVE = 2048,
        SYSTEM_UI_FLAG_IMMERSIVE_STICKY = 4096
    };

    constexpr int fullScreenFlags = SYSTEM_UI_FLAG_HIDE_NAVIGATION | SYSTEM_UI_FLAG_FULLSCREEN | SYSTEM_UI_FLAG_IMMERSIVE_STICKY;
    constexpr int FLAG_NOT_FOCUSABLE = 0x8;
}

//==============================================================================
static bool supportsDisplayCutout()
{
    return getAndroidSDKVersion() >= 28;
}

static BorderSize<int> androidDisplayCutoutToBorderSize (LocalRef<jobject> displayCutout, double displayScale)
{
    if (displayCutout.get() == nullptr)
        return {};

    auto* env = getEnv();

    auto getInset = [&] (jmethodID methodID)
    {
        return roundToInt (env->CallIntMethod (displayCutout.get(), methodID) / displayScale);
    };

    return { getInset (AndroidDisplayCutout.getSafeInsetTop),
             getInset (AndroidDisplayCutout.getSafeInsetLeft),
             getInset (AndroidDisplayCutout.getSafeInsetBottom),
             getInset (AndroidDisplayCutout.getSafeInsetRight) };
}

//==============================================================================
class AndroidComponentPeer  : public ComponentPeer,
                              private Timer
{
public:
    AndroidComponentPeer (Component& comp, int windowStyleFlags, void* nativeViewHandle)
        : ComponentPeer (comp, windowStyleFlags)
    {
        auto* env = getEnv();

        // NB: must not put this in the initialiser list, as it invokes a callback,
        // which will fail if the peer is only half-constructed.
        view = GlobalRef (LocalRef<jobject> (env->NewObject (ComponentPeerView, ComponentPeerView.create,
                                                             getAppContext().get(), (jboolean) component.isOpaque(),
                                                             (jlong) this)));

        if (nativeViewHandle != nullptr)
        {
            viewGroupIsWindow = false;

            // we don't know if the user is holding on to a local ref to this, so
            // explicitly create a new one
            auto nativeView = LocalRef<jobject> (env->NewLocalRef (static_cast<jobject> (nativeViewHandle)));

            if (env->IsInstanceOf (nativeView.get(), AndroidActivity))
            {
                viewGroup = GlobalRef (nativeView);
                env->CallVoidMethod (viewGroup.get(), AndroidActivity.setContentView, view.get());
            }
            else if (env->IsInstanceOf (nativeView.get(), AndroidViewGroup))
            {
                viewGroup = GlobalRef (nativeView);
                LocalRef<jobject> layoutParams (env->NewObject (AndroidLayoutParams, AndroidLayoutParams.create, -2, -2));

                env->CallVoidMethod (view.get(), AndroidView.setLayoutParams, layoutParams.get());
                env->CallVoidMethod ((jobject) viewGroup.get(), AndroidViewGroup.addView, view.get());
            }
            else
            {
                // the native handle you passed as a second argument to Component::addToDesktop must
                // either be an Activity or a ViewGroup
                jassertfalse;
            }
        }
        else
        {
            viewGroupIsWindow = true;

            LocalRef<jobject> viewLayoutParams (env->NewObject (AndroidLayoutParams, AndroidLayoutParams.create, -2, -2));
            env->CallVoidMethod (view.get(), AndroidView.setLayoutParams, viewLayoutParams.get());

            auto physicalBounds = (comp.getBoundsInParent().toFloat() * scale).toNearestInt();

            view.callVoidMethod (AndroidView.layout,
                                 physicalBounds.getX(), physicalBounds.getY(), physicalBounds.getRight(), physicalBounds.getBottom());

            LocalRef<jobject> windowLayoutParams (env->NewObject (AndroidWindowManagerLayoutParams, AndroidWindowManagerLayoutParams.create,
                                                                  physicalBounds.getWidth(), physicalBounds.getHeight(),
                                                                  physicalBounds.getX(), physicalBounds.getY(),
                                                                  TYPE_APPLICATION, FLAG_NOT_TOUCH_MODAL | FLAG_LAYOUT_IN_SCREEN | FLAG_LAYOUT_NO_LIMITS | FLAG_NOT_FOCUSABLE,
                                                                  component.isOpaque() ? PIXEL_FORMAT_OPAQUE : PIXEL_FORMAT_TRANSPARENT));

            env->SetIntField (windowLayoutParams.get(), AndroidWindowManagerLayoutParams.gravity, GRAVITY_LEFT | GRAVITY_TOP);
            env->SetIntField (windowLayoutParams.get(), AndroidWindowManagerLayoutParams.windowAnimations, 0x01030000 /* android.R.style.Animation */);

            if (supportsDisplayCutout())
            {
                jfieldID layoutInDisplayCutoutModeFieldId = env->GetFieldID (AndroidWindowManagerLayoutParams,
                                                                             "layoutInDisplayCutoutMode",
                                                                             "I");

                if (layoutInDisplayCutoutModeFieldId != nullptr)
                    env->SetIntField (windowLayoutParams.get(),
                                      layoutInDisplayCutoutModeFieldId,
                                      LAYOUT_IN_DISPLAY_CUTOUT_MODE_ALWAYS);
            }

            if (Desktop::getInstance().getKioskModeComponent() != nullptr)
                setNavBarsHidden (true);

            LocalRef<jobject> activity (getCurrentActivity());

            if (activity == nullptr)
                activity = getMainActivity();

            viewGroup = GlobalRef (LocalRef<jobject> (env->CallObjectMethod (activity.get(), AndroidContext.getSystemService, javaString ("window").get())));
            env->CallVoidMethod (viewGroup.get(), AndroidViewManager.addView, view.get(), windowLayoutParams.get());
        }

        if (supportsDisplayCutout())
        {
            jmethodID setOnApplyWindowInsetsListenerMethodId = env->GetMethodID (AndroidView,
                                                                                 "setOnApplyWindowInsetsListener",
                                                                                 "(Landroid/view/View$OnApplyWindowInsetsListener;)V");

            if (setOnApplyWindowInsetsListenerMethodId != nullptr)
                env->CallVoidMethod (view.get(), setOnApplyWindowInsetsListenerMethodId,
                                     CreateJavaInterface (new ViewWindowInsetsListener,
                                                          "android/view/View$OnApplyWindowInsetsListener").get());
        }

        if (isFocused())
            handleFocusGain();
    }

    ~AndroidComponentPeer() override
    {
        stopTimer();

        auto* env = getEnv();

        env->CallVoidMethod (view, ComponentPeerView.clear);
        frontWindow = nullptr;

        GlobalRef localView (view);
        GlobalRef localViewGroup (viewGroup);

        callOnMessageThread ([env, localView, localViewGroup]
        {
            if (env->IsInstanceOf (localViewGroup.get(), AndroidActivity))
                env->CallVoidMethod (localViewGroup.get(), AndroidActivity.setContentView, nullptr);
            else
                env->CallVoidMethod (localViewGroup.get(), AndroidViewManager.removeView, localView.get());
        });
    }

    void* getNativeHandle() const override
    {
        return (void*) view.get();
    }

    void setVisible (bool shouldBeVisible) override
    {
        GlobalRef localView (view);

        callOnMessageThread ([localView, shouldBeVisible]
        {
            localView.callVoidMethod (ComponentPeerView.setVisible, shouldBeVisible);
        });
    }

    void setTitle (const String& title) override
    {
        view.callVoidMethod (ComponentPeerView.setViewName, javaString (title).get());
    }

    void setBounds (const Rectangle<int>& userRect, bool isNowFullScreen) override
    {
        auto bounds = (userRect.toFloat() * scale).toNearestInt();

        if (MessageManager::getInstance()->isThisTheMessageThread())
        {
            fullScreen = isNowFullScreen;

            view.callVoidMethod (AndroidView.layout,
                                 bounds.getX(), bounds.getY(), bounds.getRight(), bounds.getBottom());

            if (viewGroup != nullptr && viewGroupIsWindow)
            {
                auto* env = getEnv();

                LocalRef<jobject> windowLayoutParams (env->NewObject (AndroidWindowManagerLayoutParams, AndroidWindowManagerLayoutParams.create,
                                                                      bounds.getWidth(), bounds.getHeight(), bounds.getX(), bounds.getY(),
                                                                      TYPE_APPLICATION, FLAG_NOT_TOUCH_MODAL | FLAG_LAYOUT_IN_SCREEN | FLAG_LAYOUT_NO_LIMITS,
                                                                      component.isOpaque() ? PIXEL_FORMAT_OPAQUE : PIXEL_FORMAT_TRANSPARENT));

                env->SetIntField (windowLayoutParams.get(), AndroidWindowManagerLayoutParams.gravity, GRAVITY_LEFT | GRAVITY_TOP);
                env->CallVoidMethod (viewGroup.get(), AndroidViewManager.updateViewLayout, view.get(), windowLayoutParams.get());
            }
        }
        else
        {
            GlobalRef localView (view);

            MessageManager::callAsync ([localView, bounds]
            {
                localView.callVoidMethod (AndroidView.layout,
                                          bounds.getX(), bounds.getY(), bounds.getRight(), bounds.getBottom());
            });
        }
    }

    Rectangle<int> getBounds() const override
    {
        Rectangle<int> bounds (view.callIntMethod (AndroidView.getLeft),
                               view.callIntMethod (AndroidView.getTop),
                               view.callIntMethod (AndroidView.getWidth),
                               view.callIntMethod (AndroidView.getHeight));

        return (bounds.toFloat() / scale).toNearestInt();
    }

    void handleScreenSizeChange() override
    {
        ComponentPeer::handleScreenSizeChange();

        if (isFullScreen())
            setFullScreen (true);
    }

    Point<int> getScreenPosition() const
    {
        auto* env = getEnv();

        LocalRef<jintArray> position (env->NewIntArray (2));
        env->CallVoidMethod (view.get(), AndroidView.getLocationOnScreen, position.get());

        jint* const screenPosition = env->GetIntArrayElements (position.get(), nullptr);
        Point<int> pos (screenPosition[0], screenPosition[1]);
        env->ReleaseIntArrayElements (position.get(), screenPosition, 0);

        return pos;
    }

    Point<float> localToGlobal (Point<float> relativePosition) override
    {
        return relativePosition + (getScreenPosition().toFloat() / scale);
    }

    using ComponentPeer::localToGlobal;

    Point<float> globalToLocal (Point<float> screenPosition) override
    {
        return screenPosition - (getScreenPosition().toFloat() / scale);
    }

    using ComponentPeer::globalToLocal;

    void setMinimised (bool /*shouldBeMinimised*/) override
    {
        // n/a
    }

    bool isMinimised() const override
    {
        return false;
    }

    void setFullScreen (bool shouldBeFullScreen) override
    {
        if (shouldNavBarsBeHidden (shouldBeFullScreen))
        {
            if (isTimerRunning())
                return;

            startTimer (500);
        }
        else
        {
            setNavBarsHidden (false);
        }

        auto newBounds = [&]
        {
            if (navBarsHidden || shouldBeFullScreen)
                if (auto* display = Desktop::getInstance().getDisplays().getPrimaryDisplay())
                    return navBarsHidden ? display->totalArea
                                         : display->userArea;

            return lastNonFullscreenBounds.isEmpty() ? getBounds() : lastNonFullscreenBounds;
        }();

        if (! newBounds.isEmpty())
            setBounds (newBounds, shouldBeFullScreen);

        component.repaint();
    }

    bool isFullScreen() const override
    {
        return fullScreen;
    }

    void setIcon (const Image& /*newIcon*/) override
    {
        // n/a
    }

    bool contains (Point<int> localPos, bool trueIfInAChildWindow) const override
    {
        return isPositiveAndBelow (localPos.x, component.getWidth())
            && isPositiveAndBelow (localPos.y, component.getHeight())
            && ((! trueIfInAChildWindow) || view.callBooleanMethod (ComponentPeerView.containsPoint,
                                                                    (float) localPos.x * scale,
                                                                    (float) localPos.y * scale));
    }

    BorderSize<int> getFrameSize() const override
    {
        // TODO
        return {};
    }

    bool setAlwaysOnTop (bool /*alwaysOnTop*/) override
    {
        // TODO
        return false;
    }

    void toFront (bool makeActive) override
    {
        // Avoid calling bringToFront excessively: it's very slow
        if (frontWindow != this)
        {
            view.callVoidMethod (AndroidView.bringToFront);
            frontWindow = this;
        }

        if (makeActive)
            grabFocus();

        handleBroughtToFront();
    }

    void toBehind (ComponentPeer*) override
    {
        // TODO
    }

    //==============================================================================
    void handleMouseDownCallback (int index, Point<float> sysPos, int64 time)
    {
        lastMousePos = sysPos / scale;
        auto pos = globalToLocal (lastMousePos);

        // this forces a mouse-enter/up event, in case for some reason we didn't get a mouse-up before.
        handleMouseEvent (MouseInputSource::InputSourceType::touch,
                          pos,
                          ModifierKeys::currentModifiers.withoutMouseButtons(),
                          MouseInputSource::invalidPressure,
                          MouseInputSource::invalidOrientation,
                          time,
                          {},
                          index);

        if (isValidPeer (this))
            handleMouseDragCallback (index, sysPos, time);
    }

    void handleMouseDragCallback (int index, Point<float> sysPos, int64 time)
    {
        lastMousePos = sysPos / scale;
        auto pos = globalToLocal (lastMousePos);

        jassert (index < 64);
        touchesDown = (touchesDown | (1 << (index & 63)));

        ModifierKeys::currentModifiers = ModifierKeys::currentModifiers.withoutMouseButtons().withFlags (ModifierKeys::leftButtonModifier);

        handleMouseEvent (MouseInputSource::InputSourceType::touch,
                          pos,
                          ModifierKeys::currentModifiers.withoutMouseButtons().withFlags (ModifierKeys::leftButtonModifier),
                          MouseInputSource::invalidPressure,
                          MouseInputSource::invalidOrientation,
                          time,
                          {},
                          index);
    }

    void handleMouseUpCallback (int index, Point<float> sysPos, int64 time)
    {
        lastMousePos = sysPos / scale;
        auto pos = globalToLocal (lastMousePos);

        jassert (index < 64);
        touchesDown = (touchesDown & ~(1 << (index & 63)));

        if (touchesDown == 0)
            ModifierKeys::currentModifiers = ModifierKeys::currentModifiers.withoutMouseButtons();

        handleMouseEvent (MouseInputSource::InputSourceType::touch,
                          pos,
                          ModifierKeys::currentModifiers.withoutMouseButtons(),
                          MouseInputSource::invalidPressure,
                          MouseInputSource::invalidOrientation,
                          time,
                          {},
                          index);
    }

    void handleKeyDownCallback (int k, int kc)
    {
        handleKeyPress (k, static_cast<juce_wchar> (kc));
    }

    void handleKeyUpCallback (int /*k*/, int /*kc*/)
    {
    }

    void handleBackButtonCallback()
    {
        bool handled = false;

        if (auto* app = JUCEApplicationBase::getInstance())
            handled = app->backButtonPressed();

        if (isKioskModeComponent())
            setNavBarsHidden (navBarsHidden);

        if (! handled)
        {
            auto* env = getEnv();
            LocalRef<jobject> activity (getCurrentActivity());

            if (activity != nullptr)
            {
                jmethodID finishMethod = env->GetMethodID (AndroidActivity, "finish", "()V");

                if (finishMethod != nullptr)
                    env->CallVoidMethod (activity.get(), finishMethod);
            }
        }
    }

    void handleKeyboardHiddenCallback()
    {
        Component::unfocusAllComponents();
    }

    void handleAppPausedCallback() {}

    void handleAppResumedCallback()
    {
        if (isKioskModeComponent())
            setNavBarsHidden (navBarsHidden);
    }

    //==============================================================================
    AccessibilityNativeHandle* getNativeHandleForViewId (jint virtualViewId) const
    {
        if (auto* handler = (virtualViewId == HOST_VIEW_ID
                                 ? component.getAccessibilityHandler()
                                 : AccessibilityNativeHandle::getAccessibilityHandlerForVirtualViewId (virtualViewId)))
        {
            return handler->getNativeImplementation();
        }

        return nullptr;
    }

    jboolean populateAccessibilityNodeInfoCallback (jint virtualViewId, jobject info) const
    {
        if (auto* handle = getNativeHandleForViewId (virtualViewId))
        {
            handle->populateNodeInfo (info);
            return true;
        }

        return false;
    }

    jboolean handlePerformActionCallback (jint virtualViewId, jint action, jobject arguments) const
    {
        if (auto* handle = getNativeHandleForViewId (virtualViewId))
            return handle->performAction (action, arguments);

        return false;
    }

    static jobject getFocusViewIdForHandler (const AccessibilityHandler* handler)
    {
        if (handler != nullptr)
            return getEnv()->NewObject (JavaInteger,
                                        JavaInteger.constructor,
                                        handler->getNativeImplementation()->getVirtualViewId());

        return nullptr;
    }

    jobject getInputFocusViewIdCallback()
    {
        if (auto* comp = dynamic_cast<Component*> (findCurrentTextInputTarget()))
            return getFocusViewIdForHandler (comp->getAccessibilityHandler());

        return nullptr;
    }

    jobject getAccessibilityFocusViewIdCallback() const
    {
        if (auto* handler = component.getAccessibilityHandler())
        {
            if (auto* modal = Component::getCurrentlyModalComponent())
            {
                if (! component.isParentOf (modal)
                     && component.isCurrentlyBlockedByAnotherModalComponent())
                {
                    if (auto* modalHandler = modal->getAccessibilityHandler())
                    {
                        if (auto* focusChild = modalHandler->getChildFocus())
                            return getFocusViewIdForHandler (focusChild);

                        return getFocusViewIdForHandler (modalHandler);
                    }
                }
            }

            if (auto* focusChild = handler->getChildFocus())
                return getFocusViewIdForHandler (focusChild);
        }

        return nullptr;
    }

    //==============================================================================
    bool isFocused() const override
    {
        if (view != nullptr)
            return view.callBooleanMethod (AndroidView.hasFocus);

        return false;
    }

    void grabFocus() override
    {
        if (view != nullptr)
            view.callBooleanMethod (AndroidView.requestFocus);
    }

    void handleFocusChangeCallback (bool hasFocus)
    {
        if (isFullScreen())
            setFullScreen (true);

        if (hasFocus)
            handleFocusGain();
        else
            handleFocusLoss();
    }

    static const char* getVirtualKeyboardType (TextInputTarget::VirtualKeyboardType type) noexcept
    {
        switch (type)
        {
            case TextInputTarget::textKeyboard:          return "text";
            case TextInputTarget::numericKeyboard:       return "number";
            case TextInputTarget::decimalKeyboard:       return "numberDecimal";
            case TextInputTarget::urlKeyboard:           return "textUri";
            case TextInputTarget::emailAddressKeyboard:  return "textEmailAddress";
            case TextInputTarget::phoneNumberKeyboard:   return "phone";
            default:                                     jassertfalse; break;
        }

        return "text";
    }

    void textInputRequired (Point<int>, TextInputTarget& target) override
    {
        view.callVoidMethod (ComponentPeerView.showKeyboard,
                             javaString (getVirtualKeyboardType (target.getKeyboardType())).get());
    }

    void dismissPendingTextInput() override
    {
        view.callVoidMethod (ComponentPeerView.showKeyboard, javaString ("").get());

        if (! isTimerRunning())
            startTimer (500);
     }

    //==============================================================================
    void handlePaintCallback (jobject canvas, jobject paint)
    {
        auto* env = getEnv();

        jobject rect = env->CallObjectMethod (canvas, AndroidCanvas.getClipBounds);
        auto left   = env->GetIntField (rect, AndroidRect.left);
        auto top    = env->GetIntField (rect, AndroidRect.top);
        auto right  = env->GetIntField (rect, AndroidRect.right);
        auto bottom = env->GetIntField (rect, AndroidRect.bottom);
        env->DeleteLocalRef (rect);

        auto clip = Rectangle<int>::leftTopRightBottom (left, top, right, bottom);

        if (clip.isEmpty())
            return;

        auto sizeNeeded = clip.getWidth() * clip.getHeight();

        if (sizeAllocated < sizeNeeded)
        {
            buffer.clear();
            sizeAllocated = sizeNeeded;
            buffer = GlobalRef (LocalRef<jobject> ((jobject) env->NewIntArray (sizeNeeded)));
        }

        if (jint* dest = env->GetIntArrayElements ((jintArray) buffer.get(), nullptr))
        {
            {
                Image temp (new PreallocatedImage (clip.getWidth(), clip.getHeight(),
                                                   dest, ! component.isOpaque()));

                {
                    LowLevelGraphicsSoftwareRenderer g (temp);
                    g.setOrigin (-clip.getPosition());
                    g.addTransform (AffineTransform::scale (scale));
                    handlePaint (g);
                }
            }

            env->ReleaseIntArrayElements ((jintArray) buffer.get(), dest, 0);

            env->CallVoidMethod (canvas, AndroidCanvas.drawBitmap, (jintArray) buffer.get(), 0, clip.getWidth(),
                                 (jfloat) clip.getX(), (jfloat) clip.getY(),
                                 clip.getWidth(), clip.getHeight(), true, paint);
        }
    }

    void repaint (const Rectangle<int>& userArea) override
    {
        auto area = (userArea.toFloat() * scale).toNearestInt();

        GlobalRef localView (view);

        callOnMessageThread ([area, localView]
        {
            localView.callVoidMethod (AndroidView.invalidate,
                                      area.getX(), area.getY(), area.getRight(), area.getBottom());
        });
    }

    void performAnyPendingRepaintsNow() override
    {
        // TODO
    }

    void setAlpha (float /*newAlpha*/) override
    {
        // TODO
    }

    StringArray getAvailableRenderingEngines() override
    {
        return StringArray ("Software Renderer");
    }

    //==============================================================================
    static Point<float> lastMousePos;
    static int64 touchesDown;

    //==============================================================================
    struct StartupActivityCallbackListener  : public ActivityLifecycleCallbacks
    {
        void onActivityStarted (jobject /*activity*/) override
        {
            auto* env = getEnv();
            LocalRef<jobject> appContext (getAppContext());

            if (appContext.get() != nullptr)
            {
                env->CallVoidMethod (appContext.get(),
                                     AndroidApplication.unregisterActivityLifecycleCallbacks,
                                     activityCallbackListener.get());
                clear();
                activityCallbackListener.clear();

                forceDisplayUpdate();
            }
        }
    };

private:
    //==============================================================================
    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
     METHOD   (create,                           "<init>",                        "(Landroid/content/Context;ZJ)V") \
     METHOD   (clear,                            "clear",                         "()V") \
     METHOD   (setViewName,                      "setViewName",                   "(Ljava/lang/String;)V") \
     METHOD   (setVisible,                       "setVisible",                    "(Z)V") \
     METHOD   (isVisible,                        "isVisible",                     "()Z") \
     METHOD   (containsPoint,                    "containsPoint",                 "(II)Z") \
     METHOD   (showKeyboard,                     "showKeyboard",                  "(Ljava/lang/String;)V") \
     METHOD   (setSystemUiVisibilityCompat,      "setSystemUiVisibilityCompat",   "(I)V") \
     CALLBACK (handlePaintJni,                   "handlePaint",                   "(JLandroid/graphics/Canvas;Landroid/graphics/Paint;)V") \
     CALLBACK (handleMouseDownJni,               "handleMouseDown",               "(JIFFJ)V") \
     CALLBACK (handleMouseDragJni,               "handleMouseDrag",               "(JIFFJ)V") \
     CALLBACK (handleMouseUpJni,                 "handleMouseUp",                 "(JIFFJ)V") \
     CALLBACK (handleKeyDownJni,                 "handleKeyDown",                 "(JII)V") \
     CALLBACK (handleKeyUpJni,                   "handleKeyUp",                   "(JII)V") \
     CALLBACK (handleBackButtonJni,              "handleBackButton",              "(J)V") \
     CALLBACK (handleKeyboardHiddenJni,          "handleKeyboardHidden",          "(J)V") \
     CALLBACK (viewSizeChangedJni,               "viewSizeChanged",               "(J)V") \
     CALLBACK (focusChangedJni,                  "focusChanged",                  "(JZ)V") \
     CALLBACK (handleAppPausedJni,               "handleAppPaused",               "(J)V") \
     CALLBACK (handleAppResumedJni,              "handleAppResumed",              "(J)V") \
     CALLBACK (populateAccessibilityNodeInfoJni, "populateAccessibilityNodeInfo", "(JILandroid/view/accessibility/AccessibilityNodeInfo;)Z") \
     CALLBACK (handlePerformActionJni,           "handlePerformAction",           "(JIILandroid/os/Bundle;)Z") \
     CALLBACK (getInputFocusViewIdJni,           "getInputFocusViewId",           "(J)Ljava/lang/Integer;") \
     CALLBACK (getAccessibilityFocusViewIdJni,   "getAccessibilityFocusViewId",   "(J)Ljava/lang/Integer;") \

    DECLARE_JNI_CLASS_WITH_BYTECODE (ComponentPeerView, "com/rmsl/juce/ComponentPeerView", 16, javaComponentPeerView, sizeof (javaComponentPeerView))
    #undef JNI_CLASS_MEMBERS

    static void JNICALL handlePaintJni          (JNIEnv*, jobject /*view*/, jlong host, jobject canvas, jobject paint)           { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handlePaintCallback (canvas, paint); }
    static void JNICALL handleMouseDownJni      (JNIEnv*, jobject /*view*/, jlong host, jint i, jfloat x, jfloat y, jlong time)  { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMouseDownCallback (i, Point<float> ((float) x, (float) y), (int64) time); }
    static void JNICALL handleMouseDragJni      (JNIEnv*, jobject /*view*/, jlong host, jint i, jfloat x, jfloat y, jlong time)  { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMouseDragCallback (i, Point<float> ((float) x, (float) y), (int64) time); }
    static void JNICALL handleMouseUpJni        (JNIEnv*, jobject /*view*/, jlong host, jint i, jfloat x, jfloat y, jlong time)  { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMouseUpCallback   (i, Point<float> ((float) x, (float) y), (int64) time); }
    static void JNICALL viewSizeChangedJni      (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMovedOrResized(); }
    static void JNICALL focusChangedJni         (JNIEnv*, jobject /*view*/, jlong host, jboolean hasFocus)                       { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleFocusChangeCallback (hasFocus); }
    static void JNICALL handleKeyDownJni        (JNIEnv*, jobject /*view*/, jlong host, jint k, jint kc)                         { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleKeyDownCallback ((int) k, (int) kc); }
    static void JNICALL handleKeyUpJni          (JNIEnv*, jobject /*view*/, jlong host, jint k, jint kc)                         { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleKeyUpCallback ((int) k, (int) kc); }
    static void JNICALL handleBackButtonJni     (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleBackButtonCallback(); }
    static void JNICALL handleKeyboardHiddenJni (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleKeyboardHiddenCallback(); }
    static void JNICALL handleAppPausedJni      (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleAppPausedCallback(); }
    static void JNICALL handleAppResumedJni     (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleAppResumedCallback(); }

    static jboolean JNICALL populateAccessibilityNodeInfoJni (JNIEnv*, jobject /*view*/, jlong host, jint virtualViewId, jobject info)
    {
        if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host))
            return myself->populateAccessibilityNodeInfoCallback (virtualViewId, info);

        return false;
    }

    static jboolean JNICALL handlePerformActionJni (JNIEnv*, jobject /*view*/, jlong host, jint virtualViewId, jint action, jobject arguments)
    {
        if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host))
            return myself->handlePerformActionCallback (virtualViewId, action, arguments);

        return false;
    }

    static jobject JNICALL getInputFocusViewIdJni (JNIEnv*, jobject /*view*/, jlong host)
    {
        if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host))
            return myself->getInputFocusViewIdCallback();

        return nullptr;
    }

    static jobject JNICALL getAccessibilityFocusViewIdJni (JNIEnv*, jobject /*view*/, jlong host)
    {
        if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host))
            return myself->getAccessibilityFocusViewIdCallback();

        return nullptr;
    }

    //==============================================================================
    struct ViewWindowInsetsListener  : public juce::AndroidInterfaceImplementer
    {
        jobject onApplyWindowInsets (LocalRef<jobject> v, LocalRef<jobject> insets)
        {
            auto* env = getEnv();

            LocalRef<jobject> displayCutout (env->CallObjectMethod (insets.get(), AndroidWindowInsets.getDisplayCutout));

            if (displayCutout != nullptr)
            {
                const auto& mainDisplay = *Desktop::getInstance().getDisplays().getPrimaryDisplay();
                auto newSafeAreaInsets = androidDisplayCutoutToBorderSize (displayCutout, mainDisplay.scale);

                if (newSafeAreaInsets != mainDisplay.safeAreaInsets)
                    forceDisplayUpdate();

                auto* fieldId = env->GetStaticFieldID (AndroidWindowInsets, "CONSUMED", "Landroid/view/WindowInsets");
                jassert (fieldId != nullptr);

                return env->GetStaticObjectField (AndroidWindowInsets, fieldId);
            }

            jmethodID onApplyWindowInsetsMethodId = env->GetMethodID (AndroidView,
                                                                      "onApplyWindowInsets",
                                                                      "(Landroid/view/WindowInsets;)Landroid/view/WindowInsets;");

            jassert (onApplyWindowInsetsMethodId != nullptr);

            return env->CallObjectMethod (v.get(), onApplyWindowInsetsMethodId, insets.get());
        }

    private:
        jobject invoke (jobject proxy, jobject method, jobjectArray args) override
        {
            auto* env = getEnv();
            auto methodName = juce::juceString ((jstring) env->CallObjectMethod (method, JavaMethod.getName));

            if (methodName == "onApplyWindowInsets")
            {
                jassert (env->GetArrayLength (args) == 2);

                LocalRef<jobject> windowView (env->GetObjectArrayElement (args, 0));
                LocalRef<jobject> insets     (env->GetObjectArrayElement (args, 1));

                return onApplyWindowInsets (std::move (windowView), std::move (insets));
            }

            // invoke base class
            return AndroidInterfaceImplementer::invoke (proxy, method, args);
        }
    };

    //==============================================================================
    struct PreallocatedImage  : public ImagePixelData
    {
        PreallocatedImage (int width_, int height_, jint* data_, bool hasAlpha_)
            : ImagePixelData (Image::ARGB, width_, height_), data (data_), hasAlpha (hasAlpha_)
        {
            if (hasAlpha_)
                zeromem (data_, static_cast<size_t> (width * height) * sizeof (jint));
        }

        ~PreallocatedImage() override
        {
            if (hasAlpha)
            {
                auto pix = (PixelARGB*) data;

                for (int i = width * height; --i >= 0;)
                {
                    pix->unpremultiply();
                    ++pix;
                }
            }
        }

        std::unique_ptr<ImageType> createType() const override
        {
            return std::make_unique<SoftwareImageType>();
        }

        std::unique_ptr<LowLevelGraphicsContext> createLowLevelContext() override
        {
            return std::make_unique<LowLevelGraphicsSoftwareRenderer> (Image (this));
        }

        void initialiseBitmapData (Image::BitmapData& bm, int x, int y, Image::BitmapData::ReadWriteMode /*mode*/) override
        {
            bm.lineStride = width * static_cast<int> (sizeof (jint));
            bm.pixelStride = static_cast<int> (sizeof (jint));
            bm.pixelFormat = Image::ARGB;
            bm.data = (uint8*) (data + x + y * width);
        }

        ImagePixelData::Ptr clone() override
        {
            auto s = new PreallocatedImage (width, height, nullptr, hasAlpha);
            s->allocatedData.malloc (sizeof (jint) * static_cast<size_t> (width * height));
            s->data = s->allocatedData;
            memcpy (s->data, data, sizeof (jint) * static_cast<size_t> (width * height));
            return s;
        }

    private:
        jint* data;
        HeapBlock<jint> allocatedData;
        bool hasAlpha;

        JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (PreallocatedImage)
    };

    //==============================================================================
    void timerCallback() override
    {
        setNavBarsHidden (shouldNavBarsBeHidden (fullScreen));
        setFullScreen (fullScreen);
        stopTimer();
    }

    bool isKioskModeComponent() const
    {
        if (auto* kiosk = Desktop::getInstance().getKioskModeComponent())
            return kiosk->getPeer() == this;

        return false;
    }

    bool shouldNavBarsBeHidden (bool shouldBeFullScreen) const
    {
        return (shouldBeFullScreen && isKioskModeComponent());
    }

    void setNavBarsHidden (bool hidden)
    {
        if (navBarsHidden != hidden)
        {
            navBarsHidden = hidden;

            view.callVoidMethod (ComponentPeerView.setSystemUiVisibilityCompat,
                                 (navBarsHidden ? (jint) (fullScreenFlags) : (jint) (SYSTEM_UI_FLAG_VISIBLE)));
        }
    }

    template <typename Callback>
    static void callOnMessageThread (Callback&& callback)
    {
        if (MessageManager::getInstance()->isThisTheMessageThread())
            callback();
        else
            MessageManager::callAsync (std::forward<Callback> (callback));
    }

    //==============================================================================
    friend class Displays;
    static AndroidComponentPeer* frontWindow;
    static GlobalRef activityCallbackListener;

    static constexpr int GRAVITY_LEFT = 0x3, GRAVITY_TOP = 0x30;
    static constexpr int TYPE_APPLICATION = 0x2;
    static constexpr int FLAG_NOT_TOUCH_MODAL = 0x20, FLAG_LAYOUT_IN_SCREEN = 0x100, FLAG_LAYOUT_NO_LIMITS = 0x200;
    static constexpr int PIXEL_FORMAT_OPAQUE = -1, PIXEL_FORMAT_TRANSPARENT = -2;
    static constexpr int LAYOUT_IN_DISPLAY_CUTOUT_MODE_ALWAYS = 0x3;

    GlobalRef view, viewGroup, buffer;
    bool viewGroupIsWindow = false, fullScreen = false, navBarsHidden = false;
    int sizeAllocated = 0;
    float scale = (float) Desktop::getInstance().getDisplays().getPrimaryDisplay()->scale;

    //==============================================================================
    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (AndroidComponentPeer)
};

Point<float> AndroidComponentPeer::lastMousePos;
int64 AndroidComponentPeer::touchesDown = 0;
AndroidComponentPeer* AndroidComponentPeer::frontWindow = nullptr;
GlobalRef AndroidComponentPeer::activityCallbackListener;
AndroidComponentPeer::ComponentPeerView_Class AndroidComponentPeer::ComponentPeerView;

//==============================================================================
ComponentPeer* Component::createNewPeer (int styleFlags, void* nativeWindow)
{
    return new AndroidComponentPeer (*this, styleFlags, nativeWindow);
}

//==============================================================================
bool Desktop::canUseSemiTransparentWindows() noexcept
{
    return true;
}

double Desktop::getDefaultMasterScale()
{
    return 1.0;
}

Desktop::DisplayOrientation Desktop::getCurrentOrientation() const
{
    enum
    {
        ROTATION_0   = 0,
        ROTATION_90  = 1,
        ROTATION_180 = 2,
        ROTATION_270 = 3
    };

    JNIEnv* env = getEnv();
    LocalRef<jstring> windowServiceString (javaString ("window"));


    LocalRef<jobject> windowManager = LocalRef<jobject> (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, windowServiceString.get()));

    if (windowManager.get() != nullptr)
    {
        LocalRef<jobject> display = LocalRef<jobject> (env->CallObjectMethod (windowManager, AndroidWindowManager.getDefaultDisplay));

        if (display.get() != nullptr)
        {
            int rotation = env->CallIntMethod (display, AndroidDisplay.getRotation);

            switch (rotation)
            {
                case ROTATION_0:   return upright;
                case ROTATION_90:  return rotatedAntiClockwise;
                case ROTATION_180: return upsideDown;
                case ROTATION_270: return rotatedClockwise;
            }
        }
    }

    jassertfalse;
    return upright;
}

bool MouseInputSource::SourceList::addSource()
{
    addSource (sources.size(), MouseInputSource::InputSourceType::touch);
    return true;
}

bool MouseInputSource::SourceList::canUseTouch()
{
    return true;
}

Point<float> MouseInputSource::getCurrentRawMousePosition()
{
    return AndroidComponentPeer::lastMousePos;
}

void MouseInputSource::setRawMousePosition (Point<float>)
{
    // not needed
}

//==============================================================================
bool KeyPress::isKeyCurrentlyDown (int /*keyCode*/)
{
    // TODO
    return false;
}

JUCE_API void JUCE_CALLTYPE Process::hide()
{
    auto* env = getEnv();
    LocalRef<jobject> currentActivity (getCurrentActivity().get());

    if (env->CallBooleanMethod (currentActivity.get(), AndroidActivity.moveTaskToBack, true) == 0)
    {
        GlobalRef intent (LocalRef<jobject> (env->NewObject (AndroidIntent, AndroidIntent.constructor)));
        env->CallObjectMethod (intent, AndroidIntent.setAction,   javaString ("android.intent.action.MAIN")  .get());
        env->CallObjectMethod (intent, AndroidIntent.addCategory, javaString ("android.intent.category.HOME").get());

        env->CallVoidMethod (currentActivity.get(), AndroidContext.startActivity, intent.get());
    }
}

//==============================================================================
// TODO
JUCE_API bool JUCE_CALLTYPE Process::isForegroundProcess() { return true; }
JUCE_API void JUCE_CALLTYPE Process::makeForegroundProcess() {}

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (show,                   "show",                 "()V") \
 METHOD (getWindow,              "getWindow",            "()Landroid/view/Window;")

DECLARE_JNI_CLASS (AndroidDialog, "android/app/Dialog")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (construct,                   "<init>",                 "(Landroid/content/Context;)V") \
 METHOD (create,                      "create",                 "()Landroid/app/AlertDialog;") \
 METHOD (setTitle,                    "setTitle",               "(Ljava/lang/CharSequence;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setMessage,                  "setMessage",             "(Ljava/lang/CharSequence;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setCancelable,               "setCancelable",          "(Z)Landroid/app/AlertDialog$Builder;") \
 METHOD (setOnCancelListener,         "setOnCancelListener",    "(Landroid/content/DialogInterface$OnCancelListener;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setPositiveButton,           "setPositiveButton",      "(Ljava/lang/CharSequence;Landroid/content/DialogInterface$OnClickListener;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setNegativeButton,           "setNegativeButton",      "(Ljava/lang/CharSequence;Landroid/content/DialogInterface$OnClickListener;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setNeutralButton,            "setNeutralButton",       "(Ljava/lang/CharSequence;Landroid/content/DialogInterface$OnClickListener;)Landroid/app/AlertDialog$Builder;")

DECLARE_JNI_CLASS (AndroidAlertDialogBuilder, "android/app/AlertDialog$Builder")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (dismiss,    "dismiss",  "()V")

DECLARE_JNI_CLASS (AndroidDialogInterface, "android/content/DialogInterface")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \

DECLARE_JNI_CLASS (AndroidDialogOnClickListener, "android/content/DialogInterface$OnClickListener")
#undef JNI_CLASS_MEMBERS

//==============================================================================
class DialogListener  : public juce::AndroidInterfaceImplementer
{
public:
    DialogListener (std::shared_ptr<ModalComponentManager::Callback> callbackToUse, int resultToUse)
        : callback (std::move (callbackToUse)), result (resultToUse)
    {}

    void onResult (jobject dialog)
    {
        auto* env = getEnv();
        env->CallVoidMethod (dialog, AndroidDialogInterface.dismiss);

        if (callback != nullptr)
            callback->modalStateFinished (result);

        callback = nullptr;
    }

private:
    jobject invoke (jobject proxy, jobject method, jobjectArray args) override
    {
        auto* env = getEnv();
        auto methodName = juce::juceString ((jstring) env->CallObjectMethod (method, JavaMethod.getName));

        if (methodName == "onCancel" || methodName == "onClick")
        {
            onResult (env->GetObjectArrayElement (args, 0));
            return nullptr;
        }

        // invoke base class
        return AndroidInterfaceImplementer::invoke (proxy, method, args);
    }

    std::shared_ptr<ModalComponentManager::Callback> callback;
    int result;
};

//==============================================================================
static void createAndroidDialog (const MessageBoxOptions& opts,
                                 std::unique_ptr<ModalComponentManager::Callback> callback)
{
    auto* env = getEnv();

    LocalRef<jobject> builder (env->NewObject (AndroidAlertDialogBuilder, AndroidAlertDialogBuilder.construct, getMainActivity().get()));

    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setTitle,   javaString (opts.getTitle()).get()));
    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setMessage, javaString (opts.getMessage()).get()));
    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setCancelable, true));

    std::shared_ptr<ModalComponentManager::Callback> sharedCallback (std::move (callback));

    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setOnCancelListener,
                                                        CreateJavaInterface (new DialogListener (sharedCallback, 0),
                                                                             "android/content/DialogInterface$OnCancelListener").get()));

    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setPositiveButton,
                                                        javaString (opts.getButtonText (0)).get(),
                                                        CreateJavaInterface (new DialogListener (sharedCallback, 0),
                                                                             "android/content/DialogInterface$OnClickListener").get()));

    if (opts.getButtonText (1).isNotEmpty())
        builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setNegativeButton,
                                                            javaString (opts.getButtonText (1)).get(),
                                                            CreateJavaInterface (new DialogListener (sharedCallback, 1),
                                                                                 "android/content/DialogInterface$OnClickListener").get()));

    if (opts.getButtonText (2).isNotEmpty())
        builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setNeutralButton,
                                                            javaString (opts.getButtonText (2)).get(),
                                                            CreateJavaInterface (new DialogListener (sharedCallback, 2),
                                                                                 "android/content/DialogInterface$OnClickListener").get()));

    LocalRef<jobject> dialog (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.create));

    LocalRef<jobject> window (env->CallObjectMethod (dialog.get(), AndroidDialog.getWindow));

    if (Desktop::getInstance().getKioskModeComponent() != nullptr)
    {
        env->CallVoidMethod (window.get(), AndroidWindow.setFlags, FLAG_NOT_FOCUSABLE, FLAG_NOT_FOCUSABLE);
        LocalRef<jobject> decorView (env->CallObjectMethod (window.get(), AndroidWindow.getDecorView));
        env->CallVoidMethod (decorView.get(), AndroidView.setSystemUiVisibility, fullScreenFlags);
    }

    env->CallVoidMethod (dialog.get(), AndroidDialog.show);

    if (Desktop::getInstance().getKioskModeComponent() != nullptr)
        env->CallVoidMethod (window.get(), AndroidWindow.clearFlags, FLAG_NOT_FOCUSABLE);
}

void JUCE_CALLTYPE NativeMessageBox::showMessageBoxAsync (MessageBoxIconType /*iconType*/,
                                                          const String& title, const String& message,
                                                          Component* /*associatedComponent*/,
                                                          ModalComponentManager::Callback* callback)
{
    showAsync (MessageBoxOptions()
                 .withTitle (title)
                 .withMessage (message)
                 .withButton (TRANS("OK")),
               AlertWindowMappings::getWrappedCallback (callback, AlertWindowMappings::messageBox));
}

bool JUCE_CALLTYPE NativeMessageBox::showOkCancelBox (MessageBoxIconType /*iconType*/,
                                                      const String& title, const String& message,
                                                      Component* /*associatedComponent*/,
                                                      ModalComponentManager::Callback* callback)
{
    showAsync (MessageBoxOptions()
                 .withTitle (title)
                 .withMessage (message)
                 .withButton (TRANS("OK"))
                 .withButton (TRANS("Cancel")),
               AlertWindowMappings::getWrappedCallback (callback, AlertWindowMappings::okCancel));

    return false;
}

int JUCE_CALLTYPE NativeMessageBox::showYesNoCancelBox (MessageBoxIconType /*iconType*/,
                                                        const String& title, const String& message,
                                                        Component* /*associatedComponent*/,
                                                        ModalComponentManager::Callback* callback)
{
    showAsync (MessageBoxOptions()
                 .withTitle (title)
                 .withMessage (message)
                 .withButton (TRANS("Yes"))
                 .withButton (TRANS("No"))
                 .withButton (TRANS("Cancel")),
               AlertWindowMappings::getWrappedCallback (callback, AlertWindowMappings::yesNoCancel));

    return 0;
}

int JUCE_CALLTYPE NativeMessageBox::showYesNoBox (MessageBoxIconType /*iconType*/,
                                                  const String& title, const String& message,
                                                  Component* /*associatedComponent*/,
                                                  ModalComponentManager::Callback* callback)
{
    showAsync (MessageBoxOptions()
                 .withTitle (title)
                 .withMessage (message)
                 .withButton (TRANS("Yes"))
                 .withButton (TRANS("No")),
               AlertWindowMappings::getWrappedCallback (callback, AlertWindowMappings::okCancel));

    return 0;
}

void JUCE_CALLTYPE NativeMessageBox::showAsync (const MessageBoxOptions& options,
                                                ModalComponentManager::Callback* callback)
{
    createAndroidDialog (options, std::unique_ptr<ModalComponentManager::Callback> (callback));
}

void JUCE_CALLTYPE NativeMessageBox::showAsync (const MessageBoxOptions& options,
                                                std::function<void (int)> callback)
{
    showAsync (options, ModalCallbackFunction::create (callback));
}

//==============================================================================
static bool androidScreenSaverEnabled = true;

void Desktop::setScreenSaverEnabled (bool shouldEnable)
{
    constexpr auto FLAG_KEEP_SCREEN_ON = 0x80;

    if (shouldEnable != androidScreenSaverEnabled)
    {
        LocalRef<jobject> activity (getMainActivity());

        if (activity != nullptr)
        {
            auto* env = getEnv();

            LocalRef<jobject> mainWindow (env->CallObjectMethod (activity.get(), AndroidActivity.getWindow));
            env->CallVoidMethod (mainWindow.get(), AndroidWindow.setFlags, shouldEnable ? 0 : FLAG_KEEP_SCREEN_ON, FLAG_KEEP_SCREEN_ON);
        }

        androidScreenSaverEnabled = shouldEnable;
    }
}

bool Desktop::isScreenSaverEnabled()
{
    return androidScreenSaverEnabled;
}

//==============================================================================
void Desktop::setKioskComponent (Component* kioskComp, bool enableOrDisable, bool allowMenusAndBars)
{
    ignoreUnused (allowMenusAndBars);

    if (AndroidComponentPeer* peer = dynamic_cast<AndroidComponentPeer*> (kioskComp->getPeer()))
        peer->setFullScreen (enableOrDisable);
    else
        jassertfalse; // (this should have been checked by the caller)
}

//==============================================================================
static jint getAndroidOrientationFlag (int orientations) noexcept
{
    enum
    {
        SCREEN_ORIENTATION_LANDSCAPE          = 0,
        SCREEN_ORIENTATION_PORTRAIT           = 1,
        SCREEN_ORIENTATION_USER               = 2,
        SCREEN_ORIENTATION_REVERSE_LANDSCAPE  = 8,
        SCREEN_ORIENTATION_REVERSE_PORTRAIT   = 9,
        SCREEN_ORIENTATION_USER_LANDSCAPE     = 11,
        SCREEN_ORIENTATION_USER_PORTRAIT      = 12,
    };

    switch (orientations)
    {
        case Desktop::upright:                                          return (jint) SCREEN_ORIENTATION_PORTRAIT;
        case Desktop::upsideDown:                                       return (jint) SCREEN_ORIENTATION_REVERSE_PORTRAIT;
        case Desktop::upright + Desktop::upsideDown:                    return (jint) SCREEN_ORIENTATION_USER_PORTRAIT;
        case Desktop::rotatedAntiClockwise:                             return (jint) SCREEN_ORIENTATION_LANDSCAPE;
        case Desktop::rotatedClockwise:                                 return (jint) SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
        case Desktop::rotatedClockwise + Desktop::rotatedAntiClockwise: return (jint) SCREEN_ORIENTATION_USER_LANDSCAPE;
        default:                                                        return (jint) SCREEN_ORIENTATION_USER;
    }
}

void Desktop::allowedOrientationsChanged()
{
    LocalRef<jobject> activity (getMainActivity());

    if (activity != nullptr)
        getEnv()->CallVoidMethod (activity.get(), AndroidActivity.setRequestedOrientation, getAndroidOrientationFlag (allowedOrientations));
}

//==============================================================================
bool juce_areThereAnyAlwaysOnTopWindows()
{
    return false;
}

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (create,          "<init>",         "()V") \
 FIELD  (density,         "density",        "F") \
 FIELD  (widthPixels,     "widthPixels",    "I") \
 FIELD  (heightPixels,    "heightPixels",   "I")

DECLARE_JNI_CLASS (AndroidDisplayMetrics, "android/util/DisplayMetrics")
#undef JNI_CLASS_MEMBERS

//==============================================================================
class LayoutChangeListener  : public juce::AndroidInterfaceImplementer
{
public:
    virtual void onLayoutChange (LocalRef<jobject> view, int left, int top, int right, int bottom,
                                 int oldLeft, int oldTop, int oldRight, int oldBottom) = 0;

private:
    jobject invoke (jobject proxy, jobject method, jobjectArray args) override
    {
        auto* env = getEnv();
        auto methodName = juce::juceString ((jstring) env->CallObjectMethod (method, JavaMethod.getName));

        if (methodName == "onLayoutChange")
        {
            jassert (env->GetArrayLength (args) == 9);

            LocalRef<jobject> view (env->GetObjectArrayElement (args, 0));
            int dims[8];

            for (int i = 1; i < 9; ++i)
            {
                LocalRef<jobject> integer (env->GetObjectArrayElement (args, i));
                dims[i - 1] = env->CallIntMethod (integer.get(), JavaInteger.intValue);
            }

            onLayoutChange (std::move (view), dims[0], dims[1], dims[2], dims[3],
                            dims[4], dims[5], dims[6], dims[7]);

            return nullptr;
        }

        // invoke base class
        return AndroidInterfaceImplementer::invoke (proxy, method, args);
    }

    std::unique_ptr<ModalComponentManager::Callback> callback;
};

//==============================================================================
struct MainActivityWindowLayoutListener   : public LayoutChangeListener
{
    MainActivityWindowLayoutListener (std::function<void()>&& updateDisplaysCb)
        : forceDisplayUpdate (std::move (updateDisplaysCb))
    {
    }

    void onLayoutChange (LocalRef<jobject> /*view*/, int left, int top, int right, int bottom,
                         int oldLeft, int oldTop, int oldRight, int oldBottom) override
    {
        auto newBounds = Rectangle<int>::leftTopRightBottom (left, top, right, bottom);
        auto oldBounds = Rectangle<int>::leftTopRightBottom (oldLeft, oldTop, oldRight, oldBottom);

        if (newBounds != oldBounds)
        {
            const auto& mainDisplay = *Desktop::getInstance().getDisplays().getPrimaryDisplay();
            auto userArea = (newBounds.toFloat() / mainDisplay.scale).toNearestInt();

            if (userArea != mainDisplay.userArea)
                forceDisplayUpdate();
        }
    }

    std::function<void()> forceDisplayUpdate;
};

//==============================================================================
void Displays::findDisplays (float masterScale)
{
    auto* env = getEnv();

    LocalRef<jobject> usableSize (env->NewObject (AndroidPoint, AndroidPoint.create, 0, 0));
    LocalRef<jstring> windowServiceString (javaString ("window"));
    LocalRef<jobject> displayMetrics (env->NewObject (AndroidDisplayMetrics, AndroidDisplayMetrics.create));
    LocalRef<jobject> windowManager (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, windowServiceString.get()));
    LocalRef <jobject> display (env->CallObjectMethod (windowManager, AndroidWindowManager.getDefaultDisplay));

    jmethodID getRealMetricsMethod = env->GetMethodID (AndroidDisplay, "getRealMetrics", "(Landroid/util/DisplayMetrics;)V");

    if (getRealMetricsMethod != nullptr)
        env->CallVoidMethod (display.get(), getRealMetricsMethod, displayMetrics.get());
    else
        env->CallVoidMethod (display.get(), AndroidDisplay.getMetrics, displayMetrics.get());

    env->CallVoidMethod (display.get(), AndroidDisplay.getSize, usableSize.get());

    Display d;

    d.isMain = true;
    d.scale = env->GetFloatField (displayMetrics.get(), AndroidDisplayMetrics.density);
    d.dpi = (d.scale * 160.f);
    d.scale *= masterScale;

    d.totalArea = Rectangle<int> (env->GetIntField (displayMetrics.get(), AndroidDisplayMetrics.widthPixels),
                                  env->GetIntField (displayMetrics.get(), AndroidDisplayMetrics.heightPixels)) / d.scale;

    d.userArea = Rectangle<int> (env->GetIntField (usableSize.get(), AndroidPoint.x),
                                 env->GetIntField (usableSize.get(), AndroidPoint.y)) / d.scale;

    // unfortunately usableSize still contains the nav bar
    // the best workaround is to try to get the size of the top-level view of
    // the main activity
    LocalRef<jobject> activity (getMainActivity());

    if (activity != nullptr)
    {
        LocalRef<jobject> mainWindow (env->CallObjectMethod (activity.get(), AndroidActivity.getWindow));
        LocalRef<jobject> decorView (env->CallObjectMethod (mainWindow.get(), AndroidWindow.getDecorView));
        LocalRef<jobject> contentView (env->CallObjectMethod (decorView.get(), AndroidView.findViewById, 0x01020002 /* android.R.id.content */));

        if (contentView != nullptr)
        {
            Rectangle<int> activityArea (env->CallIntMethod (contentView.get(), AndroidView.getLeft),
                                         env->CallIntMethod (contentView.get(), AndroidView.getTop),
                                         env->CallIntMethod (contentView.get(), AndroidView.getWidth),
                                         env->CallIntMethod (contentView.get(), AndroidView.getHeight));

            if (! activityArea.isEmpty())
                d.userArea = activityArea / d.scale;

            if (supportsDisplayCutout())
            {
                jmethodID getRootWindowInsetsMethodId = env->GetMethodID (AndroidView,
                                                                          "getRootWindowInsets",
                                                                          "()Landroid/view/WindowInsets;");

                if (getRootWindowInsetsMethodId != nullptr)
                {
                    LocalRef<jobject> insets (env->CallObjectMethod (contentView.get(), getRootWindowInsetsMethodId));

                    if (insets != nullptr)
                    {
                        LocalRef<jobject> displayCutout (env->CallObjectMethod (insets.get(), AndroidWindowInsets.getDisplayCutout));

                        if (displayCutout.get() != nullptr)
                            d.safeAreaInsets = androidDisplayCutoutToBorderSize (displayCutout, d.scale);
                    }
                }
            }

            static bool hasAddedMainActivityListener = false;

            if (! hasAddedMainActivityListener)
            {
                hasAddedMainActivityListener = true;

                env->CallVoidMethod (contentView.get(), AndroidView.addOnLayoutChangeListener,
                                     CreateJavaInterface (new MainActivityWindowLayoutListener ([this] { refresh(); }),
                                                          "android/view/View$OnLayoutChangeListener").get());
            }
        }
    }
    else
    {
        // the main activity may have not started yet so add an activity listener
        if (AndroidComponentPeer::activityCallbackListener == nullptr)
        {
            LocalRef<jobject> appContext (getAppContext());

            if (appContext.get() != nullptr)
            {
                AndroidComponentPeer::activityCallbackListener = GlobalRef (CreateJavaInterface (
                        new AndroidComponentPeer::StartupActivityCallbackListener,
                        "android/app/Application$ActivityLifecycleCallbacks"));

                env->CallVoidMethod (appContext.get(),
                                     AndroidApplication.registerActivityLifecycleCallbacks,
                                     AndroidComponentPeer::activityCallbackListener.get());
            }
        }
    }

    displays.add (d);
}

//==============================================================================
Image juce_createIconForFile (const File& /*file*/)
{
    return Image();
}

//==============================================================================
void* CustomMouseCursorInfo::create() const                                         { return nullptr; }
void* MouseCursor::createStandardMouseCursor (MouseCursor::StandardCursorType)      { return nullptr; }
void MouseCursor::deleteMouseCursor (void* /*cursorHandle*/, bool /*isStandard*/)   {}

//==============================================================================
void MouseCursor::showInWindow (ComponentPeer*) const   {}

//==============================================================================
bool DragAndDropContainer::performExternalDragDropOfFiles (const StringArray& /*files*/, bool /*canMove*/,
                                                           Component* /*srcComp*/, std::function<void()> /*callback*/)
{
    jassertfalse;    // no such thing on Android!
    return false;
}

bool DragAndDropContainer::performExternalDragDropOfText (const String& /*text*/, Component* /*srcComp*/,
                                                          std::function<void()> /*callback*/)
{
    jassertfalse;    // no such thing on Android!
    return false;
}

//==============================================================================
void LookAndFeel::playAlertSound()
{
}

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (getText,      "getText",            "()Ljava/lang/CharSequence;") \
 METHOD (setText,      "setText",            "(Ljava/lang/CharSequence;)V")

DECLARE_JNI_CLASS (AndroidClipboardManager, "android/content/ClipboardManager")
#undef JNI_CLASS_MEMBERS

//==============================================================================
void SystemClipboard::copyTextToClipboard (const String& text)
{
    auto* env = getEnv();

    LocalRef<jobject> clipboardManager (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, javaString ("clipboard").get()));
    env->CallVoidMethod (clipboardManager.get(), AndroidClipboardManager.setText, javaString(text).get());
}

String SystemClipboard::getTextFromClipboard()
{
    auto* env = getEnv();

    LocalRef<jobject> clipboardManager (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, javaString ("clipboard").get()));
    LocalRef<jobject> charSequence (env->CallObjectMethod (clipboardManager.get(), AndroidClipboardManager.getText));

    if (charSequence == nullptr)
        return {};

    return juceString(LocalRef<jstring> ((jstring) env->CallObjectMethod(charSequence.get(), JavaCharSequence.toString)));
}

//==============================================================================
const int extendedKeyModifier       = 0x10000;

const int KeyPress::spaceKey        = ' ';
const int KeyPress::returnKey       = 66;
const int KeyPress::escapeKey       = 4;
const int KeyPress::backspaceKey    = 67;
const int KeyPress::leftKey         = extendedKeyModifier + 1;
const int KeyPress::rightKey        = extendedKeyModifier + 2;
const int KeyPress::upKey           = extendedKeyModifier + 3;
const int KeyPress::downKey         = extendedKeyModifier + 4;
const int KeyPress::pageUpKey       = extendedKeyModifier + 5;
const int KeyPress::pageDownKey     = extendedKeyModifier + 6;
const int KeyPress::endKey          = extendedKeyModifier + 7;
const int KeyPress::homeKey         = extendedKeyModifier + 8;
const int KeyPress::deleteKey       = extendedKeyModifier + 9;
const int KeyPress::insertKey       = -1;
const int KeyPress::tabKey          = 61;
const int KeyPress::F1Key           = extendedKeyModifier + 10;
const int KeyPress::F2Key           = extendedKeyModifier + 11;
const int KeyPress::F3Key           = extendedKeyModifier + 12;
const int KeyPress::F4Key           = extendedKeyModifier + 13;
const int KeyPress::F5Key           = extendedKeyModifier + 14;
const int KeyPress::F6Key           = extendedKeyModifier + 16;
const int KeyPress::F7Key           = extendedKeyModifier + 17;
const int KeyPress::F8Key           = extendedKeyModifier + 18;
const int KeyPress::F9Key           = extendedKeyModifier + 19;
const int KeyPress::F10Key          = extendedKeyModifier + 20;
const int KeyPress::F11Key          = extendedKeyModifier + 21;
const int KeyPress::F12Key          = extendedKeyModifier + 22;
const int KeyPress::F13Key          = extendedKeyModifier + 23;
const int KeyPress::F14Key          = extendedKeyModifier + 24;
const int KeyPress::F15Key          = extendedKeyModifier + 25;
const int KeyPress::F16Key          = extendedKeyModifier + 26;
const int KeyPress::F17Key          = extendedKeyModifier + 50;
const int KeyPress::F18Key          = extendedKeyModifier + 51;
const int KeyPress::F19Key          = extendedKeyModifier + 52;
const int KeyPress::F20Key          = extendedKeyModifier + 53;
const int KeyPress::F21Key          = extendedKeyModifier + 54;
const int KeyPress::F22Key          = extendedKeyModifier + 55;
const int KeyPress::F23Key          = extendedKeyModifier + 56;
const int KeyPress::F24Key          = extendedKeyModifier + 57;
const int KeyPress::F25Key          = extendedKeyModifier + 58;
const int KeyPress::F26Key          = extendedKeyModifier + 59;
const int KeyPress::F27Key          = extendedKeyModifier + 60;
const int KeyPress::F28Key          = extendedKeyModifier + 61;
const int KeyPress::F29Key          = extendedKeyModifier + 62;
const int KeyPress::F30Key          = extendedKeyModifier + 63;
const int KeyPress::F31Key          = extendedKeyModifier + 64;
const int KeyPress::F32Key          = extendedKeyModifier + 65;
const int KeyPress::F33Key          = extendedKeyModifier + 66;
const int KeyPress::F34Key          = extendedKeyModifier + 67;
const int KeyPress::F35Key          = extendedKeyModifier + 68;
const int KeyPress::numberPad0      = extendedKeyModifier + 27;
const int KeyPress::numberPad1      = extendedKeyModifier + 28;
const int KeyPress::numberPad2      = extendedKeyModifier + 29;
const int KeyPress::numberPad3      = extendedKeyModifier + 30;
const int KeyPress::numberPad4      = extendedKeyModifier + 31;
const int KeyPress::numberPad5      = extendedKeyModifier + 32;
const int KeyPress::numberPad6      = extendedKeyModifier + 33;
const int KeyPress::numberPad7      = extendedKeyModifier + 34;
const int KeyPress::numberPad8      = extendedKeyModifier + 35;
const int KeyPress::numberPad9      = extendedKeyModifier + 36;
const int KeyPress::numberPadAdd            = extendedKeyModifier + 37;
const int KeyPress::numberPadSubtract       = extendedKeyModifier + 38;
const int KeyPress::numberPadMultiply       = extendedKeyModifier + 39;
const int KeyPress::numberPadDivide         = extendedKeyModifier + 40;
const int KeyPress::numberPadSeparator      = extendedKeyModifier + 41;
const int KeyPress::numberPadDecimalPoint   = extendedKeyModifier + 42;
const int KeyPress::numberPadEquals         = extendedKeyModifier + 43;
const int KeyPress::numberPadDelete         = extendedKeyModifier + 44;
const int KeyPress::playKey         = extendedKeyModifier + 45;
const int KeyPress::stopKey         = extendedKeyModifier + 46;
const int KeyPress::fastForwardKey  = extendedKeyModifier + 47;
const int KeyPress::rewindKey       = extendedKeyModifier + 48;

//==============================================================================
#ifdef JUCE_PUSH_NOTIFICATIONS_ACTIVITY
 struct JuceActivityNewIntentListener
 {
     #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
      CALLBACK (appNewIntent, "appNewIntent", "(Landroid/content/Intent;)V")

      DECLARE_JNI_CLASS (JavaActivity, JUCE_PUSH_NOTIFICATIONS_ACTIVITY)
     #undef JNI_CLASS_MEMBERS

     static void JNICALL appNewIntent (JNIEnv*, jobject /*activity*/, jobject intentData)
     {
         juce_handleNotificationIntent (static_cast<void*> (intentData));
     }
 };

 JuceActivityNewIntentListener::JavaActivity_Class JuceActivityNewIntentListener::JavaActivity;
#endif

} // namespace juce
