/*
 * nUFRaw - new Unidentified Flying Raw converter for digital camera images
 *
 * nufraw.c - The standalone interface to nUFRaw.
 * Copyright 2004-2016 by Udi Fuchs
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "nufraw.h"
#include "uf_gtk.h"
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <glib/gi18n.h>

char *nufraw_binary;

void uf_exit(int exitCode)
{
#ifndef _WIN32
	gdk_threads_leave();
#endif
	exit(exitCode);
}

int main(int argc, char **argv)
{
	nufraw_data *uf;
	conf_data rc, cmd, conf;
	ui_conf uiconf;
	int status;
	GtkWidget *dummyWindow = NULL;
	int optInd;
	int plugin = 0;

#if !GLIB_CHECK_VERSION(2,31,0)
	g_thread_init(NULL);
#endif
#ifndef _WIN32
	gdk_threads_init();
	gdk_threads_enter();
#endif
	char *argFile = uf_win32_locale_to_utf8(argv[0]);
	nufraw_binary = g_path_get_basename(argFile);
	uf_init_locale(argFile);
	uf_win32_locale_free(argFile);
	char *gtkrcfile = g_build_filename(uf_get_home_dir(), UF_GTK_RC_FILE, NULL);
	gtk_rc_add_default_file(gtkrcfile);
	g_free(gtkrcfile);
	gtk_init(&argc, &argv);
	nufraw_icons_init();
#ifdef _WIN32
	dummyWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_icon_name(GTK_WINDOW(dummyWindow), "nufraw");
	nufraw_message(NUFRAW_SET_PARENT, (char *)dummyWindow);
#endif
	// load configuration
	conf_load(&rc, NULL);
	
	// load ui configuration
	ui_conf_load(&uiconf);
	
	// Half interpolation is an option only for the GIMP plug-in.
	// For the stand-alone tool it is disabled
	if (rc.interpolation == half_interpolation)  rc.interpolation = ahd_interpolation;

	g_strlcpy(rc.outputPath, "", UF_MAX_PATH);
	g_strlcpy(rc.inputFilename, "", UF_MAX_PATH);
	g_strlcpy(rc.outputFilename, "", UF_MAX_PATH);
	
	// Put the command-line options in cmd
	optInd = nufraw_process_args(&argc, &argv, &cmd, &rc);
	if (strlen(cmd.outputFilename) != 0) plugin = 3;

	if (cmd.silent) {
		nufraw_message(NUFRAW_ERROR,  _("The --silent option is only valid with 'nufraw-batch'"));
		optInd = -1;
	}
	if (cmd.embeddedImage) {
		nufraw_message(NUFRAW_ERROR, _("The --embedded-image option is only valid with 'nufraw-batch'"));
		optInd = -1;
	}
	if (optInd < 0) uf_exit(1);
	if (optInd == 0) uf_exit(0);

	// Create a dummyWindow for the GUI error messenger
	if (dummyWindow == NULL) {
		dummyWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_icon_name(GTK_WINDOW(dummyWindow), "nufraw");
		nufraw_message(NUFRAW_SET_PARENT, (char *)dummyWindow);
	}
	
	// load -C IDfile if any 
	conf_file_load(&conf, cmd.inputFilename);

	// if there's no args run file chooser
	if (optInd == argc) {
		nufraw_chooser(&rc, &conf, &cmd, &uiconf, NULL);
		ui_conf_save(&uiconf);  // TODO: should save the curves/profiles lists
		uf_exit(0);
	}
	
	// If there only one argument and it is a directory run file chooser on it
	argFile = uf_win32_locale_to_utf8(argv[optInd]);
	if (optInd == argc - 1 && g_file_test(argFile, G_FILE_TEST_IS_DIR)) {
		nufraw_chooser(&rc, &conf, &cmd, &uiconf, argFile);
		ui_conf_save(&uiconf);  // TODO: should save the curves/profiles lists
		uf_win32_locale_free(argFile);
		uf_exit(0);
	}
	uf_win32_locale_free(argFile);

	int exitCode = 0;
	for (; optInd < argc; optInd++) {
		argFile = uf_win32_locale_to_utf8(argv[optInd]);
		uf = nufraw_open(argFile);
		uf_win32_locale_free(argFile);
		if (uf == NULL) {
			exitCode = 1;
			nufraw_message(NUFRAW_REPORT, NULL);
			continue;
		}
		status = nufraw_config(uf, &rc, &conf, &cmd);
		if (status == NUFRAW_ERROR) {
			nufraw_close_darkframe(uf->conf);
			nufraw_close(uf);
			g_free(uf);
			uf_exit(1);
		}
		nufraw_ui(uf, &uiconf, plugin, NULL);
		g_free(uf);
	}
	if (dummyWindow != NULL) gtk_widget_destroy(dummyWindow);

	ui_conf_save(&uiconf);  // TODO: should save the curves/profiles lists
	ufobject_delete(cmd.ufobject);
	ufobject_delete(rc.ufobject);
	uf_exit(exitCode);
	return(exitCode);
}
