/*
 * nUFRaw - Unidentified Flying Raw converter for digital camera images
 *
 * nufraw_ui_options.c - GUI for UI options
 * Copyright 2016 by Matteo Lucarelli
 *
 * based on ufraw 0.23 Copyright 2004-2015 by Udi Fuchs
 * based on the GIMP plug-in by Pawel T. Jochym jochym at ifj edu pl,
 *
 * based on the GIMP plug-in by Dave Coffin
 * http://www.cybercom.net/~dcoffin/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
 
#include <glib/gi18n.h>

#include "nufraw.h"
#include "uf_gtk.h"
#include "nufraw_ui.h"
#include "curveeditor_widget.h"

// from nufraw_ui.c
extern ui_data gUiData;
extern conf_data *gConf;
extern ui_conf *gUiConf;
extern gboolean has_camera_curve();
extern void load_profile(GtkWidget *widget, long type);
extern void combo_update_simple(GtkWidget *combo, nUFRawPhase phase);
extern void start_blink();
extern void render_preview();
extern const ui_conf ui_conf_default;

static void copy_profiles_and_curves(conf_data *from, conf_data *to)
{
	int i,j;

	to->BaseCurveIndex=from->BaseCurveIndex;
	to->BaseCurveCount=from->BaseCurveCount;	
	for (i=0;i<to->BaseCurveCount;i++){
		to->BaseCurve[i]=from->BaseCurve[i];
	}
	
	to->curveIndex=from->curveIndex;
	to->curveCount=from->curveCount;	
	for (i=0;i<to->curveCount;i++){
		to->curve[i]=from->curve[i];
	}
	
	for (j=0;j<profile_types;j++){
		to->profileIndex[j]=from->profileIndex[j];
		to->profileCount[j]=from->profileCount[j];
		for (i=0;i<to->profileCount[j];i++){
			to->profile[j][i]=from->profile[j][i];
		}
		to->intent[j]=from->intent[j];
	}
}

static void container_remove(GtkWidget *widget, gpointer p)
{
	if (p){
		GtkContainer *container = GTK_CONTAINER(p);
		gtk_container_remove(container, widget);
	}
}

static void gimp_reset_clicked(GtkWidget *widget, GtkEntry *entry)
{
	(void)widget;
	gtk_entry_set_text(entry, ui_conf_default.remoteGimpCommand);
}

static void delete_from_list(GtkWidget *widget, gpointer user_data)
{
	GtkDialog *dialog = GTK_DIALOG(gtk_widget_get_ancestor(widget, GTK_TYPE_DIALOG));
	uf_long type = (uf_long)g_object_get_data(G_OBJECT(widget), "Type");
	uf_long index = (uf_long)user_data;

	if (type < profile_types) {
	
		uf_long toremove=index;
		gConf->profileCount[type]--;
		
		if (gConf->profileIndex[type] == index) gConf->profileIndex[type] = conf_default.profileIndex[type];
		else if (gConf->profileIndex[type] > index) gConf->profileIndex[type]--;
		
		for (; index < gConf->profileCount[type]; index++)
			gConf->profile[type][index] = gConf->profile[type][index + 1];
			
		gtk_combo_box_set_active(gUiData.ProfileCombo[type], gConf->profileIndex[type]);
		gtk_combo_box_remove_text(gUiData.ProfileCombo[type], toremove);
		                     
	} else if (type == profile_types + base_curve) {

		if (has_camera_curve()) gtk_combo_box_remove_text(gUiData.BaseCurveCombo, index);
		else gtk_combo_box_remove_text(gUiData.BaseCurveCombo, index - 2);

		gConf->BaseCurveCount--;

		if (gConf->BaseCurveIndex == index) gConf->BaseCurveIndex = conf_default.BaseCurveIndex;
		else if (gConf->BaseCurveIndex > index) gConf->BaseCurveIndex--;

		if (gConf->BaseCurveIndex == camera_curve && !has_camera_curve())  gConf->BaseCurveIndex = linear_curve;

		for (; index < gConf->BaseCurveCount; index++) gConf->BaseCurve[index] = gConf->BaseCurve[index + 1];

		if (gConf->BaseCurveIndex > camera_curve && !has_camera_curve()) gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex - 2);
		else gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex);

		curveeditor_widget_set_curve(gUiData.BaseCurveWidget, &gConf->BaseCurve[gConf->BaseCurveIndex]);
				             
	} else if (type == profile_types + luminosity_curve) {

		gtk_combo_box_remove_text(gUiData.CurveCombo, index);
		gConf->curveCount--;
		if (gConf->curveIndex == index) gConf->curveIndex = conf_default.curveIndex;
		else if (gConf->curveIndex > index)  gConf->curveIndex--;

		for (; index < gConf->curveCount; index++)
		gConf->curve[index] = gConf->curve[index + 1];
		gtk_combo_box_set_active(gUiData.CurveCombo, gConf->curveIndex);
		curveeditor_widget_set_curve(gUiData.CurveWidget, &gConf->curve[gConf->curveIndex]);
	}
	gtk_dialog_response(dialog, GTK_RESPONSE_APPLY);
}

static void load_monitor_profile(GtkWidget *widget, long type)
{
	GtkDialog *dialog = GTK_DIALOG(gtk_widget_get_ancestor(widget, GTK_TYPE_DIALOG));
	load_profile(widget,type);
	gtk_dialog_response(dialog, GTK_RESPONSE_APPLY);
}

void show_options_dialog()
{
	GtkWidget *optionsDialog;
	GtkTable *profileTable[profile_types];
	GtkWidget  *label, *button, *text, *box, *image;
	GtkNotebook *notebook;
	GtkScrolledWindow *scroll;
	GtkTable *baseCurveTable, *curveTable, *settingsTable;
	GtkTextBuffer *confBuffer, *buffer;
	GtkComboBox *combo;
	char txt[UF_MAX_NAME], *buf;
	uf_long i, j;
	long response;

	if (gUiData.FreezeDialog) return;
	
	conf_data conf_backup;
	copy_profiles_and_curves(gConf,&conf_backup);
	
	optionsDialog = gtk_dialog_new_with_buttons(_("nUFRaw options"),
	                                            GTK_WINDOW(gtk_widget_get_toplevel(gUiData.PreviewWidget)),
	                                            GTK_DIALOG_DESTROY_WITH_PARENT,
	                                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                            GTK_STOCK_SAVE, GTK_RESPONSE_OK, 
	                                            NULL);
	                                            
	gtk_window_resize(GTK_WINDOW(optionsDialog), 600, 400);
	nufraw_focus(optionsDialog, TRUE);
	gtk_dialog_set_default_response(GTK_DIALOG(optionsDialog), GTK_RESPONSE_CANCEL);

	notebook = GTK_NOTEBOOK(gtk_notebook_new());
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(optionsDialog)->vbox),GTK_WIDGET(notebook));

	// SETTINGS TAB ////////////////////////////////////////////////////////////

	label = gtk_label_new(_("Settings"));
	scroll = GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL, NULL));
	gtk_scrolled_window_set_policy(scroll,GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_notebook_append_page(notebook, GTK_WIDGET(scroll), label); 
	box = gtk_vbox_new(FALSE, 0);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), box);

	// settings
	  
	settingsTable = uf_expander_with_frame(box, _("Settings"), TRUE, NULL, FALSE );
		                                                   
	// display profile
	label = gtk_label_new(_("Display ICC profile"));
	gtk_misc_set_alignment(GTK_MISC(label),1,0.5);
	gtk_table_attach(settingsTable, label, 0, 1, 0, 1, GTK_SHRINK|GTK_FILL, 0, 0, 0);

	gUiData.ProfileCombo[display_profile] = GTK_COMBO_BOX(uf_combo_box_new_text()); 
	for (i = 0; i < gConf->profileCount[display_profile]; i++){
		if (i < conf_default.profileCount[display_profile]) {
			gtk_combo_box_append_text(gUiData.ProfileCombo[display_profile],_(gConf->profile[display_profile][i].name));
		} else {
			gtk_combo_box_append_text(gUiData.ProfileCombo[display_profile], gConf->profile[display_profile][i].name);
		}
	}
	uf_combo_box_set_data(gUiData.ProfileCombo[display_profile], &gConf->profileIndex[display_profile]);
	g_signal_connect_after(G_OBJECT(gUiData.ProfileCombo[display_profile]), "changed", G_CALLBACK(combo_update_simple), (gpointer)nufraw_develop_phase);
	gtk_table_attach(settingsTable, GTK_WIDGET(gUiData.ProfileCombo[display_profile]),  1, 8, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	button = uf_stock_icon_button(GTK_STOCK_OPEN, NULL, G_CALLBACK(load_monitor_profile), (gpointer)display_profile);
	gtk_table_attach(settingsTable, button, 8, 9, 0, 1, GTK_SHRINK, GTK_FILL, 0, 0);

	// display intent
	label = gtk_label_new(_("Display intent"));
	gtk_misc_set_alignment(GTK_MISC(label),1,0.5);
	gtk_table_attach(settingsTable, label, 1, 2, 1, 2, GTK_SHRINK|GTK_FILL, 0, 0, 0);
	combo = GTK_COMBO_BOX(uf_combo_box_new_text());
	gtk_combo_box_append_text(combo, _("Perceptual"));
	gtk_combo_box_append_text(combo, _("Relative colorimetric"));
	gtk_combo_box_append_text(combo, _("Saturation"));
	gtk_combo_box_append_text(combo, _("Absolute colorimetric"));
	gtk_combo_box_append_text(combo, _("Disable soft proofing"));
	uf_combo_box_set_data(GTK_COMBO_BOX(combo),(int *)&gConf->intent[display_profile]);
	g_signal_connect_after(G_OBJECT(combo), "changed", G_CALLBACK(combo_update_simple), (gpointer)nufraw_develop_phase);
	gtk_table_attach(settingsTable, GTK_WIDGET(combo), 2, 8, 1, 2, GTK_EXPAND|GTK_FILL, 0, 0, 0);             

	// Remote Gimp command
	label = gtk_label_new(_("Remote Gimp command"));
	gtk_misc_set_alignment(GTK_MISC(label),1,0.5);
	gtk_table_attach(settingsTable, label, 0, 1, 5, 6, GTK_SHRINK|GTK_FILL, 0, 0, 0);
	GtkEntry *gimpEntry = GTK_ENTRY(gtk_entry_new());
	gtk_entry_set_max_length(gimpEntry, UF_MAX_PATH);
	gtk_entry_set_text(gimpEntry, gUiConf->remoteGimpCommand);
	gtk_table_attach(settingsTable, GTK_WIDGET(gimpEntry), 1, 8, 5, 6, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	button = uf_reset_button( _("Reset command to default"), G_CALLBACK(gimp_reset_clicked), gimpEntry);
	gtk_table_attach(settingsTable, button, 8, 9, 5, 6, 0, 0, 0, 0);

	// remember path
	GtkWidget *outputButton = gtk_check_button_new_with_label(_("Remember output path"));
	gtk_table_attach(settingsTable, outputButton, 0, 9, 7, 8, GTK_FILL, 0, 0, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(outputButton),gUiConf->rememberOutputPath);

	// overwrite  
	GtkWidget *overwriteButton = gtk_check_button_new_with_label( _("Overwrite existing files without asking"));
	gtk_table_attach(settingsTable, overwriteButton, 0, 9, 8, 9, GTK_FILL, 0, 0, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(overwriteButton),gConf->overwrite);

	// blinkOverUnder toggle button
	GtkWidget *blinkButton = gtk_check_button_new_with_label( _("Blink Over/Underexposure Indicators"));
	gtk_table_attach(settingsTable, blinkButton, 0, 9, 9, 10, GTK_FILL, 0, 0, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(blinkButton),gUiConf->blinkOverUnder);

	// TODO: close preview after save
	
	// TODO: thumbnail size in chooser
	
	// TODO: 

	// lists
	profileTable[in_profile] =  uf_expander_with_frame(box, _("Input color profiles"), FALSE, NULL, FALSE);
	profileTable[out_profile] = uf_expander_with_frame(box, _("Output color profiles"), FALSE, NULL, FALSE);
	profileTable[display_profile] = uf_expander_with_frame(box, _("Display color profiles"), FALSE, NULL, FALSE);
	baseCurveTable = uf_expander_with_frame(box, _("Base Curves"), FALSE, NULL, FALSE);
	curveTable = uf_expander_with_frame(box, _("Luminosity Curves"), FALSE, NULL, FALSE);
    
	// CONFIGURATION TAB ///////////////////////////////////////////////////////                                 

	label = gtk_label_new(_("Configuration"));
	box = gtk_vbox_new(FALSE, 0);
	gtk_notebook_append_page(notebook, GTK_WIDGET(box), label);

	scroll =  GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL, NULL));
	gtk_scrolled_window_set_policy(scroll, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	text = gtk_text_view_new();
	gtk_container_add(GTK_CONTAINER(scroll), text);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(text), FALSE);
	confBuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
	gtk_box_pack_start(GTK_BOX(box),GTK_WIDGET(scroll),TRUE,TRUE,0);

	// LOG TAB /////////////////////////////////////////////////////////////////

	label = gtk_label_new(_("Log"));
	scroll =  GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL, NULL));
	gtk_notebook_append_page(notebook, GTK_WIDGET(scroll), label);
	gtk_scrolled_window_set_policy(scroll,GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	text = gtk_text_view_new();
	gtk_container_add(GTK_CONTAINER(scroll), text);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(text), FALSE);
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
	char *log = nufraw_message(NUFRAW_GET_LOG, NULL);
	if (log != NULL) {
		char *utf8_log = g_filename_display_name(log);
		gtk_text_buffer_set_text(buffer, utf8_log, -1);
		g_free(utf8_log);
	}

	// ABOUT TAB ///////////////////////////////////////////////////////////////

	label = gtk_label_new(_("About"));
	box = gtk_vbox_new(FALSE, 0);
	gtk_notebook_append_page(notebook, box, label);
	image = gtk_image_new_from_stock("nufraw", GTK_ICON_SIZE_DIALOG);
	gtk_box_pack_start(GTK_BOX(box), image, FALSE, FALSE, 20);
	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label),
		             "<span size='xx-large'><b>nUFRaw " VERSION "</b></span>\n");
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _(
		                 "The <b>n</b>ew <b>U</b>nidentified <b>F</b>lying <b>Raw</b> "
		                 "(<b>nUFRaw</b>) is a utility to\n"
		                 "read and manipulate raw images from digital cameras\n"
		                 "nUFRaw relies on <b>D</b>igital <b>C</b>amera "
		                 "<b>Raw</b> (<b>DCRaw</b>)\n"
		                 "for the actual encoding of the raw images.\n\n"
		                 "Author: Matteo Lucarelli (nUFRaw), Udi Fuchs (UFRaw)\n"
		                 "Homepage: http://nufraw.sourceforge.net/\n\n"));
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

	while (1) {

		for (j = 0; j < profile_types; j++) {

			gtk_container_foreach(GTK_CONTAINER(profileTable[j]),(GtkCallback)(container_remove), profileTable[j]);
			GtkTable *table = profileTable[j];
			for (i = conf_default.profileCount[j]; i < gConf->profileCount[j]; i++) {

				g_snprintf(txt, UF_MAX_NAME, "%s (%s)",gConf->profile[j][i].name,gConf->profile[j][i].productName);
				label = gtk_label_new(txt);
				gtk_table_attach_defaults(table, label, 0, 1, i, i + 1);
				button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
				g_object_set_data(G_OBJECT(button), "Type", (gpointer)j);
				g_signal_connect(G_OBJECT(button), "clicked",G_CALLBACK(delete_from_list), (gpointer)i);
				gtk_table_attach(table, button, 1, 2, i, i + 1, 0, 0, 0, 0);
			}
		}

		gtk_container_foreach(GTK_CONTAINER(baseCurveTable),(GtkCallback)(container_remove), baseCurveTable);
		for (i = camera_curve + 1; i < gConf->BaseCurveCount; i++) {

			label = gtk_label_new(gConf->BaseCurve[i].name);
			gtk_table_attach_defaults(baseCurveTable, label, 0, 1, i, i + 1);
			button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
			g_object_set_data(G_OBJECT(button), "Type",(gpointer)profile_types + base_curve);
			g_signal_connect(G_OBJECT(button), "clicked",G_CALLBACK(delete_from_list), (gpointer)i);
			gtk_table_attach(baseCurveTable, button, 1, 2, i, i + 1, 0, 0, 0, 0);
		}

		gtk_container_foreach(GTK_CONTAINER(curveTable),(GtkCallback)(container_remove), curveTable);
		for (i = linear_curve + 1; i < gConf->curveCount; i++) {

			label = gtk_label_new(gConf->curve[i].name);
			gtk_table_attach_defaults(curveTable, label, 0, 1, i, i + 1);
			button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
			g_object_set_data(G_OBJECT(button), "Type",(gpointer)profile_types + luminosity_curve);
			g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(delete_from_list), (gpointer)i);
			gtk_table_attach(curveTable, button, 1, 2, i, i + 1, 0, 0, 0, 0);
		}
		
		conf_save(gConf, NULL, &buf);
		gtk_text_buffer_set_text(confBuffer, buf, -1);
		g_free(buf);

		gtk_widget_show_all(optionsDialog);
		response = gtk_dialog_run(GTK_DIALOG(optionsDialog));

		if (response == GTK_RESPONSE_APPLY){
		
			// something has changed and we need to refresh
			continue;

		}else if (response == GTK_RESPONSE_OK) {

			// copy settings from window to gConf
			g_strlcpy(gUiConf->remoteGimpCommand, gtk_entry_get_text(gimpEntry), UF_MAX_PATH);
			gUiConf->rememberOutputPath = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(outputButton)); 
			gConf->overwrite = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(overwriteButton)); 
			gUiConf->blinkOverUnder  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(blinkButton));
			
			// save options only gConf values to .nnufraw-rc
			conf_data rc;
			conf_load(&rc, NULL);
			rc.overwrite=gConf->overwrite;
			copy_profiles_and_curves(gConf,&rc);
			conf_save(&rc,NULL,NULL);

		} else { 
		
			// restore previous profiles and curves lists
			copy_profiles_and_curves(&conf_backup,gConf);

		}
		nufraw_focus(optionsDialog, FALSE);
		gtk_widget_destroy(optionsDialog);
		start_blink();
		return;
	}
}

