/*
 *      callbacks.c
 *
 *      Copyright 2006 - 2012 Florian Sievers <florian.sievers@gmail.com>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; version 2 of the License.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gpg-crypter.h"
#include "callbacks.h"
#include "file-operations.h"
#include "edit-operations.h"
#include "crypto-operations.h"
#include "error-dialogs.h"

extern application_data *app_data;

/************/
/*** menu ***/
/************/

/* window operations */

G_MODULE_EXPORT gboolean
on_main_window_quit (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_main_quit();
	return FALSE;
}

/* file operations */

G_MODULE_EXPORT void
on_file_plaintext_new_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	app_data->plaintext_dialog_file =
		new_file (app_data->plaintext_dialog_file,
			  app_data->plaintext_textview,
			  app_data->plaintext_label,
			  _("Plaintext"));
}

G_MODULE_EXPORT void
on_file_plaintext_open_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	app_data->plaintext_dialog_file =
		open_file (app_data->plaintext_dialog_file,
			   app_data->plaintext_textview,
			   app_data->plaintext_label,
			   _("Plaintext"));
}

G_MODULE_EXPORT void
on_file_plaintext_save_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	app_data->plaintext_dialog_file =
		save_file (app_data->plaintext_dialog_file,
			   app_data->plaintext_textview,
			   app_data->plaintext_label,
			   _("Plaintext"), FALSE);
}

G_MODULE_EXPORT void
on_file_plaintext_save_as_activate (GtkMenuItem  *menuitem, gpointer user_data)
{
	app_data->plaintext_dialog_file =
		save_file (app_data->plaintext_dialog_file,
			   app_data->plaintext_textview,
			   app_data->plaintext_label,
			   _("Plaintext"), TRUE);
}

G_MODULE_EXPORT void
on_file_cipher_new_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	app_data->cipher_dialog_file =
		new_file (app_data->cipher_dialog_file,
			  app_data->cipher_textview,
			  app_data->cipher_label,
			  _("Cipher"));
}

G_MODULE_EXPORT void
on_file_cipher_open_activate (GtkMenuItem  *menuitem, gpointer user_data)
{
	app_data->cipher_dialog_file =
		open_file (app_data->cipher_dialog_file,
			   app_data->cipher_textview,
			   app_data->cipher_label,
			   _("Cipher"));
}

G_MODULE_EXPORT void
on_file_cipher_save_activate (GtkMenuItem  *menuitem, gpointer user_data)
{
	app_data->cipher_dialog_file =
		save_file (app_data->cipher_dialog_file,
			   app_data->cipher_textview,
			   app_data->cipher_label,
			   _("Cipher"), FALSE);
}

G_MODULE_EXPORT void
on_file_cipher_save_as_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	app_data->cipher_dialog_file =
		save_file (app_data->cipher_dialog_file,
			   app_data->cipher_textview,
			   app_data->cipher_label,
			   _("Cipher"), TRUE);
}

G_MODULE_EXPORT void
on_file_quit_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	on_main_window_quit (NULL, NULL, NULL);
}

/* edit operations */

G_MODULE_EXPORT void
on_edit_cut_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	edit_cut_clipboard (NULL);
}

G_MODULE_EXPORT void
on_edit_copy_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	edit_copy_clipboard (NULL);
}

G_MODULE_EXPORT void
on_edit_paste_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	edit_paste_clipboard (NULL);
}

G_MODULE_EXPORT void
on_edit_delete_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	edit_delete_clipboard (NULL);
}

G_MODULE_EXPORT void
on_edit_select_all_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	edit_select_all (NULL);
}

/* help operations */

G_MODULE_EXPORT void
on_help_about_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	gtk_dialog_run (GTK_DIALOG (app_data->about_dialog));
	gtk_widget_hide (app_data->about_dialog);
}

/************/
/*** keys ***/
/************/

/* combo_box operations */

G_MODULE_EXPORT void
on_key_combo_box_changed (GtkComboBox *combo_box, gpointer user_data)
{
	gpg_select_key (gtk_combo_box_get_active (combo_box));
}

G_MODULE_EXPORT void
on_passphrase_callback_entry_activate (GtkEntry *entry, gpointer user_data)
{
	/* response to enter key in the passphrase callback entry */
	gtk_dialog_response (GTK_DIALOG (app_data->passphrase_callback_dialog), GTK_RESPONSE_OK);
}

/***********************/
/*** textcrypter tab ***/
/***********************/

/* button operations */

G_MODULE_EXPORT void
on_bt_clear_plaintext_clicked (GtkButton *button, gpointer user_data)
{
	app_data->plaintext_dialog_file =
		new_file (app_data->plaintext_dialog_file,
			  app_data->plaintext_textview,
			  app_data->plaintext_label,
			  _("Plaintext"));
}

G_MODULE_EXPORT void
on_bt_select_all_plaintext_clicked (GtkButton *button, gpointer user_data)
{
	edit_select_all (app_data->plaintext_textview);
}

G_MODULE_EXPORT void
on_bt_plaintext_copy_to_clipboard_clicked (GtkButton *button, gpointer user_data)
{
	edit_copy_clipboard (app_data->plaintext_textview);
}

G_MODULE_EXPORT void
on_bt_plaintext_paste_from_clipboard_clicked (GtkButton *button, gpointer user_data)
{
	edit_paste_clipboard (app_data->plaintext_textview);
}

G_MODULE_EXPORT void
on_bt_clear_cipher_clicked (GtkButton *button, gpointer user_data)
{
	app_data->cipher_dialog_file =
		new_file (app_data->cipher_dialog_file,
			  app_data->cipher_textview,
			  app_data->cipher_label,
			  _("Cipher"));
}

G_MODULE_EXPORT void
on_bt_select_all_cipher_clicked (GtkButton *button, gpointer user_data)
{
	edit_select_all (app_data->cipher_textview);
}

G_MODULE_EXPORT void
on_bt_cipher_copy_to_clipboard_clicked (GtkButton *button, gpointer user_data)
{
	edit_copy_clipboard (app_data->cipher_textview);
}

G_MODULE_EXPORT void
on_bt_cipher_paste_from_clipboard_clicked (GtkButton *button, gpointer user_data)
{
	edit_paste_clipboard (app_data->cipher_textview);
}

G_MODULE_EXPORT void
on_bt_encrypt_plaintext_clicked (GtkButton *button, gpointer user_data)
{
	encrypt_buffer ();
}

G_MODULE_EXPORT void
on_bt_decrypt_cipher_clicked (GtkButton *button, gpointer user_data)
{
	decrypt_buffer ();
}

/***********************/
/*** filecrypter tab ***/
/***********************/

/* entry operations */

G_MODULE_EXPORT void
on_plaintext_file_entry_focus_in (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gchar *plaintext_file, *cipher_file;

	plaintext_file = g_strdup (gtk_entry_get_text (GTK_ENTRY (app_data->plaintext_file_entry)));
	cipher_file = g_strdup (gtk_entry_get_text (GTK_ENTRY (app_data->cipher_file_entry)));

	g_strstrip (plaintext_file);
	g_strstrip (cipher_file);

	if (strlen (plaintext_file) == 0 &&
	    strlen (cipher_file) > 0 &&
	    g_str_has_suffix (cipher_file, ".gpg") == TRUE) {
		gchar *new_plaintext_file = g_strndup (cipher_file, strlen (cipher_file) - 4);
		gtk_entry_set_text (GTK_ENTRY (app_data->plaintext_file_entry), new_plaintext_file);
		gtk_editable_select_region (GTK_EDITABLE (app_data->plaintext_file_entry), 0, -1);
		g_free (new_plaintext_file);
	}

	g_free (plaintext_file);
	g_free (cipher_file);
}

G_MODULE_EXPORT void
on_cipher_file_entry_focus_in (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gchar *plaintext_file, *cipher_file;

	plaintext_file = g_strdup (gtk_entry_get_text (GTK_ENTRY (app_data->plaintext_file_entry)));
	cipher_file = g_strdup (gtk_entry_get_text (GTK_ENTRY (app_data->cipher_file_entry)));

	g_strstrip (plaintext_file);
	g_strstrip (cipher_file);

	if (strlen (plaintext_file) > 0 &&
	    strlen (cipher_file) == 0 &&
	    g_str_has_suffix (plaintext_file, ".gpg") == FALSE) {
		gchar *new_cipher_file = g_strconcat (plaintext_file, ".gpg", NULL);
		gtk_entry_set_text (GTK_ENTRY (app_data->cipher_file_entry), new_cipher_file);
		gtk_editable_select_region (GTK_EDITABLE (app_data->cipher_file_entry), 0, -1);
		g_free (new_cipher_file);
	}

	g_free (plaintext_file);
	g_free (cipher_file);
}

/* button operations */

G_MODULE_EXPORT void
on_bt_plaintext_file_select_clicked (GtkButton *button, gpointer user_data)
{
	select_file (app_data->plaintext_file_entry, _("Select plaintext file"));
}

G_MODULE_EXPORT void
on_bt_cipher_file_select_clicked (GtkButton *button, gpointer user_data)
{
	select_file (app_data->cipher_file_entry, _("Select cipher file"));
}

G_MODULE_EXPORT void
on_bt_encrypt_plaintext_file_clicked (GtkButton *button, gpointer user_data)
{
	encrypt_file (
		gtk_entry_get_text (GTK_ENTRY (app_data->plaintext_file_entry)),
		gtk_entry_get_text (GTK_ENTRY (app_data->cipher_file_entry)));
}

G_MODULE_EXPORT void
on_bt_decrypt_cipher_file_clicked (GtkButton *button, gpointer user_data)
{
	decrypt_file (
		gtk_entry_get_text (GTK_ENTRY (app_data->cipher_file_entry)),
		gtk_entry_get_text (GTK_ENTRY (app_data->plaintext_file_entry)));
}

/* checkbutton operations */

G_MODULE_EXPORT void
on_chbt_cipher_file_ascii_armor_clicked (GtkToggleButton *button, gpointer user_data)
{
	app_data->ascii_mode = gtk_toggle_button_get_active (button) ? 1 : 0;
}
