/*
 *      crypto-operations.c
 *
 *      Copyright 2006 - 2012 Florian Sievers <florian.sievers@gmail.com>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; version 2 of the License.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gpg-crypter.h"
#include "crypto-operations.h"
#include "error-dialogs.h"

#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

extern application_data *app_data;

static gpgme_error_t gpg_passphrase_callback (void *hook, const char *uid_hint, const char *passphrase_info, int prev_was_bad, int fd);
static void gpg_get_keys (void);
static gchar* to_utf8 (const gchar *string);

/* initialize gpgme */
gboolean gpg_init (void)
{
	gpgme_error_t err = 0;

	app_data->key_array        = g_ptr_array_new ();
	app_data->key_string_array = g_ptr_array_new ();

	g_ptr_array_set_free_func (app_data->key_array, g_free);
	g_ptr_array_set_free_func (app_data->key_string_array, g_free);

	setlocale (LC_ALL, "");
	gpgme_check_version (NULL);
	gpgme_set_locale (NULL, LC_CTYPE, setlocale (LC_CTYPE, NULL));
#ifdef HAVE_LC_MESSAGES
	gpgme_set_locale (NULL, LC_MESSAGES, setlocale (LC_MESSAGES, NULL));
#endif

	err = gpgme_engine_check_version (GPGME_PROTOCOL_OpenPGP);
	if (err) {
		error_dialog (_("OpenPGP protocol (GnuPG) not found.\n%s"), gpgme_strerror (err));
		goto gpg_init_end;
	}

	err = gpgme_new (&(app_data)->ctx);
	if (err) {
		error_dialog (_("Cannot create gpgme context.\n%s"), gpgme_strerror (err));
		goto gpg_init_end;
	}

	gpg_get_keys ();
	if (app_data->num_keys == 0) {
		error_dialog (_("No GPG key found.\nPlease, create a key with: gpg --gen-key"));
		goto gpg_init_end;
	}

	gpgme_set_passphrase_cb (app_data->ctx, gpg_passphrase_callback, app_data);

gpg_init_end:
	if ((err != 0) || (app_data->num_keys == 0)) {
		gpg_exit ();
		return FALSE;
	} else
		return TRUE;
}

/* release all resources that were allocated for gpgme and app_data */
void gpg_exit (void)
{
	if (app_data->ctx) {
		gpgme_set_passphrase_cb (app_data->ctx, NULL, NULL);
		gpgme_release (app_data->ctx);
	}

	g_ptr_array_free (app_data->key_array, TRUE);
	g_ptr_array_free (app_data->key_string_array, TRUE);
}

/* select app_data->key_array[] where the index is the selected combo_box entry */
void gpg_select_key (int num_key)
{
	gpgme_error_t err;

	g_return_if_fail (num_key >= 0);
	g_return_if_fail (num_key < app_data->num_keys);

	err = gpgme_op_keylist_start (app_data->ctx, g_ptr_array_index (app_data->key_array, num_key), 0);
	if (err) {
		error_dialog (_("Cannot start listing of keys.\n%s"), gpgme_strerror (err));
		return;
	}

	err = gpgme_op_keylist_next (app_data->ctx, app_data->key);
	if (err) {
		error_dialog (_("Cannot return next key in the list.\n%s"), gpgme_strerror (err));
		return;
	}
}

/* encrypt the plaintext file */
void encrypt_file (const gchar *plaintext_file, const gchar *cipher_file)
{
	int plaintext_fd = -1, cipher_fd = -1;
	gpgme_data_t dh_plain = 0, dh_cipher = 0;
	gpgme_error_t err;
	struct stat filestats;

	g_return_if_fail (plaintext_file != NULL);
	g_return_if_fail (cipher_file != NULL);

	/* set armor */
	gpgme_set_armor (app_data->ctx, app_data->ascii_mode);

	/* consistency checks on filenames */
	if (strlen (plaintext_file) == 0 ||
	    g_file_test (plaintext_file, G_FILE_TEST_IS_REGULAR) == FALSE) {
		error_dialog (_("No valid plaintext file selected."));
		goto encrypt_file_end;
	}

	if (strlen (cipher_file) == 0) {
		error_dialog (_("No valid cipher file selected."));
		goto encrypt_file_end;
	}

	if (strcmp (plaintext_file, cipher_file) == 0) {
		error_dialog (_("Plaintext and cipher files cannot be the same."));
		goto encrypt_file_end;
	}

	/* get plaintext file attributes */
	if (stat (plaintext_file, &filestats) == -1) {
		error_dialog (_("Cannot get attributes of plaintext file: %s\n%s"),
			plaintext_file, g_strerror (errno));

		goto encrypt_file_end;
	}

	/* open plaintext file for reading */
	plaintext_fd = open (plaintext_file, O_RDONLY);
	if (plaintext_fd == -1) {
		error_dialog (_("Cannot open plaintext file for reading: %s\n%s"),
			plaintext_file, g_strerror (errno));

		goto encrypt_file_end;
	}

	/* try open cipher file for writing, error if exists */
	cipher_fd = open (cipher_file, O_WRONLY | O_CREAT | O_EXCL, 600);
	if (cipher_fd == -1) {
		if (errno != EEXIST) {
			error_dialog (_("Cannot open cipher file for writing: %s\n%s"),
				cipher_file, g_strerror (errno));

			goto encrypt_file_end;
		}

		/* ask to overwrite file if already exists */
		if (warn_dialog (_("Do you want to overwrite the cipher file?\n%s"), cipher_file) == FALSE) {
			/* don't want to overwrite */
			goto encrypt_file_end;
		}

		/* try open cipher file for writing, overwrite if exists */
		cipher_fd = open (cipher_file, O_WRONLY | O_CREAT | O_TRUNC, 600);
		if (cipher_fd == -1) {
			error_dialog (_("Cannot open cipher file for writing: %s\n%s"),
				cipher_file, g_strerror (errno));

			goto encrypt_file_end;
		}
	}

	/* create data buffers */
	err = gpgme_data_new_from_fd (&dh_plain, plaintext_fd);
	if (err) {
		error_dialog (_("Cannot create data buffer for plaintext file: %s\n%s"),
			plaintext_file, gpgme_strerror (err));

		goto encrypt_file_end;
	}

	err = gpgme_data_new_from_fd (&dh_cipher, cipher_fd);
	if (err) {
		error_dialog (_("Cannot create data buffer for cipher file: %s\n%s"),
			cipher_file, gpgme_strerror (err));

		goto encrypt_file_end;
	}

	/* encrypt plaintext */
	err = gpgme_op_encrypt (app_data->ctx, app_data->key, GPGME_ENCRYPT_ALWAYS_TRUST, dh_plain, dh_cipher);
	if (err) {
		error_dialog (_("Cannot encrypt plaintext file: %s\n%s"),
			plaintext_file, gpgme_strerror (err));

		goto encrypt_file_end;
	}

encrypt_file_end:
	/* release data buffers */
	if (dh_plain)
		gpgme_data_release (dh_plain);

	if (dh_cipher)
		gpgme_data_release (dh_cipher);

	if (plaintext_fd != -1) {
		/* close plaintext file */
		if (close (plaintext_fd) == -1) {
			error_dialog (_("Error when closing plaintext file: %s\n%s"),
				plaintext_file, g_strerror (errno));
		}
	}

	if (cipher_fd != -1) {
		/* close cipher file */
		if (close (cipher_fd) == -1) {
			error_dialog (_("Error when closing cipher file: %s\n%s"),
				cipher_file, g_strerror (errno));
		}

		/* set cipher file attributes to the ones of plaintext file */
		if (chmod (cipher_file, filestats.st_mode) == -1) {
			error_dialog (_("Cannot set attributes of cipher file: %s\n%s"),
				cipher_file, g_strerror (errno));
		}
	}
}

/* decrypt the cipher file */
void decrypt_file (const gchar *cipher_file, const gchar *plaintext_file)
{
	int plaintext_fd = -1, cipher_fd = -1;
	gpgme_data_t dh_plain = 0, dh_cipher = 0;
	gpgme_error_t err;
	struct stat filestats;

	g_return_if_fail (cipher_file != NULL);
	g_return_if_fail (plaintext_file != NULL);

	/* consistency checks on filenames */
	if (strlen (cipher_file) == 0 ||
	    g_file_test (cipher_file, G_FILE_TEST_IS_REGULAR) == FALSE) {
		error_dialog (_("No valid cipher file selected."));
		goto decrypt_file_end;
	}

	if (strlen (plaintext_file) == 0) {
		error_dialog (_("No valid plaintext file selected."));
		goto decrypt_file_end;
	}

	if (strcmp (cipher_file, plaintext_file) == 0) {
		error_dialog (_("Plaintext and cipher files cannot be the same."));
		goto decrypt_file_end;
	}

	/* get cipher file attributes */
	if (stat (cipher_file, &filestats) == -1) {
		error_dialog (_("Cannot get attributes of cipher file: %s\n%s"),
			cipher_file, g_strerror (errno));

		goto decrypt_file_end;
	}

	/* open cipher file for reading */
	cipher_fd = open (cipher_file, O_RDONLY);
	if (cipher_fd == -1) {
		error_dialog (_("Cannot open cipher file for reading: %s\n%s"),
			cipher_file, g_strerror (errno));

		goto decrypt_file_end;
	}

	/* try open plaintext file for writing, error if exists */
	plaintext_fd = open (plaintext_file, O_WRONLY | O_CREAT | O_EXCL, 0600);
	if (plaintext_fd == -1) {
		if (errno != EEXIST) {
			error_dialog (_("Cannot open plaintext file for writing: %s\n%s"),
				plaintext_file, g_strerror (errno));

			goto decrypt_file_end;
		}

		/* ask to overwrite file if already exists */
		if (warn_dialog (_("Do you want to overwrite the plaintext file?\n%s"), plaintext_file) == FALSE) {
			/* don't want to overwrite */
			goto decrypt_file_end;
		}

		/* try open plaintext file for writing, overwrite if exists */
		plaintext_fd = open (plaintext_file, O_WRONLY | O_CREAT | O_TRUNC, 0600);
		if (plaintext_fd == -1) {
			error_dialog (_("Cannot open plaintext file for writing: %s\n%s"),
				plaintext_file, g_strerror (errno));

			goto decrypt_file_end;
		}
	}

	/* create data buffers */
	err = gpgme_data_new_from_fd (&dh_cipher, cipher_fd);
	if (err) {
		error_dialog (_("Cannot create data buffer for cipher file: %s\n%s"),
			cipher_file, gpgme_strerror (err));

		goto decrypt_file_end;
	}

	err = gpgme_data_new_from_fd (&dh_plain, plaintext_fd);
	if (err) {
		error_dialog (_("Cannot create data buffer for plaintext file: %s\n%s"),
			plaintext_file, gpgme_strerror (err));

		goto decrypt_file_end;
	}

	/* decrypt cipher */
	err = gpgme_op_decrypt (app_data->ctx, dh_cipher, dh_plain);
	if (err) {
		error_dialog (_("Cannot decrypt cipher file: %s\n%s"),
			cipher_file, gpgme_strerror (err));

		goto decrypt_file_end;
	}

decrypt_file_end:
	/* release data buffers */
	if (dh_cipher)
		gpgme_data_release (dh_cipher);

	if (dh_plain)
		gpgme_data_release (dh_plain);

	if (cipher_fd != -1) {
		/* close cipher file */
		if (close (cipher_fd) == -1) {
			error_dialog (_("Error when closing cipher file: %s\n%s"),
				cipher_file, g_strerror (errno));
		}
	}

	if (plaintext_fd != -1) {
		/* close plaintext file */
		if (close (plaintext_fd) == -1) {
			error_dialog (_("Error when closing plaintext file: %s\n%s"),
				plaintext_file, g_strerror (errno));
		}

		/* set plaintext file attributes to the ones of cipher file */
		if (chmod (plaintext_file, filestats.st_mode) == -1) {
			error_dialog (_("Cannot set attributes of plaintext file: %s\n%s"),
				plaintext_file, g_strerror (errno));
		}
	}
}

/* encrypt the plaintext buffer */
void encrypt_buffer (void)
{
	GtkTextBuffer *plaintext_buffer, *cipher_buffer;
	GtkTextIter iter_start, iter_end;
	gchar *plaintext;
	gpgme_data_t dh_plain = 0, dh_cipher = 0;
	gpgme_error_t err;

	size_t ciphertext_length;
	gchar *ciphertext, *ciphertext_padded;

	/* force ASCII-armor */
	gpgme_set_armor (app_data->ctx, 1);

	/* get plaintext */
	plaintext_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (app_data->plaintext_textview));
	cipher_buffer    = gtk_text_view_get_buffer (GTK_TEXT_VIEW (app_data->cipher_textview));

	gtk_text_buffer_get_start_iter (plaintext_buffer, &iter_start);
	gtk_text_buffer_get_end_iter   (plaintext_buffer, &iter_end);

	plaintext = gtk_text_buffer_get_text (plaintext_buffer, &iter_start, &iter_end, FALSE);

	/* create data buffers */
	err = gpgme_data_new_from_mem (&dh_plain, plaintext, strlen (plaintext), 0);
	if (err) {
		error_dialog (_("Cannot create data buffer for plaintext.\n%s"), gpgme_strerror (err));
		goto encrypt_buffer_end;
	}

	err = gpgme_data_new (&dh_cipher);
	if (err) {
		error_dialog (_("Cannot create data buffer for cipher.\n%s"), gpgme_strerror (err));
		goto encrypt_buffer_end;
	}

	/* encrypt plaintext */
	err = gpgme_op_encrypt (app_data->ctx, app_data->key, GPGME_ENCRYPT_ALWAYS_TRUST, dh_plain, dh_cipher);
	if (err) {
		error_dialog (_("Cannot encrypt plaintext.\n%s"), gpgme_strerror (err));
		goto encrypt_buffer_end;
	}

	/* output ascii armored cipher */
	ciphertext = gpgme_data_release_and_get_mem (dh_cipher, &ciphertext_length);
	dh_cipher = 0; /* invalidated by above call */

	if (ciphertext == NULL) {
		error_dialog (_("gpgme error when getting buffer memory.\n%s"), gpgme_strerror (err));
		goto encrypt_buffer_end;
	}

	ciphertext_padded = g_strndup (ciphertext, ciphertext_length);
	gpgme_free (ciphertext);

	gtk_text_buffer_set_text (cipher_buffer, ciphertext_padded, -1);
	g_free (ciphertext_padded);

encrypt_buffer_end:
	/* release data buffers */
	g_free (plaintext);

	if (dh_plain)
		gpgme_data_release (dh_plain);

	if (dh_cipher)
		gpgme_data_release (dh_cipher);
}

/* decrypt the cipher buffer */
void decrypt_buffer (void)
{
	GtkTextBuffer *plaintext_buffer, *cipher_buffer;
	GtkTextIter iter_start, iter_end;
	gchar *ciphertext;
	gpgme_data_t dh_plain = 0, dh_cipher = 0;
	gpgme_error_t err;

	size_t plaintext_length = 0;
	gchar *plaintext, *plaintext_padded, *plaintext_utf8;

	/* get cipher text */
	plaintext_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (app_data->plaintext_textview));
	cipher_buffer    = gtk_text_view_get_buffer (GTK_TEXT_VIEW (app_data->cipher_textview));

	gtk_text_buffer_get_start_iter (cipher_buffer, &iter_start);
	gtk_text_buffer_get_end_iter   (cipher_buffer, &iter_end);

	ciphertext = gtk_text_buffer_get_text (cipher_buffer, &iter_start, &iter_end, FALSE);

	/* create data buffers */
	err = gpgme_data_new_from_mem (&dh_cipher, ciphertext, strlen (ciphertext), 0);
	if (err) {
		error_dialog (_("Cannot create data buffer for cipher.\n%s"), gpgme_strerror (err));
		goto decrypt_buffer_end;
	}

	err = gpgme_data_new (&dh_plain);
	if (err) {
		error_dialog (_("Cannot create data buffer for plaintext.\n%s"), gpgme_strerror (err));
		goto decrypt_buffer_end;
	}

	/* decrypt cipher text */
	err = gpgme_op_decrypt (app_data->ctx, dh_cipher, dh_plain);
	if (err) {
		error_dialog (_("Cannot decrypt cipher.\n%s"), gpgme_strerror (err));
		goto decrypt_buffer_end;
	}

	/* output ascii armored cipher */
	plaintext = gpgme_data_release_and_get_mem (dh_plain, &plaintext_length);
	dh_plain = 0; /* invalidated by above call */

	if (plaintext == NULL) {
		error_dialog (_("gpgme error when getting buffer memory.\n%s"), gpgme_strerror (err));
		goto decrypt_buffer_end;
	}

	plaintext_padded = g_strndup (plaintext, plaintext_length);
	gpgme_free (plaintext);

	if (g_utf8_validate (plaintext_padded, -1, NULL) == FALSE)
		error_dialog (_("Invalid UTF-8 string.\nOutput might be truncated."));

	plaintext_utf8 = to_utf8 (plaintext_padded);
	g_free (plaintext_padded);

	gtk_text_buffer_set_text (plaintext_buffer, plaintext_utf8, -1);
	g_free (plaintext_utf8);

decrypt_buffer_end:
	/* release data buffers */
	g_free (ciphertext);

	if (dh_cipher)
		gpgme_data_release (dh_cipher);

	if (dh_plain)
		gpgme_data_release (dh_plain);
}

/* gpgme passphrase callback dialog */
static gpgme_error_t gpg_passphrase_callback (void *hook, const char *uid_hint, const char *passphrase_info, int prev_was_bad, int fd)
{
	const char nl = '\n';
	const gchar *passphrase;

	if (gtk_dialog_run (GTK_DIALOG (app_data->passphrase_callback_dialog)) == GTK_RESPONSE_OK) {
		passphrase = gtk_entry_get_text (GTK_ENTRY (app_data->passphrase_callback_entry));

		write (fd, passphrase, strlen (passphrase));
		write (fd, &nl, 1);

		gtk_entry_set_text (GTK_ENTRY (app_data->passphrase_callback_entry), "");
		gtk_widget_hide (app_data->passphrase_callback_dialog);

		return GPG_ERR_NO_ERROR;
	} else {
		write (fd, &nl, 1);

		gtk_entry_set_text (GTK_ENTRY (app_data->passphrase_callback_entry), "");
		gtk_widget_hide (app_data->passphrase_callback_dialog);

		return GPG_ERR_CANCELED;
	}
}

/* get all gpg-keys and insert them into app_data->key_array[] */
static void gpg_get_keys (void)
{
	gpgme_key_t new_key;
	gpgme_error_t err;
	gchar *utf8_uid, *utf8_name, *utf8_comment, *utf8_email;

	err = gpgme_op_keylist_start (app_data->ctx, NULL, 0);
	if (err) {
		error_dialog (_("Cannot start listing of keys.\n%s"), gpgme_strerror (err));
		return;
	}

	/* fetch first key */
	err = gpgme_op_keylist_next (app_data->ctx, &new_key);
	while (gpgme_err_code (err) == GPG_ERR_NO_ERROR) {
		app_data->num_keys++;

		/* convert key values */
		utf8_uid     = to_utf8 (new_key->uids->uid);
		utf8_name    = to_utf8 (new_key->uids->name);
		utf8_comment = to_utf8 (new_key->uids->comment);
		utf8_email   = to_utf8 (new_key->uids->email);

		/* add key values */
		g_ptr_array_add (app_data->key_array, 
					g_strconcat (
						utf8_uid,
						NULL));

		g_ptr_array_add (app_data->key_string_array,
					g_strconcat (
						utf8_name, " (",
						utf8_comment, ")\n    ",
						utf8_email, NULL));

		/* free key values */
		g_free (utf8_uid);
		g_free (utf8_name);
		g_free (utf8_comment);
		g_free (utf8_email);

		/* release key */
		gpgme_key_release (new_key);

		/* fetch next key */
		err = gpgme_op_keylist_next (app_data->ctx, &new_key);
	}

	if (gpgme_err_code (err) != GPG_ERR_EOF)
		error_dialog (_("Cannot return next key in the list.\n%s"), gpgme_strerror (err));
}

static gchar* to_utf8 (const gchar *string)
{
	g_return_val_if_fail (string != NULL, NULL);

	gchar *converted_string = g_locale_to_utf8 (string, -1, NULL, NULL, NULL);
	return converted_string == NULL ? g_strdup ("") : converted_string;		
}
