/*
 *      file-operations.c
 *
 *      Copyright 2007 - 2012 Florian Sievers <florian.sievers@gmail.com>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; version 2 of the License.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gpg-crypter.h"
#include "file-operations.h"
#include "error-dialogs.h"

extern application_data *app_data;

static void toggle_hidden_file (GtkToggleButton *togglebutton, gpointer user_data);

gchar* new_file (gchar *filename, GtkWidget *textview, GtkWidget *label, gchar *label_prefix)
{
	gchar *markup;

	g_return_val_if_fail (GTK_IS_TEXT_VIEW (textview) != FALSE, filename);
	g_return_val_if_fail (GTK_IS_LABEL (label) != FALSE, filename);
	g_return_val_if_fail (label_prefix != NULL, filename);

	gtk_text_buffer_set_text (gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview)), "", -1);

	markup = g_markup_printf_escaped ("<b>%s</b>", label_prefix);
	gtk_label_set_markup (GTK_LABEL (label), markup);
	g_free (markup);

	g_free (filename);
	filename = NULL;

	return filename;
}

gchar* open_file (gchar *filename, GtkWidget *textview, GtkWidget *label, gchar *label_prefix)
{
	GtkWidget *file_chooser, *toggle_hidden;
	gchar *text_from_file, *markup;
	GError *error = NULL;

	g_return_val_if_fail (GTK_IS_TEXT_VIEW (textview) != FALSE, filename);
	g_return_val_if_fail (GTK_IS_LABEL (label) != FALSE, filename);
	g_return_val_if_fail (label_prefix != NULL, filename);

	file_chooser = gtk_file_chooser_dialog_new (
				_("Open file"),
				GTK_WINDOW (app_data != NULL ? app_data->main_window : NULL),
				GTK_FILE_CHOOSER_ACTION_OPEN,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				NULL);

	toggle_hidden = gtk_check_button_new_with_label (_("Show hidden files"));
	g_signal_connect (G_OBJECT (toggle_hidden), "toggled", G_CALLBACK (toggle_hidden_file), file_chooser);
	gtk_widget_show (toggle_hidden);

	gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (file_chooser), toggle_hidden);
	gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (file_chooser), FALSE);
	gtk_dialog_set_default_response (GTK_DIALOG (file_chooser), GTK_RESPONSE_ACCEPT);

	if (gtk_dialog_run (GTK_DIALOG (file_chooser)) == GTK_RESPONSE_ACCEPT) {
		g_free (filename);
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_chooser));

		if (g_file_get_contents (filename, &text_from_file, NULL, &error) == TRUE) {
			gtk_text_buffer_set_text (gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview)), text_from_file, -1);
			g_free (text_from_file);

			markup = g_markup_printf_escaped ("<b>%s</b> - %s", label_prefix, filename);
			gtk_label_set_markup (GTK_LABEL (label), markup);
			g_free (markup);
		} else {
			error_dialog (_("Cannot open file: %s\n%s"), filename, error->message);
			g_error_free (error); error = NULL;

			g_free (filename);
			filename = NULL;

			gtk_text_buffer_set_text (gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview)), "", -1);

			markup = g_markup_printf_escaped ("<b>%s</b>", label_prefix);
			gtk_label_set_markup (GTK_LABEL (label), markup);
			g_free (markup);
		}
	}

	gtk_widget_destroy (file_chooser);
	return filename;
}

gchar* save_file (gchar *filename, GtkWidget *textview, GtkWidget *label, gchar *label_prefix, gboolean save_as)
{
	GtkWidget *file_chooser, *toggle_hidden;
	GtkTextBuffer *text_buffer;
	GtkTextIter iter_start, iter_end;
	gchar *text_from_buffer, *markup;
	GError *error = NULL;

	g_return_val_if_fail (GTK_IS_TEXT_VIEW (textview) != FALSE, filename);
	g_return_val_if_fail (GTK_IS_LABEL (label) != FALSE, filename);
	g_return_val_if_fail (label_prefix != NULL, filename);

	if (filename == NULL || save_as == TRUE) {
		file_chooser = gtk_file_chooser_dialog_new (
					_("Save file"),
					GTK_WINDOW (app_data != NULL ? app_data->main_window : NULL),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);

		toggle_hidden = gtk_check_button_new_with_label (_("Show hidden files"));
		g_signal_connect (G_OBJECT (toggle_hidden), "toggled", G_CALLBACK (toggle_hidden_file), file_chooser);
		gtk_widget_show (toggle_hidden);

		gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (file_chooser), toggle_hidden);
		gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (file_chooser), TRUE);
		gtk_dialog_set_default_response (GTK_DIALOG (file_chooser), GTK_RESPONSE_ACCEPT);

		if (filename != NULL)
			gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (file_chooser), filename);

		if (gtk_dialog_run (GTK_DIALOG (file_chooser)) == GTK_RESPONSE_ACCEPT) {
			g_free (filename);
			filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_chooser));

			gtk_widget_destroy (file_chooser);
		} else {
			gtk_widget_destroy (file_chooser);

			return filename;
		}
	}

	text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

	gtk_text_buffer_get_start_iter (GTK_TEXT_BUFFER (text_buffer), &iter_start);
	gtk_text_buffer_get_end_iter   (GTK_TEXT_BUFFER (text_buffer), &iter_end);

	text_from_buffer = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (text_buffer), &iter_start, &iter_end, FALSE);

	if (g_file_set_contents (filename, text_from_buffer, -1, &error) == TRUE) {
		markup = g_markup_printf_escaped ("<b>%s</b> - %s", label_prefix, filename);
		gtk_label_set_markup (GTK_LABEL (label), markup);
		g_free (markup);
	} else {
		error_dialog (_("Cannot save file: %s\n%s"), filename, error->message);
		g_error_free (error); error = NULL;

		g_free (filename);
		filename = NULL;

		markup = g_markup_printf_escaped ("<b>%s</b>", label_prefix);
		gtk_label_set_markup (GTK_LABEL (label), markup);
		g_free (markup);
	}

	g_free (text_from_buffer);
	return filename;
}

void select_file (GtkWidget *entry, gchar *title)
{
	GtkWidget *file_chooser;
	GtkWidget *toggle_hidden;

	g_return_if_fail (GTK_IS_ENTRY (entry) != FALSE);
	g_return_if_fail (title != NULL);

	file_chooser = gtk_file_chooser_dialog_new (
				title,
				GTK_WINDOW (app_data ? app_data->main_window : NULL),
				GTK_FILE_CHOOSER_ACTION_OPEN,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				NULL);

	toggle_hidden = gtk_check_button_new_with_label (_("Show hidden files"));
	g_signal_connect (G_OBJECT (toggle_hidden), "toggled", G_CALLBACK (toggle_hidden_file), file_chooser);
	gtk_widget_show (toggle_hidden);

	gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (file_chooser), toggle_hidden);
	gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (file_chooser), FALSE);

	gtk_dialog_set_default_response (GTK_DIALOG (file_chooser), GTK_RESPONSE_ACCEPT);

	if (gtk_dialog_run (GTK_DIALOG (file_chooser)) == GTK_RESPONSE_ACCEPT) {
		gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_chooser));
		gtk_entry_set_text (GTK_ENTRY (entry), filename);
		g_free (filename);
	}

	gtk_widget_destroy (file_chooser);
}

static void toggle_hidden_file (GtkToggleButton *togglebutton, gpointer user_data)
{
	gtk_file_chooser_set_show_hidden (GTK_FILE_CHOOSER (user_data), gtk_toggle_button_get_active (togglebutton));
}
