/*
 *      gpg-crypter.c
 *
 *      Copyright 2006 - 2012 Florian Sievers <florian.sievers@gmail.com>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; version 2 of the License.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gpg-crypter.h"
#include "callbacks.h"
#include "crypto-operations.h"

#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1

application_data *app_data;

static gboolean interface_init (void);
static void add_combo_box_string (gchar* string, GtkComboBoxText *combo_box);

/* run Forest, run! */
int main (int argc, char *argv[])
{
#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	/* initialize gtk */
	gtk_init (&argc, &argv);

	/* initialize application data */
	app_data = g_malloc0 (sizeof (application_data));

	/* initialize interface */
	if (interface_init () == FALSE)
		return EXIT_FAILURE;

	/* initialize gpg and gpg keys */
	if (gpg_init () == FALSE)
		return EXIT_FAILURE;

	/* load gpg keys in the combo_box */
	g_ptr_array_foreach (app_data->key_string_array, (GFunc)add_combo_box_string, app_data->key_combo_box);
	gtk_combo_box_set_active (GTK_COMBO_BOX (app_data->key_combo_box), 0);

	/* start me up */
	gtk_widget_show (app_data->main_window);
	gtk_main ();

	/* shut me down */
	gpg_exit ();
	g_free (app_data);

	return EXIT_SUCCESS;
}

static gboolean interface_init (void)
{
	GtkBuilder *builder;
	GError *error_package_path = NULL;
	GError *error_test_path = NULL;
	const gchar *maintainers[] = { "Valère Monseur <valere.monseur@ymail.com>", NULL };

	builder = gtk_builder_new ();

	if (gtk_builder_add_from_file (builder, DATA_DIR "/" PACKAGE "/data/gpg-crypter.glade", &error_package_path) == 0 &&
	    gtk_builder_add_from_file (builder, DATA_DIR "/data/gpg-crypter.glade", &error_test_path) == 0) {
		g_warning ("%s", error_package_path->message);
		g_warning ("%s", error_test_path->message);
		goto interface_init_end;
	}

	app_data->main_window                = GTK_WIDGET (gtk_builder_get_object (builder, "main_window"));
	app_data->key_combo_box              = GTK_WIDGET (gtk_builder_get_object (builder, "key_combo_box"));
	app_data->menu_file_quit             = GTK_WIDGET (gtk_builder_get_object (builder, "menu_file_quit"));
	app_data->text_crypter_box           = GTK_WIDGET (gtk_builder_get_object (builder, "text_crypter_box"));
	app_data->file_crypter_box           = GTK_WIDGET (gtk_builder_get_object (builder, "file_crypter_box"));
	app_data->passphrase_callback_dialog = GTK_WIDGET (gtk_builder_get_object (builder, "passphrase_callback_dialog"));
	app_data->passphrase_callback_entry  = GTK_WIDGET (gtk_builder_get_object (builder, "passphrase_callback_entry"));
	app_data->plaintext_label            = GTK_WIDGET (gtk_builder_get_object (builder, "label_plaintext"));
	app_data->cipher_label               = GTK_WIDGET (gtk_builder_get_object (builder, "label_cipher"));
	app_data->plaintext_textview         = GTK_WIDGET (gtk_builder_get_object (builder, "plaintext_textview"));
	app_data->cipher_textview            = GTK_WIDGET (gtk_builder_get_object (builder, "cipher_textview"));
	app_data->plaintext_file_entry       = GTK_WIDGET (gtk_builder_get_object (builder, "plaintext_file_entry"));
	app_data->cipher_file_entry          = GTK_WIDGET (gtk_builder_get_object (builder, "cipher_file_entry"));
	app_data->about_dialog               = GTK_WIDGET (gtk_builder_get_object (builder, "aboutdialog"));
	app_data->progress_bar               = GTK_WIDGET (gtk_builder_get_object (builder, "progressbar"));

	gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (app_data->about_dialog), VERSION);

#if GTK_CHECK_VERSION(3,4,0)
	gtk_about_dialog_add_credit_section (GTK_ABOUT_DIALOG (app_data->about_dialog), _("Maintained by"), maintainers);
#endif

	gtk_builder_connect_signals (builder, NULL);

interface_init_end:
	g_object_unref (G_OBJECT (builder));

	if (error_package_path != NULL)
		g_error_free (error_package_path);

	if (error_test_path != NULL)
		g_error_free (error_test_path);

	return ((error_package_path != NULL) && (error_test_path != NULL)) ? FALSE : TRUE;
}

static void add_combo_box_string (gchar* string, GtkComboBoxText *combo_box)
{
	g_return_if_fail (string != NULL);
	g_return_if_fail (combo_box != NULL);
	g_return_if_fail (GTK_IS_COMBO_BOX (combo_box) != FALSE);

	gtk_combo_box_text_append_text (combo_box, string);
}
