/* libnetbrake.c -- limit the bandwidth used by a network process
 * Copyright(C) 2001 Salvatore Sanfilippo <antirez@invece.org>
 * All rights reserved.
 *
 * COPYRIGHT AND PERMISSION NOTICE
 *
 * Copyright (c) 2001 Salvatore Sanfilippo
 * 
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <asm/unistd.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <time.h>
#include <locale.h>
#define _GNU_SOURCE
#define __USE_GNU
#include <dlfcn.h>

#include "libnetbrake.h"
#include "bitmap.h"

char *strptime(const char *s, const char  *format,  struct tm *tm);

/* globals */
NB_BITMAP_DECLARE(nb_fd_bitmap,NB_MAX_FD);
static struct nb_loadtable_ele nb_loadtable[NB_LOADTABLE_SZ];
static int nb_loadtable_index = 0;
static int opt_debug = NB_DEFAULT_OPT_DEBUG;
static int opt_bpslimit = NB_DEFAULT_OPT_BPSLIMIT;
static int opt_limit_tcp = NB_DEFAULT_OPT_LIMIT_TCP;
static int opt_limit_udp = NB_DEFAULT_OPT_LIMIT_UDP;
static int opt_rcvbuf_size = NB_DEFAULT_OPT_RCVBUF_SIZE;
static int opt_limit_getc = NB_DEFAULT_OPT_LIMIT_GETC;
static int opt_print_bps = NB_DEFAULT_OPT_PRINT_BPS;
static int opt_netlimit = NB_DEFAULT_OPT_NETLIMIT;
#ifdef NB_HTTPFS
static int opt_httpfs = NB_DEFAULT_OPT_HTTPFS;
static int opt_include_httphdr = NB_DEFAULT_OPT_INCLUDE_HTTPHDR;
#endif

/* prototypes */
static void nb_get_env_options(void);
static long long nb_getusec(void);
static void nb_log(char *fmt, ...);
static void nb_close(int fd);
static void nb_socket(int fd, int domain, int type, int protocol);
static void nb_limit_io(int out, int fd, int load);
static void nb_loadtable_add(size_t load);
static void nb_loadtable_get_bps(int *bps, long long *time, size_t *bytes);

#ifdef NB_HTTPFS
static char *nb_seems_url(const char *p);
static int nb_httpfs_connect(const char *url, const char *method);
static int nb_httpfs_do_open(const char *filename);
static int nb_httpfs_do_stat(const char *filename, struct stat *buf);
static ssize_t nb_net_write(int fd, const void *buf, size_t count);
static FILE *nb_httpfs_fopen(const char *path, const char *mode, int *neterr);
static int nb_httpfs_stat(const char *file_name, struct stat *buf, int *neterr);
static void nb_copy_httphdr(int s, char *dest, size_t len);
static void nb_skip_httphdr(int s);
#endif

/* pointers to the real libc functions */
static int (*next_socket)(int domain, int type, int protocol);
static int (*next_close)(int fd);
static ssize_t (*next_read)(int fd, void *buf, size_t size);
static ssize_t (*next_write)(int fd, const void *buf, size_t size);
static int (*next_recv)(int s, void *buf, size_t len, int flags);
static int (*next_send)(int s, const void *msg, size_t len , int flags);
static int (*next_sendto)(int s, const void *msg, size_t len, int flags,
			  const struct sockaddr *to, socklen_t tolen);
static int (*next_recvfrom)(int s, void *buf, size_t len, int flags,
			    struct sockaddr *from, socklen_t *fromlen);

/* fdopen() allows to call the high level I/O against our sockets */
static int (*next_fclose)(FILE *stream);
static int (*next_fgetc)(FILE *stream);
static int (*next__IO_getc)(FILE *stream);

#ifdef NB_HTTPFS
/* http wrappers */
static int (*next_open)(const char *filename, int oflag, ...);
static int (*next_open64)(const char *filename, int oflag, ...);
static FILE *(*next_fopen)(const char *path, const char *mode);
static FILE *(*next_fopen64)(const char *path, const char *mode);
static int (*next___lxstat)(int ver, const char *file_name, struct stat *buf);
static int (*next___lxstat64)(int ver, const char *file_name, struct stat *buf);
static int (*next___xstat)(int ver, const char *file_name, struct stat *buf);
static int (*next___xstat64)(int ver, const char *file_name, struct stat *buf);
#endif

/* ----------------------- Initiialization related -------------------------- */
#define NB_GET_SYM(name, dest) do { \
	char *error; \
	dest = dlsym(RTLD_NEXT, name); \
	if ((error = dlerror()) != NULL) { \
		fprintf(stderr, "dlsym(" name "): %s\n", error); \
		exit(1); \
	} \
} while(0)

void _init(void)
{
	NB_GET_SYM("socket", next_socket);
	NB_GET_SYM("close", next_close);
	NB_GET_SYM("read", next_read);
	NB_GET_SYM("write", next_write);
	NB_GET_SYM("recv", next_recv);
	NB_GET_SYM("recvfrom", next_recvfrom);
	NB_GET_SYM("send", next_send);
	NB_GET_SYM("sendto", next_sendto);

	NB_GET_SYM("fclose", next_fclose);
	NB_GET_SYM("fgetc", next_fgetc);
	NB_GET_SYM("_IO_getc", next__IO_getc);

#ifdef NB_HTTPFS
	NB_GET_SYM("open", next_open);
	NB_GET_SYM("open64", next_open64);
	NB_GET_SYM("fopen", next_fopen);
	NB_GET_SYM("fopen64", next_fopen64);
	NB_GET_SYM("__lxstat", next___lxstat);
	NB_GET_SYM("__lxstat64", next___lxstat64);
	NB_GET_SYM("__xstat", next___xstat);
	NB_GET_SYM("__xstat64", next___xstat64);
#endif

	nb_get_env_options();
	NB_BITMAP_INIT(nb_fd_bitmap, 0);
	memset(&nb_loadtable, 0, sizeof(nb_loadtable));
}

void nb_get_env_options(void)
{
	char *p;

	if ((p = getenv("NETBRAKE_DEBUG")) != NULL)
		opt_debug = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_BPS")) != NULL)
		opt_bpslimit = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_LIMIT_TCP")) != NULL)
		opt_limit_tcp = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_LIMIT_UDP")) != NULL)
		opt_limit_udp = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_RCVBUF_SIZE")) != NULL)
		opt_rcvbuf_size = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_LIMIT_GETC")) != NULL)
		opt_limit_getc = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_PRINT_BPS")) != NULL)
		opt_print_bps = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_NETLIMIT")) != NULL)
		opt_netlimit = strtoul(p, NULL, 10);
#ifdef NB_HTTPFS
	if ((p = getenv("NETBRAKE_HTTP_FS")) != NULL)
		opt_httpfs = strtoul(p, NULL, 10);
	if ((p = getenv("NETBRAKE_HTTP_HDR")) != NULL)
		opt_include_httphdr = strtoul(p, NULL, 10);
#endif
}

/* -------------------------- netbrake wrappers ----------------------------- */
int socket(int domain, int type, int protocol)
{
	int retval;

	retval = next_socket(domain, type, protocol);
	if (retval >= 0)
		nb_socket(retval, domain, type, protocol);
	return retval;
}

int close(int fd)
{
	int retval;

	retval = next_close(fd);
	if (retval == 0)
		nb_close(fd);
	return retval;
}

ssize_t read(int fd, void *buf, size_t size)
{
	int retval;

	retval = next_read(fd, buf, size);
	nb_limit_io(0, fd, retval);
	return retval;
}

ssize_t write(int fd, const void *buf, size_t size)
{
	int retval;

	retval = next_write(fd, buf, size);
	nb_limit_io(1, fd, retval);
	return retval;
}

int recv(int s, void *buf, size_t len, int flags)
{
	int retval;

	retval = next_recv(s, buf, len, flags);
	nb_limit_io(0, s, retval);
	return retval;
}

int recvfrom(int s, void *buf, size_t len, int flags,
	     struct sockaddr *from, socklen_t *fromlen)
{
	int retval;

	retval = next_recvfrom(s, buf, len, flags, from, fromlen);
	nb_limit_io(0, s, retval);
	return retval;
}

int send(int s, const void *msg, size_t len , int flags)
{
	int retval;

	retval = next_send(s, msg, len, flags);
	nb_limit_io(1, s, retval);
	return retval;
}

int sendto(int s, const void *msg, size_t len, int flags,
	   const struct sockaddr *to, socklen_t tolen)
{
	int retval;
	retval = next_sendto(s, msg, len, flags, to, tolen);
	nb_limit_io(1, s, retval);
	return retval;
}

int fclose(FILE *stream)
{
	int fd = fileno(stream);
	int retval;

	retval = next_fclose(stream);
	if (retval == 0)
		nb_close(fd);
	return retval;
}

/* for default fgetc() and get(c) are not limited. BTW some silly
 * application like the linux ftp client seems to like a lot
 * to call fgetc() for every char to read for commands like 'ls'. */

int fgetc(FILE *stream)
{
	int fd;
	int retval;

	retval = next_fgetc(stream);
	if (opt_limit_getc) {
		fd = fileno(stream);
		if (retval != EOF)
			nb_limit_io(0, fd, 1);
	}
	return retval;
}

int _IO_getc(FILE *stream)
{
	int fd;
	int retval;

	retval = next__IO_getc(stream);
	if (opt_limit_getc) {
		fd = fileno(stream);
		if (retval != EOF)
			nb_limit_io(0, fd, 1);
	}
	return retval;
}

/* ----------------------- socket descriptors tracking ---------------------- */
void nb_socket(int fd, int domain, int type, int protocol)
{
	int size;

	/* bandwidth limitation disabled? */
	if (opt_netlimit == 0)
		return;

	/* Only trace the right sockets */
	if (domain != PF_INET) return;
	if (type == SOCK_DGRAM && opt_limit_udp == 0) return;
	if (type == SOCK_STREAM && opt_limit_tcp == 0) return;

	if (fd >= NB_MAX_FD || fd < 0) {
		nb_log("netbrake: out of table filedes (%d) in nb_socket()\n",
			fd);
		return;
	}
	NB_BITMAP_SET(nb_fd_bitmap, fd);
	nb_log("Tracing socketdes %d\n", fd);
	/* We need to set a little recv socket buffer, otherwise for
	 * little files our bandwidth limitation will not work */
	size = opt_rcvbuf_size;
	if (type == SOCK_STREAM)
		if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, &size, sizeof(size))
		    == -1)
			nb_log("netbrake: setsockopt() error: %s\n",
				strerror(errno));
}

void nb_close(int fd)
{
	if (fd >= NB_MAX_FD || fd < 0) {
		nb_log("netbrake: out of table filedes (%d) in nb_close()\n",
			fd);
		return;
	}
	if (NB_BITMAP_TEST(nb_fd_bitmap, fd) == 0)
		return;
	NB_BITMAP_UNSET(nb_fd_bitmap, fd);
	nb_log("Socketdes is no longer traced %d\n", fd);
}

/* ------------------------------- I/O limit -------------------------------- */
void nb_loadtable_add(size_t load)
{
	nb_loadtable[nb_loadtable_index].usec = nb_getusec();
	nb_loadtable[nb_loadtable_index].load = load;
	nb_loadtable_index = (nb_loadtable_index+1)%NB_LOADTABLE_SZ;
}

/* This function return the byte-per-second using the last
 * NB_LOADTABLE_SZ samples.
 * The first (in report time) load is excluded from the sum
 * since we use this as initial time, otherwise you get it
 * overstimed */
void nb_loadtable_get_bps(int *bps, long long *time, size_t *bytes)
{
	long long min, max = 0;
	size_t sum = 0, load_min;
	int i;
	long difftime;

	min = nb_loadtable[0].usec;
	load_min = nb_loadtable[0].load;
	for (i = 0; i < NB_LOADTABLE_SZ; i++) {
		if (nb_loadtable[i].usec == 0)
			continue;
		if (min > nb_loadtable[i].usec) {
			min = nb_loadtable[i].usec;
			load_min = nb_loadtable[i].load;
		}
		if (max < nb_loadtable[i].usec)
			max = nb_loadtable[i].usec;
		sum += nb_loadtable[i].load;
	}
	difftime = max-min;
	if (!difftime) {
		*bps = 0;
		return;
	}
	sum -= load_min;
	*bps = ((float)sum/((float)difftime/1000000));
	if (time) *time = difftime;
	if (bytes) *bytes = sum;
}

void nb_limit_io(int out, int fd, int load)
{
	int bps, average_bps;
	long long time, total_time;
	size_t bytes;
	static size_t total_bytes = 0;
	static long long initial_time = 0;

	/* Syscall on error? */
	if (load < 0)
		return;
	
	/* Invalid filedes range? */
	if (fd < 0 || fd >= NB_MAX_FD) {
		nb_log("netbrake: out of table filedes (%d) in nb_limit_io()\n",
			fd);
		return;
	}

	/* Is this descriptor relevant? (i.e. set in the table) */
	if (NB_BITMAP_TEST(nb_fd_bitmap, fd) == 0)
		return;

	/* limit the (rougly speaking) instantaous speed... */
	/* Add the new sample in the table */
	nb_loadtable_add(load);
	nb_loadtable_get_bps(&bps, &time, &bytes);
	nb_log("Inst BPS: %d\n", bps);
	if (bps > opt_bpslimit) {
		long long ideal_time;
		unsigned int sleep_sec, sleep_usec;

		ideal_time = ((float)bytes/opt_bpslimit)*1000000;
		sleep_sec = (ideal_time - time)/1000000;
		sleep_usec = (ideal_time - time)%1000000;
		nb_log("Inst sleep s=%u ms=%u\n", sleep_sec, sleep_usec);
		sleep(sleep_sec);
		usleep(sleep_usec);
	}

	/* then limit the average speed -- this saves us from approximations
	 * usleep() limits and so on */
	/* We need the initial time to compute the average speed */
	if (initial_time == 0)
		initial_time = nb_getusec();

	total_bytes += load;
	total_time = nb_getusec() - initial_time;
	if (total_time > 1000000) {
		long long ideal_time;
		unsigned int sleep_sec, sleep_usec;

		average_bps = (float)total_bytes/((float)total_time/1000000);
		if (opt_print_bps) {
			fprintf(stderr, "bandwidth used: current %d/s"
					", average %d/s\n", bps, average_bps);
		}
		nb_log("average BPS: %d\n", average_bps);
		if (average_bps > opt_bpslimit) {
			ideal_time = ((float)total_bytes/opt_bpslimit)*1000000;
			sleep_sec = (ideal_time - total_time)/1000000;
			sleep_usec = (ideal_time - total_time)%1000000;
			nb_log("avr sleep s=%u ms=%u\n", sleep_sec, sleep_usec);
			sleep(sleep_sec);
			usleep(sleep_usec);
		}
	}
}

/* -------------------------------- HTTP warp ------------------------------- */
#ifdef NB_HTTPFS
int open(const char *filename, int oflag, ...)
{
	int mode;
	int retval;

	if (opt_httpfs && (oflag & (~0x8000)) == O_RDONLY) {
		if (nb_seems_url(filename) != NULL)
			return nb_httpfs_do_open(filename);
	}

	if (oflag & O_CREAT) {
		va_list arg;
		va_start(arg, oflag);
		mode = va_arg(arg, int);
		va_end(arg);
		retval = next_open(filename, oflag, mode);
	} else {
		retval = next_open(filename, oflag);
	}
	return retval;
}

int open64(const char *filename, int oflag, ...)
{
	int mode;
	int retval;

	if (opt_httpfs && (oflag & (~0x8000)) == O_RDONLY) {
		if (nb_seems_url(filename) != NULL)
			return nb_httpfs_do_open(filename);
	}

	if (oflag & O_CREAT) {
		va_list arg;
		va_start(arg, oflag);
		mode = va_arg(arg, int);
		va_end(arg);
		retval = next_open(filename, oflag, mode);
	} else {
		retval = next_open(filename, oflag);
	}
	return retval;
}

FILE *fopen(const char *path, const char *mode)
{
	int neterr;
	FILE *fp;

	fp = nb_httpfs_fopen(path, mode, &neterr);
	if (fp != NULL)
		return fp;
	else if (neterr)
		return NULL;
	return next_fopen(path, mode);
}

FILE *fopen64(const char *path, const char *mode)
{
	int neterr;
	FILE *fp;

	fp = nb_httpfs_fopen(path, mode, &neterr);
	if (fp != NULL)
		return fp;
	else if (neterr)
		return NULL;
	return next_fopen64(path, mode);
}

int __lxstat(int ver, const char *file_name, struct stat *buf)
{
	int neterr;
	int ret;

	ret = nb_httpfs_stat(file_name, buf, &neterr);
	if (ret == 0)
		return 0;
	else if (neterr)
		return -1;
	return next___lxstat(ver, file_name, buf);
}

int __lxstat64(int ver, const char *file_name, struct stat *buf)
{
	int neterr;
	int ret;

	ret = nb_httpfs_stat(file_name, buf, &neterr);
	if (ret == 0)
		return 0;
	else if (neterr)
		return -1;
	return next___lxstat64(ver, file_name, buf);
}

int __xstat(int ver, const char *file_name, struct stat *buf)
{
	int neterr;
	int ret;

	ret = nb_httpfs_stat(file_name, buf, &neterr);
	if (ret == 0)
		return 0;
	else if (neterr)
		return -1;
	return next___xstat(ver, file_name, buf);
}

int __xstat64(int ver, const char *file_name, struct stat *buf)
{
	int neterr;
	int ret;

	ret = nb_httpfs_stat(file_name, buf, &neterr);
	if (ret == 0)
		return 0;
	else if (neterr)
		return -1;
	return next___xstat64(ver, file_name, buf);
}

/* -------------------------- HTTP FS calls --------------------------------- */

int nb_httpfs_stat(const char *file_name, struct stat *buf, int *neterr)
{
	*neterr = 1;
	if (opt_httpfs && (nb_seems_url(file_name) != NULL)) {
		if (nb_httpfs_do_stat(file_name, buf) == -1)
			return -1;
		return 0;
	}
	*neterr = 0;
	return -1;
}

FILE *nb_httpfs_fopen(const char *path, const char *mode, int *neterr)
{
	*neterr = 1;
	if (opt_httpfs && strcmp(mode, "r") == 0) {
		if (nb_seems_url(path) != NULL) {
			int s;
			FILE *fp;
			if ((s = nb_httpfs_do_open(path)) == -1) {
				return NULL;
			}
			if ((fp = fdopen(s, mode)) == NULL) {
				close(s);
				return NULL;
			}
			return fp;
		}
	}
	*neterr = 0;
	return NULL;
}

char *nb_seems_url(const char *p)
{
	return strstr(p, "http://");
}

#define NB_HTTP_REQUEST \
"%s %s HTTP/1.0\r\n" \
"User-Agent: Netbrake/0.1\r\n" \
"Host: %s:%s\r\n" \
"Accept: */*\r\n\r\n"

#define URLELE_SZ 128
#define HTTPREQ_SZ 1024
#define mystrncpy(d,s,l) do { strncpy(d,s,l); d[(l)-1] = '\0'; } while(0)
int nb_httpfs_connect(const char *url, const char *method)
{
	struct sockaddr_in sa;
	int s;
	char host[URLELE_SZ];
	char port[URLELE_SZ];
	char path[URLELE_SZ];
	char request[HTTPREQ_SZ];
	char *p;

	p = strchr(url, '/') + 2;
	mystrncpy(host, p, URLELE_SZ);
	p = strchr(host, '/');
	if (p) {
		char *j;
		*p = '\0';
		j = strrchr(host, ':');
		if (j) {
			*j = '\0';
			mystrncpy(port, j+1, URLELE_SZ);
		} else {
			mystrncpy(port, "80", URLELE_SZ);
		}
		*p = '/';
		mystrncpy(path, p, URLELE_SZ);
		*p = '\0';
	} else {
		mystrncpy(port, "80", URLELE_SZ);
		mystrncpy(path, "/", URLELE_SZ);
	}

	memset(&sa, 0, sizeof(sa));
	sa.sin_family = AF_INET;
	if (inet_aton(host, &sa.sin_addr) == 0) {
		struct hostent *he;
		he = gethostbyname(host);
		if (he == NULL) {
			errno = ENOENT;
			return -1;
		}
		sa.sin_addr.s_addr = ((struct in_addr*) he->h_addr)->s_addr;
	}
	sa.sin_port = htons(strtoul(port, NULL, 10));
	if ((s = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
		return -1;
	}
	if (connect(s, (struct sockaddr*)&sa, sizeof(sa)) == -1) {
		close(s);
		return -1;
	}
	snprintf(request, HTTPREQ_SZ, NB_HTTP_REQUEST,
			method, path, host, port);
	request[HTTPREQ_SZ-1] = '\0';
	if (nb_net_write(s, request, strlen(request)) != strlen(request)) {
		close(s);
		return -1;
	}
	return s;
}

int nb_httpfs_do_open(const char *filename)
{
	char *url = nb_seems_url(filename);
	int s;

	if ((s = nb_httpfs_connect(url, "GET")) == -1)
		return -1;
	/* skip the HTTP reply header, if needed */
	if (!opt_include_httphdr)
		nb_skip_httphdr(s);
	return s;
}

#define HTTPHDR_BUFSZ 4096
#define HTTP_CONTENT_LENGTH "Content-Length:"
#define HTTP_LAST_MODIFIED "Last-Modified:"
#define HTTP_DATE_FORMAT " %A, %e %h %Y %T "
int nb_httpfs_do_stat(const char *filename, struct stat *buf)
{
	char *url = nb_seems_url(filename);
	char header[HTTPHDR_BUFSZ];
	int s;
	off_t st_size;
	time_t st_time;
	ino_t st_ino;
	char *p;

	if ((s = nb_httpfs_connect(url, "GET")) == -1)
		return -1;
	nb_copy_httphdr(s, header, HTTPHDR_BUFSZ);
	memset(buf, 0, sizeof(struct stat));
	/* Try to fill the stat buffer with sane values,
	 * at least for the fields we have some info */
	if ((p = strstr(header, HTTP_CONTENT_LENGTH)) != NULL) {
		p += strlen(HTTP_CONTENT_LENGTH);
		st_size = strtoul(p, NULL, 10);
	} else {
		/* set it to 100MB hoping that the application will
		 * handle the EOF in a sane way */
		st_size = 1024*1024*100;
	}

	if ((p = strstr(header, HTTP_LAST_MODIFIED)) != NULL) {
		char *l, *a;
		struct tm t;

		p += strlen(HTTP_LAST_MODIFIED);
		if ((l = setlocale(LC_TIME, NULL)) != NULL)
			if (setlocale(LC_TIME, "C") == NULL)
				l = NULL;
		if ((a = strptime(p, HTTP_DATE_FORMAT, &t)) != NULL) {
			st_time = mktime(&t);
		} else {
			st_time = time(NULL); /* we can't get better... */
		}
		if (l != NULL)
			setlocale(LC_TIME, l);
	} else {
		st_time = time(NULL); /* we can't get better... */
	}

	/* hash the url to get the inode */
	st_ino = 0;
	p = (char*) url;
	while(*p) {
		if ((p-url)%2)
			st_ino += *p;
		else
			st_ino *= *p;
		p++;
	}

	buf->st_dev = 0;
	buf->st_ino = st_ino;
	buf->st_mode = S_IRUSR | S_IFREG;
	buf->st_nlink = 1;
	buf->st_uid = getuid();
	buf->st_gid = getgid();
	buf->st_rdev = 0;
	buf->st_size = st_size;
	/*
	buf->st_blksize = 1024;
	buf->st_blocks = (st_size+1023) & (~1023);
	*/
	buf->st_atime = buf->st_mtime = buf->st_ctime = st_time;

	close(s);
	return 0;
}

ssize_t nb_net_write(int fd, const void *buf, size_t count)
{
	int n_written;
	ssize_t saved_count = count;

	while(count) {
		n_written = next_write(fd, buf, count);
		if (n_written <= 0)
			return n_written;
		buf += n_written;
		count -= n_written;
	}
	return saved_count;
}

/* This function copy the http header in the dest buffer.
 * If dest is == NULL the header is just discarded from
 * the stream. The nb_skip_httphdr() call nb_copy_httphdr()
 * this way.
 *
 * We need to copy/skip the http header but we aren't bufferized,
 * the only way to avoid to do a syscall for every char seems
 * to use the MSG_PEEK flag of recv(2) */
#define HTTPHDR_TERM "\r\n\r\n"
#define SKIPHDR_BUFSZ 1024
void nb_copy_httphdr(int s, char *dest, size_t len)
{
	char buffer[SKIPHDR_BUFSZ];
	int n_read, offset;
	char *t;
	size_t termlen = strlen(HTTPHDR_TERM), copylen;

	/* If dest is not null the buffer should be at least 1 byte */
	if (dest && len == 0)
		return;
	len--; /* reserve space for the nul term */

	while(1) {
		n_read = recv(s, buffer, SKIPHDR_BUFSZ-1, MSG_PEEK);
		if (n_read <= 0) {
			if (dest)
				*dest = '\0';
			return;
		}
		buffer[n_read] = '\0';
		if ((t = strstr(buffer, HTTPHDR_TERM)) != NULL) {
			offset = t-buffer;
			/* note that the next read can't return
			 * less than `offset' bytes since we already
			 * know they are in the socket buffer */

			/* discard this data */
			read(s, buffer, offset+termlen);
			if (dest) {
				copylen = MIN(offset+termlen, len);
				memcpy(dest, buffer, copylen);
				dest += copylen;
				len -= copylen;
				*dest = '\0';
			}
			return;
		}
		/* we need to trash out the readed bytes - the size of
		 * the http header terminator. Otherwise we may miss
		 * a crossing match */

		/* If we peek less than bytes than the terminator size
		 * we just retry the peek to get hopefully more data */
		if (n_read <= termlen)
			continue;
		/* trash it, again we don't care about the retval
		 * since the data was already in the buffer */
		read(s, buffer, n_read - termlen);
		if (dest) {
			copylen = MIN(n_read - termlen, len);
			memcpy(dest, buffer, copylen);
			dest += copylen;
			len -= copylen;
		}
	}
	/* unreached */
}

void nb_skip_httphdr(int s)
{
	nb_copy_httphdr(s, NULL, 0);
}

#endif /* NB_HTTPFS */
/* ---------------------------------- misc ---------------------------------- */
long long nb_getusec(void)
{
	struct timeval tv;

	gettimeofday(&tv, NULL);
	return ((long long)tv.tv_sec*1000000)+tv.tv_usec;
}

void nb_log(char *fmt, ...)
{
	va_list ap;

	if (opt_debug == 0)
		return;
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}
