use crate::util::BuilderHelper;
use glib::{clone, Cast};
use gtk::{Box, Button, ButtonExt, ContainerExt, Popover, PopoverExt, Revealer, RevealerExt, WidgetExt};
use libhandy::{Leaflet, LeafletExt, Squeezer, SqueezerExt};
use parking_lot::RwLock;
use std::sync::Arc;

#[derive(Clone, Debug)]
pub struct ResponsiveLayout {
    pub state: Arc<RwLock<ResponsiveState>>,
    pub left_back_button: Button,
    pub right_back_button: Button,
    pub update_button: Button,
    pub update_button_top: Button,
    pub offline_status_button: Button,
    pub offline_status_button_top: Button,
    pub offline_popover: Popover,
    pub online_popover: Popover,
    pub major_leaflet: Leaflet,
    pub minor_leaflet: Leaflet,
    pub sidebar_box: Box,
    pub article_list_box: Box,
    pub article_view_box: Box,
    pub article_view_footer_revealer: Revealer,
    pub article_view_header_squeezer: Squeezer,
}

impl ResponsiveLayout {
    pub fn new(builder: &BuilderHelper) -> Arc<ResponsiveLayout> {
        let state = Arc::new(RwLock::new(ResponsiveState::new()));

        let minor_leaflet = builder.get::<Leaflet>("minor_leaflet");
        let major_leaflet = builder.get::<Leaflet>("major_leaflet");
        let left_back_button = builder.get::<Button>("left_back_button");
        let right_back_button = builder.get::<Button>("right_back_button");
        let update_button = builder.get::<Button>("update_button");
        let update_button_top = builder.get::<Button>("update_button_top");
        let offline_status_button = builder.get::<Button>("offline_status_button");
        let offline_status_button_top = builder.get::<Button>("offline_status_button_top");
        let offline_popover = builder.get::<Popover>("offline_popover");
        let online_popover = builder.get::<Popover>("online_popover");
        let sidebar_box = builder.get::<Box>("feedlist_box");
        let article_list_box = builder.get::<Box>("articlelist_box");
        let article_view_box = builder.get::<Box>("articleview_box");
        let article_view_footer_revealer = builder.get::<Revealer>("articleview_footer_revealer");
        let article_view_header_squeezer = builder.get::<Squeezer>("articleview_header_squeezer");
        let layout = ResponsiveLayout {
            state,
            left_back_button,
            right_back_button,
            update_button,
            update_button_top,
            offline_popover,
            online_popover,
            offline_status_button,
            offline_status_button_top,
            major_leaflet,
            minor_leaflet,
            sidebar_box,
            article_list_box,
            article_view_box,
            article_view_footer_revealer,
            article_view_header_squeezer,
        };
        let layout = Arc::new(layout);
        Self::setup_signals(&layout);
        layout
    }

    fn setup_signals(layout: &Arc<ResponsiveLayout>) {
        let major_leaflet = layout.major_leaflet.clone();
        let minor_leaflet = layout.minor_leaflet.clone();

        layout.major_leaflet.connect_property_folded_notify(clone!(
            @weak layout => @default-panic, move |_leaflet|
        {
            if minor_leaflet.get_folded() {
                layout.state.write().minor_leaflet_folded = true;
                Self::process_state_change(&layout);
            }
            layout.state.write().major_leaflet_folded = true;
            Self::process_state_change(&layout);
        }));

        layout.minor_leaflet.connect_property_folded_notify(clone!(
            @weak layout => @default-panic, move |_leaflet|
        {
            if !major_leaflet.get_folded() {
                return;
            }
            layout.state.write().minor_leaflet_folded = true;
            Self::process_state_change(&layout);
        }));

        // swipe back should be handled the same way as back button press
        layout.minor_leaflet.connect_property_visible_child_name_notify(
            clone!(@weak layout => @default-panic, move |leaflet| {
                if leaflet.get_visible_child_name().map(|s| s.as_str().to_owned()) == Some("feedlist_box".into()) {
                    layout.state.write().left_button_clicked = true;
                    Self::process_state_change(&layout);
                }
            }),
        );

        layout
            .left_back_button
            .connect_clicked(clone!(@weak layout => @default-panic, move |_button| {
                layout.state.write().left_button_clicked = true;
                Self::process_state_change(&layout);
            }));

        layout
            .right_back_button
            .connect_clicked(clone!(@weak layout => @default-panic, move |_button| {
                layout.state.write().right_button_clicked = true;
                Self::process_state_change(&layout);
            }));

        layout
            .article_view_header_squeezer
            .connect_property_visible_child_notify(clone!(@weak layout => @default-panic, move |squeezer| {
                    if let Some(widget) = squeezer.get_visible_child() {
                        if let Ok(squeezer_box) = widget.downcast::<Box>() {
                            layout.state.write().header_squeezer_hidden = squeezer_box.get_children().is_empty();
                            Self::process_state_change(&layout);
                        }
                    }
            }));
    }

    pub fn process_state_change(&self) {
        self.article_view_footer_revealer
            .set_reveal_child(self.state.read().header_squeezer_hidden);

        if self.state.read().major_leaflet_folded {
            // article view (dis)appeared
            if !self.major_leaflet.get_folded() {
                self.right_back_button.set_visible(false);
                self.left_back_button.set_visible(false);
                self.major_leaflet.set_visible_child(&self.minor_leaflet);
                self.minor_leaflet.set_visible_child(&self.sidebar_box);
                self.update_button_top.set_visible(false);
            }

            self.state.write().major_leaflet_folded = false;
            return;
        }

        if self.state.read().minor_leaflet_folded {
            // article list (dis)appeared
            if !self.minor_leaflet.get_folded() {
                self.left_back_button.set_visible(false);
                self.minor_leaflet.set_visible_child(&self.sidebar_box);
            }

            if self.update_button.get_visible() {
                if self.minor_leaflet.get_folded() {
                    self.update_button_top.set_visible(true);
                } else {
                    self.update_button_top.set_visible(false);
                }
            } else {
                self.update_button_top.set_visible(false);
            }

            if self.offline_status_button.get_visible() {
                if self.minor_leaflet.get_folded() {
                    self.offline_status_button_top.set_visible(true);
                } else {
                    self.offline_status_button_top.set_visible(false);
                }
            } else {
                self.offline_status_button_top.set_visible(false);
            }

            if self.update_button_top.get_visible() {
                self.offline_popover.set_relative_to(Some(&self.update_button_top));
                self.online_popover.set_relative_to(Some(&self.update_button_top));
            } else {
                self.offline_popover.set_relative_to(Some(&self.update_button));
                self.online_popover.set_relative_to(Some(&self.update_button));
            }

            self.state.write().minor_leaflet_folded = false;
            return;
        }

        if self.state.read().left_button_clicked {
            // left back
            self.minor_leaflet.set_visible_child(&self.sidebar_box);

            self.state.write().left_button_clicked = false;
            return;
        }

        if self.state.read().right_button_clicked {
            // right back
            //self.minor_leaflet.set_visible_child(&self.article_list_box);
            self.major_leaflet.set_visible_child(&self.minor_leaflet);
            self.right_back_button.set_visible(false);

            self.state.write().right_button_clicked = false;
            return;
        }

        if self.state.read().major_leaflet_selected {
            // article selected
            if self.major_leaflet.get_folded() {
                self.major_leaflet.set_visible_child(&self.article_view_box);
                self.right_back_button.set_visible(true);
            }

            self.state.write().major_leaflet_selected = false;
            return;
        }

        if self.state.read().minor_leaflet_selected {
            // sidebar selected
            if self.minor_leaflet.get_folded() {
                self.minor_leaflet.set_visible_child(&self.article_list_box);
                self.left_back_button.set_visible(true);
            }

            self.state.write().minor_leaflet_selected = false;
            return;
        }
    }
}

#[derive(Clone, Debug)]
pub struct ResponsiveState {
    pub left_button_clicked: bool,
    pub right_button_clicked: bool,
    pub major_leaflet_selected: bool,
    pub major_leaflet_folded: bool,
    pub minor_leaflet_selected: bool,
    pub minor_leaflet_folded: bool,
    pub header_squeezer_hidden: bool,
}

impl ResponsiveState {
    pub fn new() -> Self {
        ResponsiveState {
            left_button_clicked: false,
            right_button_clicked: false,
            major_leaflet_selected: false,
            major_leaflet_folded: false,
            minor_leaflet_selected: false,
            minor_leaflet_folded: false,
            header_squeezer_hidden: false,
        }
    }
}
