/******************************************************************************
*
* Copyright (C) 2002 Hugo PEREIRA <mailto: hugo.pereira@free.fr>
*
* This is free software; you can redistribute it and/or modify it under the
* terms of the GNU General Public License as published by the Free Software
* Foundation; either version 2 of the License, or (at your option) any later
* version.
*
* This software is distributed in the hope that it will be useful, but WITHOUT
* Any WARRANTY; without even the implied warranty of MERCHANTABILITY or
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
* for more details.
*
* You should have received a copy of the GNU General Public License along with
* this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "AskPassQt.h"

#include "CompositeEngine.h"
#include "IconEngine.h"
#include "LineEditor.h"
#include "RoundedRegion.h"
#include "ShadowLabel.h"
#include "SvgEngine.h"
#include "XmlOptions.h"
#include "WinUtil.h"

#include <QFile>
#include <QApplication>
#include <QDesktopWidget>
#include <QLabel>
#include <QLayout>
#include <QPushButton>
#include <QPainter>

#include <cstdio>
#include <iostream>

//______________________________________________________
AskPassQt::AskPassQt( QWidget* parent ) :
    Transparency::TransparentWidget( parent ),
    widgetDragMonitor_( this )
{
    setWindowTitle( tr( "Ssh Password" ) );
    setWindowFlags( Qt::FramelessWindowHint | Qt::WindowStaysOnTopHint );

    widgetDragMonitor_.setEnabled( true );

    // initialize svg engine
    Svg::SvgEngine::get().reload();

    // check if custom palette should be used for widgets
    bool hasThemePalette(
        XmlOptions::get().get<bool>( "USE_SVG" ) &&
        XmlOptions::get().get<bool>( "SVG_USE_PLASMA_INTERFACE" ) &&
        Svg::SvgEngine::get().hasThemePalette() );

    // main layout
    auto mainLayout = new QVBoxLayout;
    setLayout( mainLayout );
    layout()->setMargin(0);
    layout()->setSpacing(0);

    auto hLayout( new QHBoxLayout );
    hLayout->setMargin(0);
    layout()->addItem( hLayout );

    // pixmap
    auto&& pixmap = IconEngine::get( "dialog-password" ).pixmap( QSize( 48, 48 ) );
    if( !pixmap.isNull() )
    {
        QLabel* label = new QLabel( this );
        label->setMargin(10);
        label->setPixmap( pixmap );
        hLayout->addWidget( label, 0, Qt::AlignHCenter );
    }

    editor_ = new LineEditor( this );
    if( hasThemePalette ) editor_->setPalette( Svg::SvgEngine::get().themePalette() );

    hLayout->addWidget( editor_ );
    editor_->setEchoMode( QLineEdit::Password );
    editor_->setPlaceholderText( tr("Enter your SSH pass phrase here") );
    connect( editor_, SIGNAL(returnPressed()), SLOT(accept()) );

    // buttons
    hLayout = new QHBoxLayout;
    hLayout->setMargin(0);
    layout()->addItem( hLayout );

    // stretch
    hLayout->addStretch( 1 );
    QPushButton* button;

    // ok button
    button = new QPushButton( this );
    if( hasThemePalette ) button->setPalette( Svg::SvgEngine::get().themePalette() );
    hLayout->addWidget( button );
    button->setIcon( IconEngine::get( "accept.png") );
    button->setText( tr( "OK" ) );
    button->setFlat( true );
    connect( button, SIGNAL(clicked()), SLOT(accept()) );

    // cancel button
    button = new QPushButton( this );
    if( hasThemePalette ) button->setPalette( Svg::SvgEngine::get().themePalette() );
    hLayout->addWidget( button );
    button->setIcon( IconEngine::get( "cancel.png" ) );
    button->setShortcut( Qt::Key_Escape );
    button->setText( tr( "Cancel" ) );
    button->setFlat( true );
    connect( button, SIGNAL(clicked()), SLOT(cancel()) );

    // margins
    auto&& margins( Svg::SvgEngine::get().margins() );
    setContentsMargins( margins.left(), margins.top(), margins.right(), margins.bottom() );
    _setMargins( margins );

    // outer padding
    _setOuterPadding( Svg::SvgEngine::get().outerPadding() );

    // resize
    QSize size( 400, minimumSizeHint().height() );
    resize( size );

    // move widget at desktop center
    QPoint position(
        ( ( QApplication::desktop()->width() - size.width() ) / 2 ),
        ( ( QApplication::desktop()->height() - size.height() ) / 2 ) );
    move( position );

}

//______________________________________________________
void AskPassQt::paintEvent( QPaintEvent* event )
{

    #if defined(Q_OS_WIN)

    QPixmap pixmap( size() );
    pixmap.fill( Qt::transparent );
    _paint( pixmap, event->rect() );
    WinUtil( this ).update( pixmap, 1 );

    #else

    _paint( *this, event->rect() );

    #endif

}

//______________________________________________________
void AskPassQt::_paint( QPaintDevice& device, const QRect& clip )
{

    // draw background
    QPainter painter( &device );

    #if !defined(Q_OS_WIN)
    painter.setClipRect( clip );
    #endif

    // draw background
    if( Transparency::CompositeEngine::get().isAvailable() )
    {

        painter.setRenderHints(QPainter::SmoothPixmapTransform);
        painter.setCompositionMode(QPainter::CompositionMode_Source );
        painter.fillRect( rect(), Qt::transparent );

    }

    // draw svg
    painter.drawPixmap( QPoint( 0, 0 ), Svg::SvgEngine::get().get( Svg::SvgId( size() ) ) );
    painter.end();

    // loop over children and render them
    #if defined(Q_OS_WIN)
    for( const auto& widget:findChildren<QWidget*>() )
    { widget->render( &device, widget->geometry().topLeft(), widget->rect().intersected( rect().translated( -widget->geometry().topLeft() ) ), 0 ); }
    #endif

}


//______________________________________________________
void AskPassQt::resizeEvent( QResizeEvent* event )
{
    setMask( RoundedRegion( rect() ).get() );

    // announce blur region
    Base::Margins outerPadding( Svg::SvgEngine::get().outerPadding() );
    if( outerPadding.isNull() )
    {
        outerPadding = Svg::SvgEngine::get().margins();
        outerPadding -= Base::Margins( 4 );
    }

    _updateBlurRegion( outerPadding.adjustedRect( rect() ) );

    return QWidget::resizeEvent( event );
}

//______________________________________________________
void AskPassQt::accept()
{
    puts( editor_->text().toUtf8() );
    close();
}

//______________________________________________________
void AskPassQt::cancel()
{ close(); }
