#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2021, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
import logging; logger = logging.getLogger(__name__); logger.info("import")

#Standard Library Modules

#Third Party Modules
from PyQt5 import QtWidgets, QtCore, QtGui

#Our modules
import engine.api as api
from template.engine.duration import D4

MAX_QT_SIZE = 2147483647-1

def convertSubdivisionsSubMenu(mainWindow):
    class Submenu(QtWidgets.QDialog):
        def __init__(self, mainWindow):
            super().__init__(mainWindow) #if you don't set the parent to the main window the whole screen will be the root and the dialog pops up in the middle of it.
            #self.setModal(True) #we don't need this when called with self.exec() instead of self.show()
            self.layout = QtWidgets.QFormLayout()
            self.setLayout(self.layout)
            self.numberOfSubdivisions = QtWidgets.QSpinBox()
            self.numberOfSubdivisions.setMinimum(1)
            self.numberOfSubdivisions.setMaximum(4)#
            self.numberOfSubdivisions.setValue(mainWindow.numberOfSubdivisions.value())

            self.layout.addRow(QtCore.QCoreApplication.translate("convertSubdivisionsSubMenu", "New Grouping"), self.numberOfSubdivisions)

            self.errorHandling = QtWidgets.QComboBox()
            self.errorHandling.addItems([
                QtCore.QCoreApplication.translate("convertSubdivisionsSubMenu", "Do nothing"),
                QtCore.QCoreApplication.translate("convertSubdivisionsSubMenu", "Delete wrong steps"),
                QtCore.QCoreApplication.translate("convertSubdivisionsSubMenu", "Merge wrong steps"),
                ])
            self.layout.addRow(QtCore.QCoreApplication.translate("convertSubdivisionsSubMenu", "If not possible"), self.errorHandling)

            #self.setFocus(); #self.grabKeyboard(); #redundant for a proper modal dialog. Leave here for documentation reasons.

            buttonBox = QtWidgets.QDialogButtonBox(QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel)
            buttonBox.accepted.connect(self.accept)
            buttonBox.rejected.connect(self.reject)
            self.layout.addWidget(buttonBox)

        def __call__(self):
            """This instance can be called like a function"""
            self.exec() #blocks until the dialog gets closed

    s = Submenu(mainWindow)
    s()
    if s.result():
        value = s.numberOfSubdivisions.value()
        error= ("fail", "delete", "merge")[s.errorHandling.currentIndex()]
        api.convert_subdivisions(value, error)


def globalOffsetSubmenu(mainWindow):
    class Submenu(QtWidgets.QDialog):
        def __init__(self, mainWindow):
            super().__init__(mainWindow) #if you don't set the parent to the main window the whole screen will be the root and the dialog pops up in the middle of it.
            #self.setModal(True) #we don't need this when called with self.exec() instead of self.show()


            self.layout = QtWidgets.QFormLayout()
            self.setLayout(self.layout)
            quarterticks = D4
            offsetMeasures, offsetTicks, completeOffsetinTicks = api.getGlobalOffset()

            explanation = QtCore.QCoreApplication.translate("globalOffsetSubmenu", "Current Offset in Ticks: {}\n\nFor advanced users who are using multiple music programs with JackTransport-Sync.\n\nHere you can set a global rhythm offset to let Patroneo 'wait' before it's playback.\nThe shift will be invisible in the GUI.\n\nFull Measures are current Patroneo measures. If you change their duration the offset will change as well.\n\nThe 'Ticks' value is absolute. A quarter note has {} ticks.\n\nYou can give even give negative numbers, but it impossible to shift before position 0. As an advanced user this is your responsibility :)")
            explanationLabel = QtWidgets.QLabel(explanation.format(completeOffsetinTicks, quarterticks))
            explanationLabel.setWordWrap(True)
            self.layout.addRow(explanationLabel) #only one parameter will span both columns

            self.offsetMeasures = QtWidgets.QSpinBox()
            self.offsetMeasures.setMinimum(-1 * MAX_QT_SIZE)
            self.offsetMeasures.setMaximum(MAX_QT_SIZE)
            self.offsetMeasures.setValue(offsetMeasures)

            self.offsetTicks = QtWidgets.QSpinBox()
            self.offsetTicks.setMinimum(-1 * MAX_QT_SIZE)
            self.offsetTicks.setMaximum(MAX_QT_SIZE)
            self.offsetTicks.setValue(offsetTicks)


            self.layout.addRow(QtCore.QCoreApplication.translate("globalOffsetSubmenu", "Full Patroneo Measures"), self.offsetMeasures)
            self.layout.addRow(QtCore.QCoreApplication.translate("globalOffsetSubmenu", "Ticks (Quarter={})").format(quarterticks), self.offsetTicks)

            #self.setFocus(); #self.grabKeyboard(); #redundant for a proper modal dialog. Leave here for documentation reasons.

            buttonBox = QtWidgets.QDialogButtonBox(QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel)
            buttonBox.accepted.connect(self.accept)
            buttonBox.rejected.connect(self.reject)
            self.layout.addWidget(buttonBox)


        def __call__(self):
            """This instance can be called like a function"""
            self.exec() #blocks until the dialog gets closed

    s = Submenu(mainWindow)
    s()
    if s.result():
        api.setGlobalOffset(s.offsetMeasures.value(), s.offsetTicks.value())
