/*
 * $Id: procstats.c,v 1.1 2001/03/15 22:16:13 jpormann Exp jpormann $
 *
 * procstatd - Copyright (c) 1999 by Robert G. Brown, rgb@phy.duke.edu
 *         GPL version 2b (b for beverage) granted.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * procstatd - A daemon to extract statistics from /proc/stat and publish them
 *         on demand via a socket connection or broadcast.
 */

#include "procstatd.h"

/*
 * procstats routines are simple shells that initialize, retrieve, process,
 * and transmit statistics derived from /proc/* or various systems calls.
 * This program component contains none of the actual code for handling a
 * particular statistics family.  All code for a particular statistic can be
 * found in e.g. proc_stat, proc_meminfo, etc.  This should make it 
 * RELATIVELY easy to add, for example, a new statistic or information on 
 * running processes to the bundle of things being transmitted because one 
 * can work on the code associated with each statistic in isolation
 * following a fairly simple presentation format) without touching or 
 * breaking the working routines -- only small, obvious changes to 
 * the subroutines below should be necessary to add a new statistic.
 *
 * We can thus add a statistic as an "object", sort of, although homie 
 * don't do c++.  Adding a statistic ALSO should not break applications
 * that talk to the daemons or read the broadcast information, if they
 * simply parse the information they get and ignore either extra fields
 * or missing fields.
 *
 * Note, the API of the data portion of the transmitted packet(s), such as 
 * it is, is basically defined below.  Eventually it will be codified and
 * described in detail in README.API, but not until we are at least in
 * some sort of beta...
 */


/*
 * init_statlist goes through all the sources listed in Sources,
 * determines which of these sources are available, and verifies that
 * the data therein can be extracted and stored in the current/previous
 * vectors.
 */

void init_statlist()
{

 int i,numfields;

 /* 
  * Open each /proc source file and leave it open.  The test of whether
  * or not the open succeeded wraps each attempted access, so don't
  * worry that this looks like there is no error checking.
  */

 /* PROC_STAT */
 init_proc_stat();
 /* PROC_LOADAVG */
 init_proc_loadavg();
 /* PROC_MEMINFO */
 init_proc_meminfo();
 /* PROC_NETDEV */
 init_proc_net_dev();
 /* PROC_SENSORS */
 init_proc_sensors();
 /* PROC_UPTIME */
 init_proc_uptime();
 /* TIME */
 init_time();
 /* USERS */
 init_users();

 /* 
  * SYS_DATA
  *
  * These things are evaluated "by hand" and not from stuff directly in
  * proc files.  It is very useful stuff nonetheless.  We simply add each
  * item and evaluate it as we go.  I believe that all are "raw" fields.
  */

} /* End init_statlist() */


void get_statlist()
{

 int i,numfields;
 static int cur_time,prev_time;

 /* 
  * Set the global interval for use in evaluating rates below. 
  */
 cur_time = (long) time(NULL);
 interval=(cur_time - prev_time);
 prev_time = cur_time;

 /* Get PROC_STAT stats */
 get_proc_stat();
 /* Get PROC_LOADAVG stats */
 get_proc_loadavg();
 /* Get PROC_MEMINFO stats */
 get_proc_meminfo();
 /* Get PROC_NET_DEV stats */
 get_proc_net_dev();
 /* Get PROC_SENSORS stats */
 get_proc_sensors();
 /* Get PROC_UPTIME stats */
 get_proc_uptime();
 /* Get TIME stats */
 get_time();
 /* Get USERS stats */
 get_users();

 /* 
  * SYS_DATA
  *
  * These things are evaluated "by hand" and not from stuff directly in
  * proc files.  It is very useful stuff nonetheless.  We simply add each
  * item and evaluate it as we go.  I believe that all are "raw" fields.
  */

} /* End get_statlist */

/*
 * get_statlist() retrieves and parses out all values for current/previous.
 * eval_statlist derives and sets the rates, depending on just how much
 * time elapsed between current/previous.
 */

void eval_statlist()
{

 int i,numfields;

 /* Evaluate all the statistics in /proc/stat */
 eval_proc_stat();
 /* Evaluate all the statistics in /proc/loadavg */
 eval_proc_loadavg();
 /* Evaluate all the statistics in /proc/meminfo */
 eval_proc_meminfo();
 /* Evaluate all the statistics in /proc/net_dev */
 eval_proc_net_dev();
 /* Evaluate all the statistics in /proc/sensors */
 eval_proc_sensors();
 /* Evaluate all the statistics in /proc/uptime */
 eval_proc_uptime();
 /* Evaluate the node/client time */
 eval_time();
 /* Evaluate the node/client number of users */
 eval_users();


 for(i=0;i<N_STATS;i++){
   switch(i) {
     /* CPU */
     case CPU:
     case CPU_USER:
     case CPU_NICE:
     case CPU_SYS:
     case CPU_IDLE:
       stats[i].rate=100.0*(stats[i].current-stats[i].previous)/
                    (stats[CPU].current + stats[CPU_IDLE].current
                      - stats[CPU].previous - stats[CPU_IDLE].previous);
       if(stats[i].rate > 100.0) stats[i].rate = 100.0;
       if(stats[i].rate < 0.0) stats[i].rate = 0.0;
       break;
     /* CPU0 */
     case CPU0:
     case CPU0_USER:
     case CPU0_NICE:
     case CPU0_SYS:
     case CPU0_IDLE:
       stats[i].rate=100.0*(stats[i].current-stats[i].previous)/
                    (stats[CPU0].current + stats[CPU0_IDLE].current
                      - stats[CPU0].previous - stats[CPU0_IDLE].previous);
       if(stats[i].rate > 100.0) stats[i].rate = 100.0;
       if(stats[i].rate < 0.0) stats[i].rate = 0.0;
       break;
     /* CPU1 */
     case CPU1:
     case CPU1_USER:
     case CPU1_NICE:
     case CPU1_SYS:
     case CPU1_IDLE:
       stats[i].rate=100.0*(stats[i].current-stats[i].previous)/
                    (stats[CPU1].current + stats[CPU1_IDLE].current
                      - stats[CPU1].previous - stats[CPU1_IDLE].previous);
       if(stats[i].rate > 100.0) stats[i].rate = 100.0;
       if(stats[i].rate < 0.0) stats[i].rate = 0.0;
       break;
     /* CPU2 */
     case CPU2:
     case CPU2_USER:
     case CPU2_NICE:
     case CPU2_SYS:
     case CPU2_IDLE:
       stats[i].rate=100.0*(stats[i].current-stats[i].previous)/
                    (stats[CPU2].current + stats[CPU2_IDLE].current
                      - stats[CPU2].previous - stats[CPU2_IDLE].previous);
       if(stats[i].rate > 100.0) stats[i].rate = 100.0;
       if(stats[i].rate < 0.0) stats[i].rate = 0.0;
       break;
     /* CPU3 */
     case CPU3:
     case CPU3_USER:
     case CPU3_NICE:
     case CPU3_SYS:
     case CPU3_IDLE:
       stats[i].rate=100.0*(stats[i].current-stats[i].previous)/
                    (stats[CPU3].current + stats[CPU3_IDLE].current
                      - stats[CPU3].previous - stats[CPU3_IDLE].previous);
       if(stats[i].rate > 100.0) stats[i].rate = 100.0;
       if(stats[i].rate < 0.0) stats[i].rate = 0.0;
       break;
     /* 
      * LOAD, PROC, MEM_USER, MEM_FREE, MEM_BUFF, MEM_CACHE, MEM_SWAP
      * USERS, TIME
      * are all straight values, not rates.
      */
     case LOAD1:
     case LOAD5:
     case LOAD15:
     case PROC:
     case MEM_TOTAL:
     case MEM_USED:
     case MEM_FREE:
     case MEM_SHARED:
     case MEM_BUFF:
     case MEM_CACHE:
     case MEM_SWAP_TOTAL:
     case MEM_SWAP_USED:
     case MEM_SWAP_FREE:
     case TEMP:
     case USERS:
     case TIME:
     case UPTIME:
     case SHM_NUM:
     case SHM_TOT:
     case SEM_NUM:
     case SEM_TOT:
     case MSG_NUM:
     case MSG_TOT:
       stats[i].rate = stats[i].current;
       if(stats[i].rate < 0.0) stats[i].rate = 0.0;
       break;
     /* 
      * DISK, PAGE, PAGE_IN, PAGE_OUT, SWAP, SWAP_IN, SWAP_OUT, INTR, CONTEXT
      * and ETH[0,1,2,3] are all rates handled by this default rule.
      */
     default:
       stats[i].rate = (stats[i].current - stats[i].previous)/interval;
       if(stats[i].rate < 0.0) stats[i].rate = 0.0;
       break;
   } /* End case switch */
 } /* End loop through enumerated types */

} /* End eval_statlist */


/* 
 * send_statlist(buffer) prints (as in inetd), transmits (as in
 * the forking daemon), or broadcast the statistics evaluated with 
 * init_statlist, get_statlist, and eval_statlist, depending on how the
 * daemon is invoked.
 */

void print_statlist(fd_out)
FILE *fd_out;
{

 int i,numfields;

 fprintf(stdout,"#========================================================================\n");
 fprintf(stdout,"# Report on derived values from /proc\n");
 fprintf(stdout,"# interval = %d seconds\n",interval);
 fprintf(stdout,"#========================================================================\n");
 for(i=0;i<N_STATS;i++){
   if(stats[i].avail) fprintf(stdout,"|%s|%.2f",stats[i].name,stats[i].rate);
 }
 fprintf(stdout,"|\n");

} /* End eval_statlist */

void send_statlist()
{

 int i;
 char my_sendbuf[64];

 if(verbose) printf("#====== list =======\n");
 sprintf(outbuf,"%s|%.2f",stats[0].name,stats[0].rate);
 for(i=1;i<N_STATS;i++){
   if(stats[i].avail){
     sprintf(my_sendbuf,"|%s|%.2f",stats[i].name,stats[i].rate);
     strncat(outbuf,my_sendbuf,strlen(my_sendbuf));
   }
 }
 strncat(outbuf,"|\n",2);
 buflen = strlen(outbuf);
 sendline(client_fd,outbuf,buflen);

} /* End eval_statlist */

