// ----------------------------------------------------------------------------
//
//  Copyright (C) 2018-2020 Fons Adriaensen <fons@linuxaudio.org>
//    
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// ----------------------------------------------------------------------------


#include <Python.h>
#include "jlr4filt.h"


static const char *capslabel = "Jlr4filt";


extern "C" void destroy (PyObject *P)
{
    delete (Jlr4filt *) PyCapsule_GetPointer (P, capslabel);
}


extern "C" PyObject* makecaps (PyObject *self, PyObject *args)
{
    Jlr4filt *J;
    PyObject *P;
    const char *client_name;
    const char *server_name;
    int ninp, nout;
    float maxdel;

    if (! PyArg_ParseTuple(args, "Osziif", &P, &client_name, &server_name,
			   &ninp, &nout, &maxdel)) return NULL;
    J = new Jlr4filt (client_name, server_name, ninp, nout, maxdel);
    return Py_BuildValue ("NN",
	 		  PyCapsule_New (J, capslabel, destroy),
                          PyCapsule_New (dynamic_cast<Jclient *>(J), "Jclient", 0));
}


extern "C" PyObject* set_filter (PyObject *self, PyObject *args)
{
    Jlr4filt  *J;
    PyObject  *P;
    int       inp, out;
    float     hpf, hps, lpf, lps;  

    if (! PyArg_ParseTuple(args, "Oiiffff", &P, &inp, &out, &hpf, &hps, &lpf, &lps)) return NULL;
    J = (Jlr4filt *) PyCapsule_GetPointer (P, capslabel);
    J->set_filter (inp, out, hpf, hps, lpf, lps);
    Py_RETURN_NONE;
}


extern "C" PyObject* set_delay (PyObject *self, PyObject *args)
{
    Jlr4filt  *J;
    PyObject  *P;
    int       out;
    float     delay;

    if (! PyArg_ParseTuple(args, "Oif", &P, &out, &delay)) return NULL;
    J = (Jlr4filt *) PyCapsule_GetPointer (P, capslabel);
    J->set_delay (out, delay);
    Py_RETURN_NONE;
}


extern "C" PyObject* set_gain (PyObject *self, PyObject *args)
{
    Jlr4filt  *J;
    PyObject  *P;
    int       out;
    float     gain;

    if (! PyArg_ParseTuple(args, "Oif", &P, &out, &gain)) return NULL;
    J = (Jlr4filt *) PyCapsule_GetPointer (P, capslabel);
    J->set_gain (out, gain);
    Py_RETURN_NONE;
}


static PyMethodDef Jacklr4filtMethods[] =
{ 
    {"makecaps",     makecaps,    METH_VARARGS, "Create object capsules."},
    {"set_filter",   set_filter,  METH_VARARGS, "Set input and filter for output."},
    {"set_delay",    set_delay,   METH_VARARGS, "Set output delay."},
    {"set_gain",     set_gain,    METH_VARARGS, "Set output gain."},
    {NULL, NULL, 0, NULL}
};



#if PY_VERSION_HEX >= 0x03000000

static struct PyModuleDef Jacklr4filtModule = 
{
   PyModuleDef_HEAD_INIT,
   "jacklr4filt_ext",
   NULL, 
   -1, 
   Jacklr4filtMethods
};

PyMODINIT_FUNC PyInit_jacklr4filt_ext(void)
{
    return PyModule_Create(&Jacklr4filtModule);
}

#else

PyMODINIT_FUNC initjacklr4filt_ext(void)
{
    (void) Py_InitModule("jacklr4filt_ext", Jacklr4filtMethods);
}

#endif
