/*
   Copyright (C) 2014-2015 Alexandr Akulich <akulichalexander@gmail.com>

   This file is a part of TelegramQt library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

 */

#include <QObject>

#include "CTestConnection.hpp"
#include "CTelegramTransport.hpp"

#include <QTest>
#include <QDebug>

#include <QDateTime>

class tst_CTelegramConnection : public QObject
{
    Q_OBJECT
public:
    explicit tst_CTelegramConnection(QObject *parent = 0);

private slots:
    void testTimestampAlwaysGrow();
    void testNewMessageId();
    void testClientTimestampNeverOdd();
    void testTimestampConversion();
    void testPQAuthRequest();
    void testAuth();
    void testAesKeyGeneration();

};

tst_CTelegramConnection::tst_CTelegramConnection(QObject *parent) :
    QObject(parent)
{
}

void tst_CTelegramConnection::testTimestampAlwaysGrow()
{
    const quint64 time = 1395335796550;

    quint64 previousTimeStamp = CTelegramConnection::formatClientTimeStamp(time - 1);
    for (int i = 0; i < 2000; ++i) {
        const quint64 newTimeStamp = CTelegramConnection::formatClientTimeStamp(time + i);
        if (newTimeStamp < previousTimeStamp) {
            // Print erroneous method results and arguments.
            qDebug() << "previous:" << previousTimeStamp << time + i - 1;
            qDebug() << "new:" << newTimeStamp << time + i;
            qDebug() << "iteration:" << i;
            // Make same call to erroneous CTelegramCore::formatClientTimeStamp() to simplify debugging via break point.
            const quint64 timeStamp = CTelegramConnection::formatClientTimeStamp(time + i);
            Q_UNUSED(timeStamp)
        }
        QVERIFY2(newTimeStamp > previousTimeStamp, "New timestamp should be more or equal than previous");
        previousTimeStamp = newTimeStamp;
    }
}

void tst_CTelegramConnection::testNewMessageId()
{
    CTestConnection connection;

    quint64 previousTimeStamp = 0;
    for (int i = 0; i < 2000; ++i) {
        const quint64 newTimeStamp = connection.testNewMessageId();
        if (newTimeStamp < previousTimeStamp) {
            // Print erroneous method results and arguments.
            qDebug() << "previous:" << previousTimeStamp;
            qDebug() << "new:" << newTimeStamp;
            qDebug() << "iteration:" << i;
        }
        QVERIFY2(newTimeStamp > previousTimeStamp, "New timestamp should be more or equal than previous");
        previousTimeStamp = newTimeStamp;
    }
}

void tst_CTelegramConnection::testClientTimestampNeverOdd()
{
    quint64 time = 1395335796550;

    for (int i = 0; i < 2000; ++i) {
        if (CTelegramConnection::formatClientTimeStamp(time + i) & 3) {
            QVERIFY(!(CTelegramConnection::formatClientTimeStamp(time + i) & 3));
            break;
        }
    }
}

void tst_CTelegramConnection::testTimestampConversion()
{
    quint64 time = 1395335796550;
    quint64 ts = CTelegramConnection::formatTimeStamp(time);
    QCOMPARE(CTelegramConnection::timeStampToMSecsSinceEpoch(ts), time);

    ts = 0x532ea31d36cecc00;
    time = 1395565341214;

    QCOMPARE(CTelegramConnection::timeStampToMSecsSinceEpoch(ts), time);
}

void tst_CTelegramConnection::testPQAuthRequest()
{
    CTestConnection connection;
    connection.requestPqAuthorization();

    QByteArray encoded = connection.transport()->lastPackage();

    QVERIFY2(encoded.at(0) == char(0xef), "Abridged version marker");
    QCOMPARE(encoded.at(1), char(0x0a)); // Package length information should be equal to 0x0a (real size / 4)
    QCOMPARE(encoded.length(), 0x0a * 4 + 2); // Package size should be equal to 0x0a * 4 + 2 (2 is size of marker+size_info)
    QVERIFY2(encoded.mid(2, 8) == QByteArray(8, char(0)), "In this method auth id should be equal zero");

    QByteArray messageBodyLength(4, char(0));
    messageBodyLength[0] = 0x14;

    QCOMPARE(encoded.mid(18, 4), messageBodyLength); // Expected payload length is 20 bytes

    QByteArray reqPqRaw;
    reqPqRaw.resize(4);
    reqPqRaw[0] = 0x78;
    reqPqRaw[1] = 0x97;
    reqPqRaw[2] = 0x46;
    reqPqRaw[3] = 0x60;

    QCOMPARE(encoded.mid(22, 4), reqPqRaw); // Expected payload length is 20 bytes
}

void tst_CTelegramConnection::testAuth()
{
    CTestConnection core;

    unsigned char clientNonceData[16] = {
        0xda, 0xc1, 0xe2, 0xf1, 0xbf, 0x26, 0x33, 0x26,
        0xa9, 0x18, 0x11, 0xe0, 0x9a, 0xf5, 0x5d, 0xe8
    };

    unsigned char serverNonceData[16] = {
        0x77, 0x6b, 0x31, 0xbf, 0xe4, 0x42, 0xb2, 0x43,
        0x22, 0xb6, 0x79, 0x16, 0xc1, 0x9a, 0x1c, 0x54
    };

    TLNumber128 nonce;
    for (int i = 0; i < nonce.size(); ++i) {
        nonce.data[i] = clientNonceData[i];
    }

    core.setClientNonce(nonce);

    for (int i = 0; i < nonce.size(); ++i) {
        nonce.data[i] = serverNonceData[i];
    }

    core.setServerNonce(nonce);

    TLNumber256 longNumber;

    unsigned char newNonceData[32] = {
        0xbc, 0xe9, 0xc5, 0x63, 0x4d, 0xaa, 0x17, 0xd4,
        0x28, 0x06, 0xd9, 0xb1, 0x3e, 0xe4, 0xd9, 0xec,
        0xc4, 0x37, 0x1e, 0xf1, 0x11, 0x2a, 0xad, 0x5f,
        0xc7, 0x0d, 0x4a, 0xcb, 0x2b, 0x9a, 0x78, 0xc1
    };

    for (int i = 0; i < longNumber.size(); ++i) {
        longNumber.data[i] = newNonceData[i];
    }

    core.setNewNonce(longNumber);

    unsigned char gA[256] = {
        0x40, 0x36, 0xe0, 0xd8, 0xdf, 0x15, 0x46, 0x7c,
        0xee, 0x89, 0x35, 0x07, 0x29, 0xe7, 0x11, 0x94,
        0x40, 0xb3, 0xe3, 0x08, 0x56, 0xd0, 0x6a, 0x70,
        0x70, 0x07, 0x79, 0x61, 0x3b, 0xd7, 0x45, 0xaa,
        0xa2, 0x5d, 0x51, 0x58, 0xa9, 0x7e, 0x27, 0x59,
        0xab, 0x20, 0x5d, 0xd3, 0x07, 0x73, 0x29, 0x99,
        0x91, 0xe2, 0xcb, 0x88, 0x42, 0x19, 0x6f, 0xcc,
        0x02, 0x2f, 0x3f, 0x93, 0xbb, 0x47, 0x64, 0xa4,
        0xc7, 0xe2, 0xa6, 0x3d, 0x43, 0x7a, 0xbd, 0x85,
        0xcf, 0x6d, 0x7f, 0x64, 0x0c, 0xc6, 0x08, 0x04,
        0x6c, 0x39, 0xaf, 0x5c, 0x29, 0xd2, 0x13, 0xa2,
        0x56, 0x3b, 0x35, 0x0c, 0x7f, 0x66, 0x5c, 0x49,
        0x22, 0x58, 0x3b, 0xe0, 0xe9, 0x77, 0x05, 0xa6,
        0xd9, 0xc8, 0x7b, 0xdb, 0x75, 0x23, 0xd3, 0xa7,
        0x51, 0x51, 0xb1, 0x4b, 0x9f, 0x9b, 0x68, 0x4b,
        0x10, 0xbd, 0x5d, 0x99, 0xbd, 0x9c, 0xd3, 0x54,
        0xaa, 0x48, 0x25, 0xdd, 0x41, 0x3d, 0x36, 0x63,
        0x81, 0xe5, 0x2f, 0x85, 0x76, 0x6c, 0xe0, 0xc1,
        0xd4, 0x1e, 0xf1, 0x72, 0xac, 0xfc, 0x0c, 0x5b,
        0xd9, 0xe8, 0x44, 0x9c, 0x0e, 0x4a, 0x98, 0xe5,
        0x9d, 0xb7, 0xe8, 0xc3, 0xd2, 0xc2, 0x8d, 0xc3,
        0x16, 0xf4, 0x78, 0x87, 0xbd, 0xd4, 0xea, 0xec,
        0xfc, 0x53, 0xc0, 0x22, 0x20, 0x0a, 0x0a, 0xd3,
        0x0b, 0xc6, 0x8a, 0xdb, 0x3f, 0xb4, 0x81, 0x67,
        0xd7, 0x67, 0xe6, 0xf3, 0x95, 0x0b, 0x2e, 0x97,
        0x2a, 0xaa, 0x3e, 0xfc, 0x2f, 0x83, 0xb2, 0x77,
        0x01, 0x10, 0x96, 0xbb, 0xce, 0x8d, 0x45, 0xe8,
        0x75, 0xd7, 0x98, 0x61, 0x61, 0x18, 0x93, 0x25,
        0x79, 0x2a, 0x12, 0xb7, 0xe0, 0x12, 0xef, 0x60,
        0xf7, 0x09, 0x3a, 0x02, 0x5c, 0x08, 0x5e, 0x83,
        0x4b, 0x38, 0x29, 0xc9, 0x35, 0xcd, 0x74, 0xce,
        0xca, 0x12, 0xa3, 0x2d, 0xde, 0x82, 0x00, 0x58
    };

    unsigned char prime[256] = {
        0xc7, 0x1c, 0xae, 0xb9, 0xc6, 0xb1, 0xc9, 0x04,
        0x8e, 0x6c, 0x52, 0x2f, 0x70, 0xf1, 0x3f, 0x73,
        0x98, 0x0d, 0x40, 0x23, 0x8e, 0x3e, 0x21, 0xc1,
        0x49, 0x34, 0xd0, 0x37, 0x56, 0x3d, 0x93, 0x0f,
        0x48, 0x19, 0x8a, 0x0a, 0xa7, 0xc1, 0x40, 0x58,
        0x22, 0x94, 0x93, 0xd2, 0x25, 0x30, 0xf4, 0xdb,
        0xfa, 0x33, 0x6f, 0x6e, 0x0a, 0xc9, 0x25, 0x13,
        0x95, 0x43, 0xae, 0xd4, 0x4c, 0xce, 0x7c, 0x37,
        0x20, 0xfd, 0x51, 0xf6, 0x94, 0x58, 0x70, 0x5a,
        0xc6, 0x8c, 0xd4, 0xfe, 0x6b, 0x6b, 0x13, 0xab,
        0xdc, 0x97, 0x46, 0x51, 0x29, 0x69, 0x32, 0x84,
        0x54, 0xf1, 0x8f, 0xaf, 0x8c, 0x59, 0x5f, 0x64,
        0x24, 0x77, 0xfe, 0x96, 0xbb, 0x2a, 0x94, 0x1d,
        0x5b, 0xcd, 0x1d, 0x4a, 0xc8, 0xcc, 0x49, 0x88,
        0x07, 0x08, 0xfa, 0x9b, 0x37, 0x8e, 0x3c, 0x4f,
        0x3a, 0x90, 0x60, 0xbe, 0xe6, 0x7c, 0xf9, 0xa4,
        0xa4, 0xa6, 0x95, 0x81, 0x10, 0x51, 0x90, 0x7e,
        0x16, 0x27, 0x53, 0xb5, 0x6b, 0x0f, 0x6b, 0x41,
        0x0d, 0xba, 0x74, 0xd8, 0xa8, 0x4b, 0x2a, 0x14,
        0xb3, 0x14, 0x4e, 0x0e, 0xf1, 0x28, 0x47, 0x54,
        0xfd, 0x17, 0xed, 0x95, 0x0d, 0x59, 0x65, 0xb4,
        0xb9, 0xdd, 0x46, 0x58, 0x2d, 0xb1, 0x17, 0x8d,
        0x16, 0x9c, 0x6b, 0xc4, 0x65, 0xb0, 0xd6, 0xff,
        0x9c, 0xa3, 0x92, 0x8f, 0xef, 0x5b, 0x9a, 0xe4,
        0xe4, 0x18, 0xfc, 0x15, 0xe8, 0x3e, 0xbe, 0xa0,
        0xf8, 0x7f, 0xa9, 0xff, 0x5e, 0xed, 0x70, 0x05,
        0x0d, 0xed, 0x28, 0x49, 0xf4, 0x7b, 0xf9, 0x59,
        0xd9, 0x56, 0x85, 0x0c, 0xe9, 0x29, 0x85, 0x1f,
        0x0d, 0x81, 0x15, 0xf6, 0x35, 0xb1, 0x05, 0xee,
        0x2e, 0x4e, 0x15, 0xd0, 0x4b, 0x24, 0x54, 0xbf,
        0x6f, 0x4f, 0xad, 0xf0, 0x34, 0xb1, 0x04, 0x03,
        0x11, 0x9c, 0xd8, 0xe3, 0xb9, 0x2f, 0xcc, 0x5b
    };

    unsigned char b[256] = {
        0x81, 0xa6, 0x94, 0xcf, 0x25, 0xfa, 0x58, 0x9f,
        0x9a, 0x4a, 0xff, 0x1c, 0xa3, 0x50, 0x69, 0xf9,
        0xc1, 0x5c, 0xac, 0xee, 0x88, 0x70, 0x80, 0x08,
        0x3f, 0x4f, 0x83, 0xcc, 0x7d, 0xf9, 0xfa, 0x93,
        0xc6, 0x3d, 0xb8, 0xa9, 0x4a, 0xf0, 0x72, 0xcc,
        0xc8, 0x8e, 0xe5, 0xa9, 0x6e, 0xff, 0xe4, 0x1f,
        0x42, 0xd9, 0x73, 0x43, 0xf8, 0x13, 0x5f, 0xba,
        0xab, 0x00, 0x94, 0xfa, 0x5d, 0xce, 0x42, 0x22,
        0x02, 0x56, 0x7b, 0xe8, 0xdc, 0x51, 0x30, 0x37,
        0x5e, 0x12, 0x2e, 0x3a, 0xa2, 0xae, 0x7e, 0x97,
        0x2e, 0x8e, 0x80, 0x08, 0x45, 0x68, 0x3a, 0xac,
        0x62, 0x49, 0xdb, 0x53, 0xba, 0x7c, 0x46, 0xe8,
        0x1d, 0x74, 0xb9, 0xb6, 0x55, 0xd5, 0x13, 0x52,
        0x2f, 0xb7, 0x63, 0x56, 0x29, 0xc4, 0x13, 0xa2,
        0xdc, 0x91, 0x71, 0xf9, 0x7a, 0x2a, 0xf0, 0x69,
        0xe1, 0xb1, 0x24, 0x02, 0x8a, 0x94, 0x9d, 0xc2,
        0xce, 0xc4, 0x10, 0x01, 0x99, 0xb2, 0x5e, 0x31,
        0xa4, 0xe3, 0x6c, 0xa5, 0xd9, 0xbe, 0x3e, 0xb1,
        0xe2, 0xf3, 0x53, 0xe9, 0x67, 0x01, 0x33, 0xff,
        0x2c, 0x45, 0xc2, 0xae, 0x87, 0x33, 0x0e, 0x86,
        0x00, 0x53, 0x75, 0x01, 0x51, 0x47, 0x62, 0x09,
        0x43, 0x77, 0x05, 0xf5, 0x3a, 0xe2, 0xf6, 0xd7,
        0xc7, 0x4f, 0xd6, 0x36, 0x5b, 0x78, 0x16, 0x12,
        0xaa, 0x3a, 0xdf, 0xa8, 0x82, 0xaa, 0x38, 0xc8,
        0x43, 0x29, 0x1d, 0xfb, 0x6a, 0x54, 0xdc, 0x72,
        0x00, 0x57, 0x2c, 0x5c, 0x70, 0xb9, 0xed, 0xfc,
        0x6f, 0x69, 0xf2, 0x19, 0x85, 0x64, 0x93, 0x0b,
        0xbf, 0x26, 0xf9, 0x0b, 0xb6, 0x5a, 0xb6, 0x36,
        0x5a, 0x93, 0x3a, 0x35, 0x0f, 0x71, 0x9e, 0x33,
        0x16, 0x58, 0x72, 0x8c, 0x0d, 0x46, 0x1b, 0x2f,
        0x19, 0xa0, 0xd2, 0x49, 0xd6, 0x4e, 0x0f, 0xea,
        0x20, 0x6e, 0x3a, 0x72, 0x26, 0xdb, 0x53, 0x03
    };

    QByteArray gAArray;
    gAArray.append((char *) gA, 256);
    core.setGA(gAArray);

    QByteArray primeArray;
    primeArray.append((char *) prime, 256);
    core.setPrime(primeArray);

    QByteArray bArray;
    bArray.append((char *) b, 256);
    core.setB(bArray);

    QByteArray payload;

    unsigned char payloadData[52] = {
        0x34, 0xf7, 0xcb, 0x3b, 0xda, 0xc1, 0xe2, 0xf1,
        0xbf, 0x26, 0x33, 0x26, 0xa9, 0x18, 0x11, 0xe0,
        0x9a, 0xf5, 0x5d, 0xe8, 0x77, 0x6b, 0x31, 0xbf,
        0xe4, 0x42, 0xb2, 0x43, 0x22, 0xb6, 0x79, 0x16,
        0xc1, 0x9a, 0x1c, 0x54, 0x17, 0xeb, 0x64, 0xf8,
        0x58, 0xb7, 0x2d, 0x91, 0x6f, 0xe4, 0xaa, 0x7c,
        0x31, 0x12, 0x3f, 0xb1
    };

    payload.append((char *) payloadData, 256);

    QVERIFY(core.processServersDHAnswer(payload));
}

void tst_CTelegramConnection::testAesKeyGeneration()
{
    unsigned char auth[192] = {
        0x26, 0x40, 0xa5, 0xb0, 0x79, 0x4a, 0x7b, 0xed,
        0x84, 0x0e, 0x0b, 0xf8, 0x48, 0x0c, 0x17, 0x4b,
        0x1e, 0xde, 0xf5, 0x29, 0x21, 0x45, 0xf2, 0xbe,
        0x19, 0xa2, 0x63, 0xeb, 0x8e, 0xfb, 0x67, 0x19,
        0x06, 0x78, 0x6f, 0x72, 0xc2, 0x1d, 0x82, 0x16,
        0x3b, 0x7b, 0xad, 0xbb, 0x0f, 0x5b, 0xa7, 0xf3,
        0x53, 0xfa, 0xcb, 0xc2, 0x9e, 0xc0, 0xb1, 0xdd,
        0x9f, 0x61, 0x8c, 0x9c, 0x77, 0x57, 0x5b, 0x7f,
        0xd4, 0x47, 0x05, 0x2a, 0x98, 0xcd, 0xe5, 0xea,
        0x4a, 0xb0, 0x12, 0x00, 0x23, 0x1e, 0xf1, 0x17,
        0x6a, 0x80, 0xee, 0x2e, 0x9c, 0xd5, 0x7d, 0xfd,
        0x29, 0x5e, 0x11, 0xe5, 0xde, 0xe4, 0x6b, 0xd4,
        0x8e, 0x94, 0x71, 0xe4, 0x48, 0xa8, 0xa3, 0x54,
        0x9f, 0xdf, 0x55, 0xb7, 0x44, 0xd2, 0xa9, 0xe4,
        0xc2, 0xbb, 0x09, 0x3b, 0xa5, 0x61, 0x58, 0xaa,
        0xa4, 0x57, 0x44, 0x67, 0x25, 0xd5, 0x80, 0xd2,
        0x34, 0x83, 0xfd, 0x65, 0x93, 0x8a, 0x3a, 0xad,
        0xe4, 0x66, 0x5b, 0x46, 0x08, 0xf3, 0xd1, 0x50,
        0x9c, 0x9d, 0xd7, 0x4f, 0x81, 0x99, 0xd1, 0xaf,
        0x1f, 0x0d, 0x79, 0xb7, 0xd0, 0xe2, 0x79, 0xb5,
        0x8a, 0xef, 0x6d, 0xc6, 0x5a, 0xc7, 0xce, 0x77,
        0x0d, 0x5e, 0xa7, 0xd1, 0x8b, 0x2f, 0x1a, 0x97,
        0xe3, 0x46, 0x34, 0x76, 0x9d, 0xe5, 0xb6, 0xe8,
        0x8e, 0xbd, 0x27, 0x17, 0x98, 0x43, 0xb7, 0x8c
    };

    unsigned char messageKey[16] = {
        0x15, 0x02, 0xfd, 0xa4, 0x01, 0xf5, 0xf2, 0x2d,
        0x69, 0x07, 0x20, 0xac, 0xea, 0x77, 0x53, 0xe9
    };

    unsigned char expectedKey[32] = {
        0x61, 0x72, 0x1b, 0x31, 0x06, 0x68, 0xdc, 0x80,
        0x23, 0x2b, 0x35, 0xbb, 0xfa, 0x50, 0x4a, 0x8a,
        0x69, 0x7f, 0xc8, 0x2b, 0x09, 0x68, 0x0f, 0x33,
        0x1d, 0xd4, 0x23, 0x9a, 0x94, 0x39, 0xf1, 0x66
    };

    unsigned char expectedIv[32] = {
        0x51, 0x54, 0xfb, 0x77, 0xac, 0x38, 0x95, 0x37,
        0x35, 0xf5, 0x6d, 0x45, 0x49, 0xb1, 0x34, 0x83,
        0x23, 0x8b, 0x42, 0x4d, 0x4e, 0x5f, 0x97, 0x00,
        0x17, 0x97, 0x42, 0xb7, 0xb3, 0x44, 0x6e, 0x7b
    };

    QByteArray authArray;
    authArray.append((char *) auth, 192);

    QByteArray messageKeyArray;
    messageKeyArray.append((char *) messageKey, 16);

    QByteArray aesKeyArray;
    aesKeyArray.append((char *) expectedKey, 32);

    QByteArray aesIvArray;
    aesIvArray.append((char *) expectedIv, 32);

    CTestConnection core;

    core.setAuthKey(authArray);

    SAesKey result = core.testGenerateClientToServerAesKey(messageKeyArray);

    QCOMPARE(result.key, aesKeyArray);
    QCOMPARE(result.iv , aesIvArray);
}

QTEST_MAIN(tst_CTelegramConnection)

#include "tst_CTelegramConnection.moc"
