#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2021, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
import logging; logger = logging.getLogger(__name__); logger.info("import")

#Template Modules
from template.helper import compress
import template.engine.api #we need direct access to the module to inject data in the provided structures. but we also need the functions directly. next line:
from template.engine.api import *
from template.engine.input_midi import MidiInput


#New callbacks
class ClientCallbacks(Callbacks): #inherits from the templates api callbacks
    def __init__(self):
        super().__init__()
        self.stepEntryNoteOn = []
        self.stepEntryNoteOff = []
        self.newEvent = []
        self.deleteEvent = []
        self.eventPositionChanged = []
        self.eventByteOneChanged = []
        self.eventByteTwoChanged = []
        self.eventFreeTextChanged = []
        self.activeLayerChanged = []
        self.layerChanged = []
        self.layerColorChanged = []
        self.layerMidiChannelChanged = []
        self.songDurationChanged = []
        self.pastedEvents = []

    def _stepEntryNoteOn(self, pitch, velocity):
        for func in self.stepEntryNoteOn:
            func(pitch, velocity)
        self._dataChanged()

    def _stepEntryNoteOff(self, pitch, velocity):
        for func in self.stepEntryNoteOff:
            func(pitch, velocity)
        self._dataChanged()

    def _newEvent(self, eventDict):
        """Incremental update for all other event """
        for func in self.newEvent:
            func(eventDict)
        self._dataChanged()

    def _deleteEvent(self, eventDict):
        for func in self.deleteEvent:
            func(eventDict)
        self._dataChanged()

    def _eventPositionChanged(self, eventDict):
        for func in self.eventPositionChanged:
            func(eventDict)
        self._dataChanged()

    def _eventByteOneChanged(self, eventDict):
        for func in self.eventByteOneChanged:
            func(eventDict)
        self._dataChanged()

    def _eventByteTwoChanged(self, eventDict):
        for func in self.eventByteTwoChanged:
            func(eventDict)
        self._dataChanged()

    def _eventFreeTextChanged(self, eventDict):
        for func in self.eventFreeTextChanged:
            func(eventDict)
        self._dataChanged()

    def _activeLayerChanged(self):
        """We order 0 before 1 but the keyboard layout trumps programming logic.
         First key is 1, therefore default layer is 1"""
        layer = session.data.track.layers[session.data.track.activeLayer]
        midiInput.setMidiThruChannel(layer.midiChannel)
        for func in self.activeLayerChanged:
            func(session.data.track.activeLayer)
        self._dataChanged()

    def _layerChanged(self, layerIndex):
        """Send a complete layer to redraw. Used 10x when loading a file or after
        heavy operations."""
        export = session.data.track.layers[layerIndex].export() #side-effect: cbox midi update
        for func in self.layerChanged:
            func(layerIndex, export)
        self._dataChanged()

    def _layerColorChanged(self, layerIndex):
        """while included in _layerChanged, this callback only changes the color"""
        export = session.data.track.layers[layerIndex].color
        for func in self.layerColorChanged:
            func(layerIndex, export)
        self._dataChanged()

    def _layerMidiChannelChanged(self, layerIndex):
        """while included in _layerChanged, this callback only changes the color"""
        layer = session.data.track.layers[session.data.track.activeLayer]
        if layerIndex == session.data.track.activeLayer:
            midiInput.setMidiThruChannel(layer.midiChannel)

        export = layer.midiChannel
        for func in self.layerMidiChannelChanged:
            func(layerIndex, export)
        self._dataChanged()

    def _songDurationChanged(self):
        """Intended to tell the GUI how far to draw their scene. Also includes the first event
        to make navigation more convenient."""
        last = session.data.track.lastEventPosition()
        first = session.data.track.firstEventPosition()
        for func in self.songDurationChanged:
            func(first, last)
        self._dataChanged()

    def _pastedEvents(self, listOfEventIdsAndLayers):
        """Inform which events just have been pasted so they can be separated from the original
        items which are still on the exact position (if not cut or paste to different layer"""
        for func in self.pastedEvents:
            func(listOfEventIdsAndLayers)

#Inject our derived Callbacks into the parent module
template.engine.api.callbacks = ClientCallbacks()
from template.engine.api import callbacks

_templateStartEngine = startEngine

midiInput = None #set in startEngine

def startEngine(nsmClient):
    logger.info("Starting main programs api engine")

    callbacks.playbackStatusChanged.append(toggleUndoRecordingCollector)
    callbacks.playbackStatusChanged.append(finalizeRecording) #this order means that finalizing happens always after toggleUndoRecordingCollector, which MUST be.

    callbacks.recordingModeChanged.append(checkIfPlaybackIsRunningWhileRecordinChanged)

    _templateStartEngine(nsmClient)

    #Setup Midi Input
    global midiInput
    midiInput = MidiInput(session, portName="in") #MidiInput is a standalone class that communicates purely with callbacks and injections.
    midiInput.setMidiThru(session.data.track.sequencerInterface.cboxMidiOutUuid)
    midiInput.midiProcessor.register_NoteOn(noteOn)
    midiInput.midiProcessor.register_NoteOff(noteOff)
    midiInput.midiProcessor.register_CC(cc)
    midiInput.midiProcessor.register_PolyphonicAftertouch(polyphonicAftertouch)
    midiInput.midiProcessor.register_ProgramChange(progamChange)
    midiInput.midiProcessor.register_ChannelPressure(channelPressure)
    midiInput.midiProcessor.register_PitchBend(pitchBend)

    #Send initial Data, and callbacks to create the first GUI state.
    for layerIndex in session.data.track.layers.keys():
        callbacks._layerColorChanged(layerIndex) #Send color BEFORE the rest so activeLayer below can already use the color
        callbacks._layerChanged(layerIndex)

    callbacks._songDurationChanged()
    callbacks._activeLayerChanged()
    callbacks._checkPlaybackStatusAndSendSignal() #already in template but that was before we registered our callbacks
    callbacks._recordingModeChanged() #already in template but that was before we registered our callbacks
    logger.info("Main programs api engine started")

def setLayerColor(layer:int, color:str):
    session.data.track.layers[layer].color = color
    callbacks._layerColorChanged(layer) #grabs the color itself

#Midi Input functions. Called by the midiInput module callbacks
def noteOn(tickindex:int, channel:int, note:int, velocity:int):
    if session.recordingEnabled and not tickindex is None:
        tickindex = int(tickindex)
        event = session.data.liveNoteOn(tickindex, note, velocity) #Forward to engine, whichs knows the activeLayer already.
        callbacks._newEvent(event.export())
        callbacks._songDurationChanged()
    else: #non-recording
        callbacks._stepEntryNoteOn(note, velocity)

def noteOff(tickindex:int, channel:int, note:int, velocity:int):
    if session.recordingEnabled and not tickindex is None:
        tickindex = int(tickindex)
        event = session.data.liveNoteOff(tickindex, note, velocity) #Forward to engine, whichs knows the activeLayer already.
        if event: #it is possible that we received a note-off without a previous note-on. Ignore.
            exp = event.export()
            callbacks._newEvent(exp)
            callbacks._songDurationChanged()
    else: #non-recording
        callbacks._stepEntryNoteOff(note, velocity)

def finalizeRecording(state:bool):
    """Called by api callback"""
    forcedNoteOffs = session.data.finalizeRecording(state) #Cbox already knows about the note offs but we need to tell the GUI.
    for noteOffEvent in forcedNoteOffs:
        callbacks._newEvent(noteOffEvent.export())

    if session.data.eventCollectorForUndo: #if either None or empty list will not generate new cbox data
        session.history.register(lambda l=session.data.eventCollectorForUndo: _deleteEvents(l), descriptionString="Record Events")
        session.data.eventCollectorForUndo = None
        session.data.track.generateCalfboxMidi() #atomic live note event callbacks are enough for the gui. And while it is playing the RT midi through is used. After recording we saved everything as proper notes but cbox still needs update.
        callbacks._songDurationChanged()

def cc(tickindex:int, channel:int, type:int, value:int):
    """0xC0"""
    if session.recordingEnabled and not tickindex is None:
        tickindex = int(tickindex)
        event = session.data.liveCC(tickindex, type, value) #Forward to engine, whichs knows the activeLayer already.
        callbacks._newEvent(event.export())
        callbacks._songDurationChanged()

def polyphonicAftertouch(tickindex:int, channel:int, note:int, value:int):
    """0xA0"""
    if session.recordingEnabled and not tickindex is None:
        tickindex = int(tickindex)
        event = session.data.livePolyphonicAftertouch(tickindex, note, value) #Forward to engine, whichs knows the activeLayer already.
        callbacks._newEvent(event.export())
        callbacks._songDurationChanged()

_lastPitchBendCoarse = None
def pitchBend(tickindex:int, channel:int, fine:int, coarse:int):
    """Pitchbend 0xE0 is a 14 bit value. Byte2 is coarse/MSB, byte1 is fine/LSB.
    Many keyboards leave byte1 as 0 and use only byte2, making PitchBend 128steps.

    There is also a CC to specify pitchbend musical range and sensitivity,
    but that is for a synth, we just send the CC without knowing its implications

    As a specific vico "feature" we always set the fine bit to 0, to have any chance
    to display it in a meaningful way in our Qt GUI.

    A special case. Pitch bend is 14bit but it is very hard to react to that many messages
    with a standard GUI or similar. We therefore discard the fine/LSB byte1 already here
    and only trigger when the coarse value is different from before.
    """
    if session.recordingEnabled and not tickindex is None:
        global _lastPitchBendCoarse
        if not _lastPitchBendCoarse == coarse:
            tickindex = int(tickindex)
            event = session.data.livePitchBend(tickindex, fine, coarse) #Forward to engine, whichs knows the activeLayer already.
            callbacks._newEvent(event.export())
            callbacks._songDurationChanged()
            _lastPitchBendCoarse = coarse


def progamChange(tickindex:int, channel:int, value:int):
    """0xB0
    Value is byte1. There is no byte2"""
    if session.recordingEnabled and not tickindex is None:
        tickindex = int(tickindex)
        event = session.data.liveProgramChange(tickindex, value) #Forward to engine, whichs knows the activeLayer already.
        callbacks._newEvent(event.export())
        callbacks._songDurationChanged()

def channelPressure(tickindex:int, channel:int, value:int):
    """0xD0
    Like Program Change, has only byte1. There is no byte2"""
    if session.recordingEnabled and not tickindex is None:
        tickindex = int(tickindex)
        event = session.data.liveChannelPressure(tickindex, value) #Forward to engine, whichs knows the activeLayer already.
        callbacks._newEvent(event.export())
        callbacks._songDurationChanged()

#General Purpose Functions

def _deleteEvents(listOfEvents):
    """complementary to insertEvents. Can be used for circular undo/redo.
    Linear efficiency. A list of one is quick, thus this is also "deleteEvent", singular.
    """
    if not listOfEvents:
        return

    session.history.register(lambda l=listOfEvents: _insertEvents(l), descriptionString="Delete Events")

    #changedLayers = set()

    for event in listOfEvents:
        #changedLayers.add(event.layer)
        session.data.track.deleteEvent(event)
        callbacks._deleteEvent(event.export())

    session.data.track.generateCalfboxMidi()
    callbacks._songDurationChanged()

def deleteEventsById(listOfEventIds):
    """We receive 2 events per note: on and off. All others are single"""
    _deleteEvents([session.data.allEventsById[evid] for evid in listOfEventIds])

def _insertEvents(listOfEvents):
    """complementary to _deleteEvents. Can be used for circular undo/redo"""
    if not listOfEvents:
        return

    session.history.register(lambda l=listOfEvents: _deleteEvents(l), descriptionString="Insert Events")

    #changedLayers = set()
    for event in listOfEvents:
        #changedLayers.add(event.layer)
        session.data.track.insertEvent(event)
        callbacks._newEvent(event.export())

    session.data.track.generateCalfboxMidi()
    callbacks._songDurationChanged()

def _copy(listOfEvents):
    """Ctrl+C. Copy a list of events into an interal buffer, ready to paste.
    We copy twice: First on copy, so the original event can be deleted.
    Then on paste, because paste can happen multiple times in a row

    Old copy buffers will be replaced.

    Sends no callbacks.
    """
    if not listOfEvents: #empty copy should not delete the existing copy buffer
        return

    session.data.copyBuffer = []
    for event in listOfEvents:
        session.data.copyBuffer.append(event.copy())

def copyById(listOfEventIds):
    """Wrapper for _copy"""
    _copy([session.data.allEventsById[evid] for evid in listOfEventIds])

def _cut(listOfEvents):
    """Like copy, but deletes the originals"""
    _copy(listOfEvents)
    _deleteEvents(listOfEvents)

def cutById(listOfEventIds):
    _cut([session.data.allEventsById[evid] for evid in listOfEventIds])

def paste():
    """Create a copy of our copy-buffer and insert them at the same position as the originals,
    but on the current layer.
    Then send a callback with a list of new event IDs, so a GUI can react by keeping them selected
    for further movement.
    """
    listOfEvents = session.data.getCopyBufferCopy(getActiveLayer())
    _insertEvents(listOfEvents)
    callbacks._pastedEvents([(id(event), event.layer) for event in listOfEvents])

statusToName = {
    0xA0: "Aftertouch",
	0xB0: "Control Change",
	0xC0: "Program Change",
	0xD0: "Channel Pressure",
	0xE0: "Pitch Bend",
	0x90: "Note On",
	0x80: "Note Off",
    }

def createEvent(tickindex:int, status:int, byte1:int, byte2:int):
    """For all events excepts notes"""
    if status == 0x90 or status == 0x80:
        logger.info("createEvent is not for notes")

    layer = session.data.track.layers[session.data.track.activeLayer]
    tickindex = int(tickindex)
    event = layer.newEvent(tickindex, status, byte1, byte2, "")
    name = statusToName[status]
    session.history.register(lambda l=[event]: _deleteEvents(l), descriptionString="Create Event")
    session.data.track.generateCalfboxMidi()
    callbacks._newEvent(event.export())
    callbacks._songDurationChanged()


def createNote(tickindex:int, pitch:int, velocity:int, duration:int):
    """Creates two engine events for note on/off"""
    layer = session.data.track.layers[session.data.track.activeLayer]
    tickindex = int(tickindex)
    evOn = layer.newEvent(tickindex, 0x90, pitch, velocity, "")
    evOff = layer.newEvent(tickindex+duration, 0x80, pitch, velocity, "")
    #newEvent already added the events to their layers. We want to call _insertEvents for convenience (history, callbacks etc.)
    #So we need to remove the events temporarily. This is a cheap and quick operation.
    layer.deleteEvent(evOn)
    layer.deleteEvent(evOff)
    _insertEvents([evOn, evOff])

def _changeBytesSeparateLists(listForOne, listForTwo, differenceInSteps):
    """
    Convenience function to have all changes in one go.

    It is possible to go below 0 and above 127. However, these values do not get exported to
    midi but will trigger a logger-info instead. This means we can go down again without loosing
    information.
    """
    if not listForOne+listForTwo:
        return

    changedLayers = set()
    session.history.register(lambda l1=listForOne, l2=listForTwo, d=-1*differenceInSteps: changeBytesByIdSeparateLists(l1, l2, d), descriptionString="Move Events")
    for event in listForOne:
        event.byte1 += differenceInSteps
        callbacks._eventByteOneChanged(event.export())
        changedLayers.add(event.layer)

    for event in listForTwo:
        event.byte2 += differenceInSteps
        callbacks._eventByteTwoChanged(event.export())
        changedLayers.add(event.layer)

    for layer in changedLayers:
        session.data.track.layers[layer].dirty = True
    session.data.track.generateCalfboxMidi()

def changeBytesByIdSeparateLists(listForOne, listForTwo, differenceInSteps):
    _changeBytesSeparateLists([session.data.allEventsById[evid] for evid in listForOne], [session.data.allEventsById[evid] for evid in listForTwo], differenceInSteps)

def repositionItemsRelative(listOfEventIds, differenceAsTicks):
    if not listOfEventIds:
        return

    changedLayers = set()
    session.history.register(lambda l=listOfEventIds, d=-1*differenceAsTicks: repositionItemsRelative(l, d), descriptionString="Reposition Events")

    for evid in listOfEventIds:
        event = session.data.allEventsById[evid]
        event.position += differenceAsTicks
        callbacks._eventPositionChanged(event.export())
        changedLayers.add(event.layer)

    for layer in changedLayers:
        session.data.track.layers[layer].dirty = True
    session.data.track.generateCalfboxMidi()

def moveEvents(dictOfTuples):
    """Change tickindex and byte1 at the same time (same callback round). This is meant for a GUI
    that moves pitch and position at the same time in a 2D canvas.

    We also change byte2 because in some cases, like CC, "moving" up and down is performed on the
    second byte.

    Parameter is id:(newPos, newByte1, newByte2)

    note on and note off arrive as two events
    """
    if not dictOfTuples:
        return

    undoDataSet = {} #same as parameter. Holds only primitive, immutable types so we can just assign values into it.

    changedLayers = set()
    for eventId, (tickposition, byte1, byte2) in dictOfTuples.items():
        tickposition = int(tickposition)
        event = session.data.allEventsById[eventId]
        undoDataSet[eventId] = (event.position, event.byte1, event.byte2)
        event.byte1 = byte1
        event.byte2 = byte2
        event.position = tickposition
        changedLayers.add(event.layer)
        ex = event.export()
        callbacks._eventByteOneChanged(ex)
        callbacks._eventByteTwoChanged(ex)
        callbacks._eventPositionChanged(ex)

    for layer in changedLayers:
        session.data.track.layers[layer].dirty = True

    session.data.track.generateCalfboxMidi()
    session.history.register(lambda d=undoDataSet: moveEvents(d), descriptionString="Move Events")


def _changeByte2(data:dict):
    """data contains id:absoluteVelocityValue.
    We allow velocities outside the boundaries 0-127 to make undo and relative movement possible
    without compressing data. The engine will warn and compress midi output.

    Notes arrive as only note-on!!
    """

    if not data:
        return

    undoDataSet = {} #same as parameter. Holds only primitive, immutable types so we can just assign values into it.
    changedLayers = set()

    for eventId, byte2 in data.items():
        event = session.data.allEventsById[eventId]
        undoDataSet[eventId] = (event.byte2)
        event.byte2 = byte2
        changedLayers.add(event.layer)
        ex = event.export()
        callbacks._eventByteTwoChanged(ex)

    for layer in changedLayers:
        session.data.track.layers[layer].dirty = True

    session.data.track.generateCalfboxMidi()
    session.history.register(lambda d=undoDataSet: _changeByte2(d), descriptionString="Change Byte2")

def changeVelocitiesRelative(listOfEventIds, differenceInSteps):
    """A wrapper around _changeByte2 which only works on velocities of note-ons"""
    #data = {}
    #for evid in listOfEventIds:
    #    event = session.data.allEventsById[evid]
    #    data[evid] = event.byte2 + differenceInSteps
    data = { evid : session.data.allEventsById[evid].byte2+differenceInSteps for evid in listOfEventIds if session.data.allEventsById[evid].status == 0x90}
    _changeByte2(data)

def setVelocities(listOfEventIds, absoluteValue):
    """A wrapper around _changeByte2 which only works on velocities of note-ons"""
    data = { evid : absoluteValue for evid in listOfEventIds if session.data.allEventsById[evid].status == 0x90}
    _changeByte2(data)

def compressVelocities(listOfEventIds, lowerBound, upperBound):
    """A wrapper around _changeByte2 which only works on velocities of note-ons"""
    c = lambda input: int(compress(input, 0, 127, outputLowest=lowerBound, outputHighest=upperBound))
    data = { evid : c(session.data.allEventsById[evid].byte2) for evid in listOfEventIds if session.data.allEventsById[evid].status == 0x90}
    _changeByte2(data)


def setFreeText(eventId:int, text:str):
    if not session.data.allEventsById[eventId].freeText == text:
        event = session.data.allEventsById[eventId]
        session.history.register(lambda t=event.freeText: setFreeText(eventId, t), descriptionString="Free Text")
        event.freeText = text
        callbacks._eventFreeTextChanged(event.export())

def toggleUndoRecordingCollector(state:bool):
    """The span of one playback session is what gets covered by undo.

    Called by the api itself.
    Works in tandem with checkIfPlaybackIsRunningWhileRecordinChanged because the user
    can start recording while playback is already running.

    However, we don't care if recording is toggled off (and on again?... ) while playback is running.
    """
    if state and session.recordingEnabled: #recording started while playback
        if session.data.eventCollectorForUndo is None:
            session.data.eventCollectorForUndo = []


def checkIfPlaybackIsRunningWhileRecordinChanged(state:bool):
    """Called by the template api.
    Works in tandem with toggleUndoRecordingCollector because the user
    can toggle recording while playback is already/still running.
    """
    if state: #recording was switch on during playback running
        if session.data.eventCollectorForUndo is None:
            session.data.eventCollectorForUndo = []

def chooseActiveLayer(value:int):
    if value < 0 or value > 9:
        raise ValueError("There are only layers from 0 to 9")
    session.data.track.activeLayer = value
    callbacks._activeLayerChanged()

def getActiveLayer()->int:
    return session.data.track.activeLayer

def getActiveColor()->str:
    return session.data.track.layers[getActiveLayer()].color

def setLayerMidiChannel(layer:int, channel:int):
    if channel < 1 or channel > 16:
        raise ValueError("Midi Channel must been between 1 and 16 inclusive, not " + str(channel))

    session.data.track.layers[layer].midiChannel = channel
    session.data.track.layers[layer].dirty = True
    session.data.track.generateCalfboxMidi()
    callbacks._layerMidiChannelChanged(layer) #grabs the channel itself

def getActiveMidiChannel()->int:
    return session.data.track.layers[getActiveLayer()].midiChannel

def getActiveMedianVelocity()->str:
    return session.data.track.layers[getActiveLayer()].cachedMedianVelocity

def layerFilterAndMove(sourceLayerIndex:int, targetLayerIndex:int, statusByte:int, byte1RangeMinimum:int, byte1RangeMaximum:int, byte2RangeMinimum:int, byte2RangeMaximum:int, callback=True):
    """send all events from one layer to another, under certain conditions.
    All ranges are inclusive at both ends.

    If statusByte is either note on or note off it will move BOTH.

    """
    if sourceLayerIndex == targetLayerIndex:
        return
    sourceLayer = session.data.track.layers[sourceLayerIndex]
    targetLayer = session.data.track.layers[targetLayerIndex]
    if statusByte == 0x90 or statusByte == 0x80:
        statusBytes = (0x90, 0x80)
    else:
        statusBytes = (statusByte,) #tuple-comma

    for statusType, listOfEvents in sourceLayer.events.items():
        for event in listOfEvents[:]:
            if event.status in statusBytes:
                byte1InRange = event.byte1 >= byte1RangeMinimum and event.byte1 <= byte1RangeMaximum
                if not event.byte2 is None:
                    byte2InRange = event.byte2 >= byte2RangeMinimum and event.byte2 <= byte2RangeMaximum
                else:
                    byte2InRange = True
                if byte1InRange and byte2InRange:
                    listOfEvents.remove(event) #we iterate over a shallow copy, we delete from the original.
                    targetLayer.events[event.status].append(event)
    if callback:
        callbacks._layerChanged(sourceLayerIndex) #includes cbox update
        callbacks._layerChanged(targetLayerIndex) #includes cbox update
    #song duration does not change

def filterAndMoveAllLayers(targetLayerIndex:int, statusByte:int, byte1RangeMinimum:int, byte1RangeMaximum:int, byte2RangeMinimum:int, byte2RangeMaximum:int):
    """like layerFilterAndMove, but for all layers as source layers, except the targetLayer itself"""
    for sourceLayerIndex in session.data.track.layers.keys():
        layerFilterAndMove(sourceLayerIndex, targetLayerIndex, statusByte, byte1RangeMinimum, byte1RangeMaximum, byte2RangeMinimum, byte2RangeMaximum, callback=False)
        callbacks._layerChanged(sourceLayerIndex)
    callbacks._layerChanged(targetLayerIndex)

def multiple_layerFilterAndMove(listOfInstructions):
    """Calls layerFilterAndMove multiple times, but only callbacks once in the end"""
    for (sourceLayerIndex, targetLayerIndex, statusByte, byte1RangeMinimum, byte1RangeMaximum, byte2RangeMinimum, byte2RangeMaximum) in listOfInstructions:
        layerFilterAndMove(sourceLayerIndex, targetLayerIndex, statusByte, byte1RangeMinimum, byte1RangeMaximum, byte2RangeMinimum, byte2RangeMaximum, callback=False)

    for layerIndex in session.data.track.layers.keys():
        callbacks._layerChanged(layerIndex)

def mulitple_filterAndMoveAllLayers(listOfInstructions):
    """Calls filterAndMoveAllLayers multiple times, but only callbacks once in the end"""
    for (targetLayerIndex, statusByte, byte1RangeMinimum, byte1RangeMaximum, byte2RangeMinimum, byte2RangeMaximum) in listOfInstructions:
        for sourceLayerIndex in session.data.track.layers.keys():
            layerFilterAndMove(sourceLayerIndex, targetLayerIndex, statusByte, byte1RangeMinimum, byte1RangeMaximum, byte2RangeMinimum, byte2RangeMaximum, callback=False)

    for layerIndex in session.data.track.layers.keys():
        callbacks._layerChanged(layerIndex)

def filterTestCC():
    instructions = [
        (1, 0x90, 0, 127, 0, 127), #move all notes to layer 1
        (2, 0xB0, 0, 127, 0, 127), #move all CCs to layer 2
        (3, 0xC0, 0, 127, 0, 127), #move all Program Changes to layer 3
    ]
    mulitple_filterAndMoveAllLayers(instructions)


def sendNoteOnToCbox(midipitch):
    """Not Realtime!
    Caller is responsible to shut off the note"""
    v = getActiveMedianVelocity()
    callbacks._stepEntryNoteOn(midipitch, v)
    cbox.send_midi_event(0x90+getActiveMidiChannel()-1, midipitch, v, output=session.data.track.sequencerInterface.cboxMidiOutUuid)

def sendNoteOffToCbox(midipitch):
    """Not Realtime!"""
    callbacks._stepEntryNoteOff(midipitch, 0)
    cbox.send_midi_event(0x80+getActiveMidiChannel()-1, midipitch, 0, output=session.data.track.sequencerInterface.cboxMidiOutUuid)

