#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2021, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
import logging; logger = logging.getLogger(__name__); logger.info("import")

#Standard Library Modules

#Third Party Modules
from PyQt5 import QtWidgets, QtCore, QtGui

#Template Modules
from template.qtgui.helper import stretchRect
from template.engine.sequencer import MAXIMUM_TICK_DURATION

#User modules
import engine.api as api
from .constantsAndConfigs import constantsAndConfigs

MAX_DURATION = MAXIMUM_TICK_DURATION / constantsAndConfigs.ticksToPixelRatio

oneRectToReturnThemAll = QtCore.QRectF(0,0,0,0) #prevent the annoying "NotImplementError" from Qt for boundingRect. For items that don't need any collision detection.

class PianoGrid(QtWidgets.QGraphicsItem):
    """We draw the piano lines to the maximum possible width.
    The Scenes sceneRect setter needs to take care of proper dimensions. Otherwise you will never
    see the scrollbars"""


    def __init__(self, parentScene):
        super().__init__()
        self.parentScene = parentScene

        self.setAcceptedMouseButtons(QtCore.Qt.NoButton) #each line has this set as well
        self.setEnabled(False)
        #self.setFiltersChildEvents(True) #intercept all child event. Does not prevent the selection rectangle from getting blocked. We need to disable mouse events in all child items.

        self.gridPen = QtGui.QPen(QtCore.Qt.SolidLine)
        self.gridPen.setCosmetic(True)

        self.linesHorizontal = [] #lines
        self.blackKeys = [] #rects, horizontal
        self._rhythmEnabled = True #user sets this to really turn off linesVertical
        self.linesVertical = [] #beat- and barlines
        self.highlights = {} #most of the time invisible

        self.setFlag(QtWidgets.QGraphicsItem.ItemHasNoContents, True) #only child items. Without this we get notImplementedError: QGraphicsItem.paint() is abstract and must be overridden
        self.setOpacity(constantsAndConfigs.gridOpacity)

        #Create two lines for the upper/lower boundaries first. They are just cosmetic
        boldPen = QtGui.QPen(QtCore.Qt.SolidLine)
        boldPen.setCosmetic(True)
        boldPen.setWidth(1)

        hlineUp = QtWidgets.QGraphicsLineItem(0, 0, MAX_DURATION, 0) #x1, y1, x2, y2
        hlineUp.setPen(boldPen)
        hlineUp.setParentItem(self)
        hlineUp.setPos(0, 0)
        hlineUp.setEnabled(False)
        hlineUp.setAcceptedMouseButtons(QtCore.Qt.NoButton)
        hlineUp.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, True)
        self.linesHorizontal.append(hlineUp)

        hlineDown = QtWidgets.QGraphicsLineItem(0, 0, MAX_DURATION, 0) #x1, y1, x2, y2
        hlineDown.setPen(boldPen)
        hlineDown.setParentItem(self)
        hlineDown.setPos(0, 128 * constantsAndConfigs.stafflineGap)
        hlineDown.setEnabled(False)
        hlineDown.setAcceptedMouseButtons(QtCore.Qt.NoButton)
        hlineDown.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, True)
        self.linesHorizontal.append(hlineDown)

        for i in range(128):
            hline = QtWidgets.QGraphicsLineItem(0, 0, MAX_DURATION, 0) #x1, y1, x2, y2
            hline.setPen(self.gridPen)
            hline.setParentItem(self)
            hline.setPos(0, i * constantsAndConfigs.stafflineGap)
            hline.setEnabled(False)
            hline.setAcceptedMouseButtons(QtCore.Qt.NoButton)
            self.linesHorizontal.append(hline)

            blackKey = i % 12 in (1, 3, 6, 8, 10)
            if blackKey:
                bk = BlackKey(self)
                self.blackKeys.append(bk)
                bk.setParentItem(self)
                bk.setPos(0, (127-i) * constantsAndConfigs.stafflineGap)

            hl = Highlight(self)
            hl.setParentItem(self)
            self.highlights[i] = hl
            hl.setPos(0, (127-i) * constantsAndConfigs.stafflineGap)

        self._cachedBBT = None
        api.callbacks.bbtStatusChanged.append(self.bbtStatusChanged)  #Beat/Bar Grid is variable

        api.callbacks.stepEntryNoteOn.append(self.highlightNoteOn)
        api.callbacks.stepEntryNoteOff.append(self.highlightNoteOff)
        api.callbacks.recordingModeChanged.append(self.allHighlightsOff)
        api.callbacks.playbackStatusChanged.append(self.allHighlightsOff)
        api.callbacks.activeLayerChanged.append(self.activeLayerChanged)
        api.callbacks.layerColorChanged.append(self.layerColorChanged)


    def toggleRhythmEnabled(self)->bool:
        """Really turns off vertical grid. Only trivial cpu time will be spent returning None
        for any update attempt"""
        self._rhythmEnabled = not self._rhythmEnabled
        self.adjustRhythmLines()
        return self._rhythmEnabled

    def activeLayerChanged(self, layerIndex:int):
        """prepare color for all highlights"""
        c = QtGui.QColor(api.getActiveColor())
        for highlight in self.highlights.values():
            highlight.setBrush(c)

    def layerColorChanged(self, layerIndex:int, colorString:str):
        if api.getActiveLayer() == layerIndex:
            self.activeLayerChanged(layerIndex)

    def highlightNoteOn(self, pitch:int, velocity:int):
        """only when transport is not rolling"""
        highlight = self.highlights[pitch]
        highlight.show()

    def highlightNoteOff(self, pitch:int, velocity:int):
        highlight = self.highlights[pitch]
        highlight.hide()

    def allHighlightsOff(self, state):
        """If the playback or recording mode switches we want to deactivate all hanging highlights.
        We do this in both directions, but from 'on' to 'off' the list is always empty anyway"""
        for pitch, highlight in self.highlights.items():
            highlight.hide()

    def boundingRect(self, *args):
        return oneRectToReturnThemAll

    def stretchXCoordinates(self, factor:float):
        """The Horizontal pitch lines do not need stretching. They are as long as they can be"""
        for line in self.linesVertical:
            line.setX(line.pos().x() * factor)
        self.adjustRhythmLines()

    def bbtStatusChanged(self, exportDict:dict):
        """
        exportDict["nominator"]
        exportDict["denominator"] #in our ticks
        exportDict["measureInTicks"]
        """
        self._cachedBBT = exportDict
        self.adjustRhythmLines()

    def adjustRhythmLines(self):
        """Called by scoreView._decideIfToGrowSceneRect and self.bbtStatusChanged"""
        if self._rhythmEnabled and self._cachedBBT:
            #Prepare Data
            width = self.parentScene.parentView.sceneRect().width() # + self.parentScene.parentView.geometry().width()
            beatPixel = self._cachedBBT["denominator"] / constantsAndConfigs.ticksToPixelRatio
            #measureInTicks = self._cachedBBT["measureInTicks"] / constantsAndConfigs.ticksToPixelRatio
            linesPerMeasure = int(self._cachedBBT["nominator"])
            shouldBeNrOfLines = int(width / beatPixel) #linesPerMeasure as offset/buffer. only cosmetic.
            existingNrOfLines = len(self.linesVertical)

            #offset = self._cachedBBT["offsetToMeasureBeginning"]
            #offset = (self._cachedBBT["tickposition"] - offset) / constantsAndConfigs.ticksToPixelRatio
            #This is not working properly. Deactivate offset for now.
            offset = 0

            if shouldBeNrOfLines > existingNrOfLines:
                weNeedNewLines = shouldBeNrOfLines - existingNrOfLines
                for i in range(weNeedNewLines):
                    rl = RhythmLine(self)
                    self.linesVertical.append(rl)
            else:
               for line in self.linesVertical[shouldBeNrOfLines:]:
                   line.hide()

            #position the exact number of lines to BBT info
            for i, line in enumerate(self.linesVertical[:shouldBeNrOfLines]):
                if constantsAndConfigs.ticksToPixelRatio <= 20:
                    line.show()
                    line.setPos(i*beatPixel+offset,0)
                    line.setBold(not i % linesPerMeasure)
                elif not i % linesPerMeasure: #only barlines.
                    line.show()
                    line.setPos(i*beatPixel+offset,0)
                    line.setBold(True)
                else:
                    line.hide()

        else: #no BBT, no vertical grid lines.
            for line in self.linesVertical:
                line.hide()

masterLine = QtCore.QLineF(0, 0, 0, 128*constantsAndConfigs.stafflineGap)  # (x1, y1, x2, y2)
class RhythmLine(QtWidgets.QGraphicsLineItem):
    def __init__(self, parentGrid):
        super().__init__(masterLine)
        self.setEnabled(False)
        self.setParentItem(parentGrid)
        self.setAcceptedMouseButtons(QtCore.Qt.NoButton) #we still need this otherwise no rubberband.

    def setBold(self, state:bool):
        if state:
            self.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, True)
            self.setOpacity(0.4)
        else:
            self.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, False)
            #uses parent opacity

class Highlight(QtWidgets.QGraphicsRectItem):
    def __init__(self, parentGrid):
        super().__init__(0, 0, MAX_DURATION, constantsAndConfigs.stafflineGap) #x, y, w, h
        self.setEnabled(False)
        self.setAcceptedMouseButtons(QtCore.Qt.NoButton) #we still need this otherwise no rubberband.
        self.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, True)
        self.setOpacity(0.4)
        self.hide()


class BlackKey(QtWidgets.QGraphicsRectItem):
    def __init__(self, parentGrid):
        super().__init__(0, 0, MAX_DURATION, constantsAndConfigs.stafflineGap) #x, y, w, h
        self.parentGrid = parentGrid
        self.setPen(QtGui.QPen(QtCore.Qt.NoPen))
        self.setBrush(QtGui.QColor("black"))
        self.setEnabled(False)
        self.setAcceptedMouseButtons(QtCore.Qt.NoButton) #we still need this otherwise no rubberband.






