#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2021, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
import logging; logger = logging.getLogger(__name__); logger.info("import")

#Standard Library Modules

#Third Party Modules
from PyQt5 import QtWidgets, QtCore, QtGui, QtOpenGL

#Template Modules
from template.qtgui.helper import stretchRect
from template.engine.duration import baseDurationToTraditionalNumber

#User modules
from .constantsAndConfigs import constantsAndConfigs
from .pianogrid import Highlight, BlackKey
import engine.api as api


MAX_DURATION = 50 #to keep the code copy/paste compatible with piano grid we use the same constant but use our own value


class VerticalPiano(QtWidgets.QGraphicsView):

    def __init__(self, mainWindow):
        super().__init__(mainWindow)
        self.mainWindow = mainWindow

        viewport = QtWidgets.QOpenGLWidget()
        viewportFormat = QtGui.QSurfaceFormat()
        viewportFormat.setSwapInterval(0) #disable VSync
        #viewportFormat.setSamples(2**8) #By default, the highest number of samples available is used.
        viewportFormat.setDefaultFormat(viewportFormat)
        viewport.setFormat(viewportFormat)
        self.setViewport(viewport)

        self.setAlignment(QtCore.Qt.AlignLeft|QtCore.Qt.AlignBottom)
        self.setDragMode(QtWidgets.QGraphicsView.NoDrag)

        self.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)
        self.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)

        self.pianoScene = _VerticalPianoScene(self)
        self.setScene(self.pianoScene)

        self.setSceneRect(QtCore.QRectF(0, 0, MAX_DURATION/2, constantsAndConfigs.scoreHeight)) #x,y,w,h
        #self.setFixedHeight(constantsAndConfigs.scoreHeight+3)  # no zoom
        self.setFixedWidth(MAX_DURATION)


        style = """
        QScrollBar:horizontal {
            border: 1px solid black;
        }

        QScrollBar::handle:horizontal {
            background: #00b2b2;
        }

        QScrollBar:vertical {
            border: 1px solid black;
        }

        QScrollBar::handle:vertical {
            background: #00b2b2;
        }

        """
        self.setStyleSheet(style)
        self.setLineWidth(0)


    def zoom(self, factor):
        """Factor is absolute. We reset before setting the new scale"""
        assert factor == constantsAndConfigs.zoomFactor
        self.resetTransform()
        self.scale(factor, factor)

class _VerticalPianoScene(QtWidgets.QGraphicsScene):
    """Most of this is copy paste from piano grid"""

    def __init__(self, parentView):
        super().__init__()

        #Set color, otherwise it will be transparent in window managers or wayland that want that.
        self.backColor = QtGui.QColor()
        self.backColor.setNamedColor("#fdfdff")
        self.setBackgroundBrush(self.backColor)

        self.linesHorizontal = []
        self.highlights = {}
        self.blackKeys = []

        self._middleMouseDown = False #For note preview

        self.gridPen = QtGui.QPen(QtCore.Qt.SolidLine)
        self.gridPen.setCosmetic(True)

        #Create two lines for the upper/lower boundaries first. They are just cosmetic
        boldPen = QtGui.QPen(QtCore.Qt.SolidLine)
        boldPen.setCosmetic(True)
        boldPen.setWidth(1)

        hlineUp = QtWidgets.QGraphicsLineItem(0, 0, MAX_DURATION*2, 0) #x1, y1, x2, y2
        hlineUp.setPen(boldPen)
        self.addItem(hlineUp)
        hlineUp.setPos(0, 0)
        hlineUp.setEnabled(False)
        hlineUp.setAcceptedMouseButtons(QtCore.Qt.NoButton)
        hlineUp.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, True)
        self.linesHorizontal.append(hlineUp)

        hlineDown = QtWidgets.QGraphicsLineItem(0, 0, MAX_DURATION*2, 0) #x1, y1, x2, y2
        hlineDown.setPen(boldPen)
        self.addItem(hlineDown)
        hlineDown.setPos(0, 128 * constantsAndConfigs.stafflineGap)
        hlineDown.setEnabled(False)
        hlineDown.setAcceptedMouseButtons(QtCore.Qt.NoButton)
        hlineDown.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, True)
        self.linesHorizontal.append(hlineDown)

        for i in range(128):
            hline = QtWidgets.QGraphicsLineItem(0, 0, MAX_DURATION*2, 0) #x1, y1, x2, y2
            hline.setPen(self.gridPen)
            self.addItem(hline)
            hline.setPos(0, i * constantsAndConfigs.stafflineGap)
            hline.setEnabled(False)
            hline.setAcceptedMouseButtons(QtCore.Qt.NoButton)
            self.linesHorizontal.append(hline)

            blackKey = i % 12 in (1, 3, 6, 8, 10)
            if blackKey:
                bk = BlackKey(self)
                self.blackKeys.append(bk)
                self.addItem(bk)
                bk.setPos(0, (127-i) * constantsAndConfigs.stafflineGap)

            numberLabel = NumberLabel(self, 127-i)
            self.addItem(numberLabel)
            numberLabel.setPos(0, i * constantsAndConfigs.stafflineGap)
            numberLabel.setZValue(10)

            hl = Highlight(self)
            self.addItem(hl)
            self.highlights[i] = hl
            hl.setPos(0, (127-i) * constantsAndConfigs.stafflineGap)

        api.callbacks.stepEntryNoteOn.append(self.highlightNoteOn)
        api.callbacks.stepEntryNoteOff.append(self.highlightNoteOff)
        api.callbacks.recordingModeChanged.append(self.allHighlightsOff)
        api.callbacks.playbackStatusChanged.append(self.allHighlightsOff)
        api.callbacks.activeLayerChanged.append(self.activeLayerChanged)
        api.callbacks.layerColorChanged.append(self.layerColorChanged)

    def activeLayerChanged(self, layerIndex:int):
        """prepare color for all highlights"""
        c = QtGui.QColor(api.getActiveColor())
        for highlight in self.highlights.values():
            highlight.setBrush(c)

    def layerColorChanged(self, layerIndex:int, colorString:str):
        if api.getActiveLayer() == layerIndex:
            self.activeLayerChanged(layerIndex)

    def highlightNoteOn(self, pitch:int, velocity:int):
        """only when transport is not rolling"""
        highlight = self.highlights[pitch]
        highlight.show()

    def highlightNoteOff(self, pitch:int, velocity:int):
        highlight = self.highlights[pitch]
        highlight.hide()

    def allHighlightsOff(self, state):
        """If the If the playback or recording mode switches we want to deactivate all hanging highlights.
        We do this in both directions, but from 'on' to 'off' the list is always empty anyway"""
        for pitch, highlight in self.highlights.items():
            highlight.hide()

    def mousePressEvent(self, event):
        self._middleMouseDown = False
        if event.button() == QtCore.Qt.MiddleButton:
            self._middleMouseDown = True
            self._lastPlayPitch = None  #this must not be in _play, otherwise you can't move the mouse while pressed down
            self._play(event)
        super().mousePressEvent(event)

    def _off(self):
        if not self._lastPlayPitch is None:
            api.sendNoteOffToCbox(self._lastPlayPitch)
            self._lastPlayPitch = None

    def _play(self, event):
        assert self._middleMouseDown

        pitch = 127 - int(event.scenePos().y() / constantsAndConfigs.stafflineGap)
        if pitch < 0 or pitch > 127:
            pitch = None

        if not pitch == self._lastPlayPitch:
            if not self._lastPlayPitch is None:
                api.sendNoteOffToCbox(self._lastPlayPitch)

            if not pitch is None:
                api.sendNoteOnToCbox(pitch) #uses active layer median velocity

        self._lastPlayPitch = pitch

    def mouseMoveEvent(self, event):
        """Event button is always 0 in a mouse move event"""
        if self._middleMouseDown:
            self._play(event)
        super().mouseMoveEvent(event)

    def mouseReleaseEvent(self, event):
        if self._middleMouseDown:
            self._off()
            self._middleMouseDown = False
        if event.button() == QtCore.Qt.MiddleButton:
            self._lastPlayPitch = None
        super().mouseReleaseEvent(event)



class NumberLabel(QtWidgets.QGraphicsSimpleTextItem):

    def __init__(self, parentGrid, number:int):

        super().__init__()
        self.parentGrid = parentGrid
        self.setText(str(number))
        self.setFlag(QtWidgets.QGraphicsItem.ItemIgnoresParentOpacity, True)
        self.setScale(0.5)
        self.setAcceptedMouseButtons(QtCore.Qt.NoButton)
        blackKey = number % 12 in (1, 3, 6, 8, 10)
        if blackKey:
            self.setBrush(QtGui.QColor("white"))
        else:
            self.setBrush(QtGui.QColor("black"))
