/*
 * Wire
 * Copyright (C) 2018 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 *
 */

import {ipcRenderer, webFrame, WebviewTag} from 'electron';
import {WebAppEvents} from '@wireapp/webapp-events';
import * as path from 'path';

import {EVENT_TYPE} from '../lib/eventType';
import * as locale from '../locale/locale';
import {getLogger} from '../logging/getLogger';
import * as EnvironmentUtil from '../runtime/EnvironmentUtil';
import {AutomatedSingleSignOn} from '../sso/AutomatedSingleSignOn';

const logger = getLogger(path.basename(__filename));

webFrame.setVisualZoomLevelLimits(1, 1);

window.locStrings = locale.LANGUAGES[locale.getCurrent()];
window.locStringsDefault = locale.LANGUAGES.en;
window.locale = locale.getCurrent();

window.isMac = EnvironmentUtil.platform.IS_MAC_OS;

const getSelectedWebview = (): WebviewTag | null => document.querySelector<WebviewTag>('.Webview:not(.hide)');
const getWebviewById = (id: string): WebviewTag | null =>
  document.querySelector<WebviewTag>(`.Webview[data-accountid="${id}"]`);

const subscribeToMainProcessEvents = (): void => {
  ipcRenderer.on(EVENT_TYPE.ACCOUNT.SSO_LOGIN, (_event, code: string) => new AutomatedSingleSignOn().start(code));
  ipcRenderer.on(EVENT_TYPE.ACTION.JOIN_CONVERSATION, async (_event, {code, key}: {code: string; key: string}) => {
    const selectedWebview = getSelectedWebview();
    if (selectedWebview) {
      await selectedWebview.send(EVENT_TYPE.ACTION.JOIN_CONVERSATION, {code, key});
    }
  });

  ipcRenderer.on(EVENT_TYPE.UI.SYSTEM_MENU, async (_event, action: string) => {
    const selectedWebview = getSelectedWebview();
    if (selectedWebview) {
      await selectedWebview.send(action);
    }
  });

  ipcRenderer.on(EVENT_TYPE.WEBAPP.CHANGE_LOCATION_HASH, async (_event, hash: string) => {
    const selectedWebview = getSelectedWebview();
    if (selectedWebview) {
      await selectedWebview.send(EVENT_TYPE.WEBAPP.CHANGE_LOCATION_HASH, hash);
    }
  });

  ipcRenderer.on(EVENT_TYPE.EDIT.COPY, () => getSelectedWebview()?.copy());
  ipcRenderer.on(EVENT_TYPE.EDIT.CUT, () => getSelectedWebview()?.cut());
  ipcRenderer.on(EVENT_TYPE.EDIT.PASTE, () => getSelectedWebview()?.paste());
  ipcRenderer.on(EVENT_TYPE.EDIT.REDO, () => getSelectedWebview()?.redo());
  ipcRenderer.on(EVENT_TYPE.EDIT.SELECT_ALL, () => getSelectedWebview()?.selectAll());
  ipcRenderer.on(EVENT_TYPE.EDIT.UNDO, () => getSelectedWebview()?.undo());

  ipcRenderer.on(EVENT_TYPE.WRAPPER.RELOAD, (): void => {
    const webviews = document.querySelectorAll<WebviewTag>('webview');
    webviews.forEach(webview => webview.reload());
  });

  ipcRenderer.on(EVENT_TYPE.ACTION.SWITCH_ACCOUNT, (event, accountIndex: number) => {
    window.dispatchEvent(new CustomEvent(EVENT_TYPE.ACTION.SWITCH_ACCOUNT, {detail: {accountIndex}}));
  });
};

const setupIpcInterface = (): void => {
  window.sendBadgeCount = (count: number, ignoreFlash: boolean): void => {
    ipcRenderer.send(EVENT_TYPE.UI.BADGE_COUNT, {count, ignoreFlash});
  };

  window.submitDeepLink = (url: string): void => {
    ipcRenderer.send(EVENT_TYPE.ACTION.DEEP_LINK_SUBMIT, url);
  };

  window.sendDeleteAccount = (accountId: string, sessionID?: string): Promise<void> => {
    return new Promise((resolve, reject) => {
      const accountWebview = getWebviewById(accountId);
      if (!accountWebview) {
        // eslint-disable-next-line prefer-promise-reject-errors
        return reject(`Webview for account "${accountId}" does not exist`);
      }

      logger.info(`Processing deletion of "${accountId}"`);
      const viewInstanceId = accountWebview.getWebContentsId();
      ipcRenderer.on(EVENT_TYPE.ACCOUNT.DATA_DELETED, () => resolve());
      ipcRenderer.send(EVENT_TYPE.ACCOUNT.DELETE_DATA, viewInstanceId, accountId, sessionID);
    });
  };

  window.sendLogoutAccount = async (accountId: string): Promise<void> => {
    const accountWebview = getWebviewById(accountId);
    logger.log(`Sending logout signal to webview for account "${accountId}".`);
    await accountWebview?.send(EVENT_TYPE.ACTION.SIGN_OUT);
  };

  window.sendConversationJoinToHost = async (accountId: string, code: string, key: string): Promise<void> => {
    const accountWebview = getWebviewById(accountId);
    logger.log(`Sending conversation join data to webview for account "${accountId}".`);
    await accountWebview?.send(WebAppEvents.CONVERSATION.JOIN, {code, key});
  };
};

const addDragRegion = (): void => {
  if (EnvironmentUtil.platform.IS_MAC_OS) {
    // add title bar ghost to prevent interactions with the content while dragging
    const titleBar = document.createElement('div');
    titleBar.className = 'drag-region';
    document.body.appendChild(titleBar);
  }
};

setupIpcInterface();
subscribeToMainProcessEvents();

window.addEventListener('DOMContentLoaded', addDragRegion);
window.addEventListener('focus', () => {
  const selectedWebview = getSelectedWebview();
  if (selectedWebview) {
    selectedWebview.blur();
    selectedWebview.focus();
  }
});
