/*
  libreiserfs - a library for manipulating reiserfs partitions
  Copyright (C) 2001-2004 Yury Umanets <torque@ukrpost.net>.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.
                                                                                                 
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.
                                                                                                 
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include <dal/file_dal.h>
#include <reiserfs/reiserfs.h>
#include <progs/gauge.h>

static unsigned long leaf_count, internal_count, total, nodes, unformatted;

static long node_func(reiserfs_block_t *node, void *data) {
	reiserfs_gauge_t *gauge = (reiserfs_gauge_t *)data;

	if (is_internal_node(node))
		internal_count++;
	else if (is_leaf_node(node)) {
		uint32_t i;
		reiserfs_item_head_t *item;

		for (i = 0; i < get_blkh_nr_items(GET_BLOCK_HEAD(node)); i++) {
			item = GET_ITEM_HEAD(node, i);

			if (is_indirect_ih(item)) {
				internal_count++;
				nodes += IH_UNFM_NUM(item);
				unformatted += IH_UNFM_NUM(item);
			}
		}
	
		leaf_count++;
	} else {		
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    "Invalid node type has detected.");
		return 0;
	}

	total++;
	nodes++;
		
	if (gauge)
		libreiserfs_gauge_touch(gauge);
	
	return 1;
}

int main(int argc, char *argv[]) {
	dal_t *dal;
	reiserfs_fs_t *fs;
	reiserfs_gauge_t *gauge;
    
	int traverse_res;

	if (argc < 2) {
		fprintf(stderr, "Usage: %s DEV\n", argv[0]);
		return 0xff;
	}

	if (!(dal = file_dal_open(argv[1], DEFAULT_BLOCK_SIZE, O_RDONLY))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    "Couldn't create device abstraction "
					    "for %s.", argv[1]);
		return 0xfe;    
	}
    
	if (!(fs = reiserfs_fs_open_fast(dal, dal))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    "Couldn't open filesystem on %s.",
					    argv[1]);
		goto error_free_dal;
	}

	total = 0; nodes = 0; unformatted = 0;
	leaf_count = 0, internal_count = 0;

	gauge = progs_gauge_create();
	libreiserfs_gauge_undetermined(gauge);
	libreiserfs_gauge_set_name(gauge, "traversing");
    
	traverse_res = reiserfs_tree_simple_traverse(reiserfs_fs_tree(fs), gauge, node_func);
    
	if (gauge) {
		libreiserfs_gauge_done(gauge);
		progs_gauge_free(gauge);
	}	
    
	if (traverse_res)
		fprintf(stderr, "unformatted: %lu\nnodes: %lu\ntotal: "
			"%lu\nleaves: %lu\ninternals: %lu\n", 
			unformatted, nodes, total, leaf_count,
			internal_count);

	reiserfs_fs_close(fs);
	file_dal_close(dal);
    
	return 0;
    
error_free_dal:
	file_dal_close(dal);
error:
	return 0xff;    
}

