/****************************************************************************
**
** Copyright (C) 2021 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <qaudiooutput.h>
#include <qaudiodevice.h>
#include <qmediadevices.h>
#include <private/qplatformaudiooutput_p.h>
#include <private/qplatformmediaintegration_p.h>

/*!
    \class QAudioOutput
    \brief Represents an output channel for audio.
    \inmodule QtMultimedia
    \ingroup multimedia
    \ingroup multimedia_audio
    \since 6.0

    This class represents an output channel that can be used together with
    QMediaPlayer or QMediaCaptureSession. It enables the selection of the
    physical output device to be used, muting the channel, and changing the
    channel's volume.
*/

/*!
    \property QAudioOutput::volume
    \brief The current volume.

    The volume is scaled linearly, ranging from \c 0 (silence) to \c 1
    (full volume).
    \note values outside this range will be clamped.

    By default the volume is \c 1.

    UI volume controls should usually be scaled non-linearly. For example,
    using a logarithmic scale will produce linear changes in perceived loudness,
    which is what a user would normally expect from a volume control.

    \sa QAudio::convertVolume()
*/

/*!
    \property QAudioOutput::muted
    \brief The muted state of the current media.

    The value will be \c true if the output is muted; otherwise \c{false}.
*/

/*!
    \property QAudioOutput::device
    \brief The audio device connected to this output.

    The device property represents the audio device connected to this output.
    A default constructed
    QAudioOutput object will be connected to the systems default audio output at
    construction time.

    This property can be used to select any other output device listed by
    QMediaDevices::audioOutputs().
*/

QAudioOutput::QAudioOutput(QObject *parent)
    : QAudioOutput(QMediaDevices::defaultAudioOutput(), parent)
{}

QAudioOutput::QAudioOutput(const QAudioDevice &device, QObject *parent)
    : QObject(parent),
    d(QPlatformMediaIntegration::instance()->createAudioOutput(this))
{
    d->device = device;
    if (!d->device.isNull() && d->device.mode() != QAudioDevice::Output)
        d->device = QMediaDevices::defaultAudioOutput();
    d->setAudioDevice(d->device);
}

QAudioOutput::~QAudioOutput()
{
    delete d;
}

QAudioDevice QAudioOutput::device() const
{
    return d->device;
}

void QAudioOutput::setDevice(const QAudioDevice &device)
{
    if (!device.isNull() && device.mode() != QAudioDevice::Output)
        return;
    if (d->device == device)
        return;
    d->device = device;
    d->setAudioDevice(device);
    emit deviceChanged();
}

float QAudioOutput::volume() const
{
    return d->volume;
}

void QAudioOutput::setVolume(float volume)
{
    volume = qBound(0., volume, 1.);
    if (d->volume == volume)
        return;
    d->volume = volume;
    d->setVolume(volume);
    emit volumeChanged(volume);
}

bool QAudioOutput::isMuted() const
{
    return d->muted;
}

void QAudioOutput::setMuted(bool muted)
{
    if (d->muted == muted)
        return;
    d->muted = muted;
    d->setMuted(muted);
    emit mutedChanged(muted);
}

#include "moc_qaudiooutput.cpp"
