/****************************************************************************
**
** Copyright (C) 2021 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qmediaformat.h"
#include "private/qplatformmediaintegration_p.h"
#include "private/qplatformmediaformatinfo_p.h"
#include <QtCore/qmimedatabase.h>

QT_BEGIN_NAMESPACE
/*!
    \class QMediaFormat
    \ingroup multimedia
    \inmodule QtMultimedia
    \brief Describes an encoding format for a multimedia file or stream.

    QMediaFormat describes an encoding format for a multimedia file or stream.

    You can check whether a certain media format can be used for encoding
    or decoding using QMediaFormat.

    \since 6.2
*/
namespace {

const char *mimeTypeForFormat[QMediaFormat::LastFileFormat + 2] =
{
    "",
    "video/x-ms-wmv",
    "video/x-msvideo",
    "video/x-matroska",
    "video/mp4",
    "video/ogg",
    "video/quicktime",
    "video/webm",
    // Audio Formats
    "audio/mp4",
    "audio/aac",
    "audio/x-ms-wma",
    "audio/flac",
    "audio/mpeg",
    "audio/wav",
};

constexpr QMediaFormat::FileFormat videoFormatPriorityList[] =
{
    QMediaFormat::MPEG4,
    QMediaFormat::QuickTime,
    QMediaFormat::AVI,
    QMediaFormat::WebM,
    QMediaFormat::WMV,
    QMediaFormat::Matroska,
    QMediaFormat::Ogg,
    QMediaFormat::UnspecifiedFormat
};

constexpr QMediaFormat::FileFormat audioFormatPriorityList[] =
{
    QMediaFormat::Mpeg4Audio,
    QMediaFormat::MP3,
    QMediaFormat::WMA,
    QMediaFormat::FLAC,
    QMediaFormat::Wave,
    QMediaFormat::UnspecifiedFormat
};

constexpr QMediaFormat::AudioCodec audioPriorityList[] =
{
    QMediaFormat::AudioCodec::AAC,
    QMediaFormat::AudioCodec::MP3,
    QMediaFormat::AudioCodec::AC3,
    QMediaFormat::AudioCodec::Opus,
    QMediaFormat::AudioCodec::EAC3,
    QMediaFormat::AudioCodec::DolbyTrueHD,
    QMediaFormat::AudioCodec::WMA,
    QMediaFormat::AudioCodec::FLAC,
    QMediaFormat::AudioCodec::Vorbis,
    QMediaFormat::AudioCodec::Wave,
    QMediaFormat::AudioCodec::Unspecified
};

constexpr QMediaFormat::VideoCodec videoPriorityList[] =
{
    QMediaFormat::VideoCodec::H265,
    QMediaFormat::VideoCodec::VP9,
    QMediaFormat::VideoCodec::H264,
    QMediaFormat::VideoCodec::AV1,
    QMediaFormat::VideoCodec::VP8,
    QMediaFormat::VideoCodec::WMV,
    QMediaFormat::VideoCodec::Theora,
    QMediaFormat::VideoCodec::MPEG4,
    QMediaFormat::VideoCodec::MPEG2,
    QMediaFormat::VideoCodec::MPEG1,
    QMediaFormat::VideoCodec::MotionJPEG,
};

}

class QMediaFormatPrivate : public QSharedData
{};

QT_DEFINE_QESDP_SPECIALIZATION_DTOR(QMediaFormatPrivate);

/*! \enum QMediaFormat::FileFormat

    Describes the container format used in a multimedia file or stream.

    \value WMA
        \l {Windows Media Audio}
    \value UnspecifiedFormat
        The format is unspecified.
    \value AAC
        \l{Advanced Audio Coding}
    \value Matroska
        \l{Matroska (MKV)}
    \value WMV
        \l{Windows Media Video}
    \value MP3
        \l{MPEG-1 Audio Layer III or MPEG-2 Audio Layer III}
    \value Wave
        \l{Waveform Audio File Format}
    \value Ogg
        \l{Ogg}
    \value MPEG4
        \l{MPEG-4}
    \value AVI
        \l{Audio Video Interleave}
    \value QuickTime
        \l{QuickTime}
    \value WebM
        \l{WebM}
    \value Mpeg4Audio
        \l{MPEG-4 Part 3 or MPEG-4 Audio (formally ISO/IEC 14496-3)}
    \value FLAC
        \l{Free Lossless Audio Codec}

    \omitvalue LastFileFormat
*/

/*! \enum QMediaFormat::AudioCodec

    Describes the audio coded used in multimedia file or stream.

    \value WMA
        \l {Windows Media Audio}
    \value AC3
        \l {Dolby Digital}
    \value AAC
        \l{Advanced Audio Coding}
    \value ALAC
        Apple Lossless Audio Codec
    \value DolbyTrueHD
        \l{Dolby TrueHD}
    \value MP3
        \l{MPEG-1 Audio Layer III or MPEG-2 Audio Layer III}
    \value Wave
        \l{Waveform Audio File Format}
    \value Vorbis
        \l{Ogg Vorbis}
    \value FLAC
        \l{Free Lossless Audio Codec}
    \value Opus
        Voice-over-IP codec
    \value EAC
        Enhanced Audio Codec
    \value Unspecified
        Unspecified codec

    \omitvalue LastAudioCodec
*/

/*! \enum QMediaFormat::VideoCodec

    Describes the video coded used in multimedia file or stream.

    \value VP8
        \l{VP8}
    \value MPEG2
        \l{MPEG-2}
    \value MPEG1
        \l{MPEG-1}
    \value WMV
        \l{Windows Media Video}
    \value H265
        \l{High Efficiency Video Coding (HEVC)}
    \value H264
        \l{Advanced Video Coding}
    \value MPEG4
        \l{MPEG-4}
    \value AV1
        \l{AOMedia Video 1}
    \value MotionJPEG
        \l{MotionJPEG}
    \value VP9
        \l{VP9}
    \value Unspecified
        Video codec not specified
    \value Theora
        \l{Theora}

    \omitvalue LastVideoCodec
*/

// these are non inline to make a possible future addition of a d pointer binary compatible

/*!
    Constructs a QMediaFormat object for \a format.
*/
QMediaFormat::QMediaFormat(FileFormat format)
    : fmt(format)
{
}

/*!
    Destroys the QMediaFormat object.
*/
QMediaFormat::~QMediaFormat() = default;

/*!
    Constructs a QMediaFormat object by copying from \a other.
*/
QMediaFormat::QMediaFormat(const QMediaFormat &other) noexcept = default;

/*!
    Copies \a other into this QMediaFormat object.
*/
QMediaFormat &QMediaFormat::operator=(const QMediaFormat &other) noexcept = default;

/*! \fn QMediaFormat::QMediaFormat(QMediaFormat &&other)

    Constructs a QMediaFormat objects by moving from \a other.
*/

/*! \fn QMediaFormat &QMediaFormat::operator=(QMediaFormat &&other)

    Moves \a other into this QMediaFormat objects.
*/

// Properties
/*! \property QMediaFormat::fileFormat

    \brief The file (container) format of the media.

    \sa QMediaFormat::FileFormat
*/

/*! \property QMediaFormat::audioCodec

    \brief The audio codec of the media.

    \sa QMediaFormat::AudioCodec
*/

/*! \property QMediaFormat::videoCodec

    \brief The video codec of the media.

    \sa QMediaFormat::VideoCodec
*/

/*! \fn void QMediaFormat::setVideoCodec(VideoCodec codec)

    Sets the video codec to \a codec.

    \sa videoCodec(), QMediaFormat::VideoCodec
*/

/*! \fn QMediaFormat::VideoCodec QMediaFormat::videoCodec() const

    Returns the video codec used in this format.

    \sa setVideoCodec(), QMediaFormat::VideoCodec
*/

/*! \fn void QMediaFormat::setAudioCodec(AudioCodec codec)

    Sets the audio codec to \a codec.

    \sa audioCodec(), QMediaFormat::AudioCodec
*/

/*! \fn QMediaFormat::AudioCodec QMediaFormat::audioCodec() const

    Returns the audio codec used in this format.

    \sa setAudioCodec(), QMediaFormat::AudioCodec
*/

/*!
    Returns true if Qt Multimedia can encode or decode this format,
    depending on \a mode.
*/

bool QMediaFormat::isSupported(ConversionMode mode) const
{
    return QPlatformMediaIntegration::instance()->formatInfo()->isSupported(*this, mode);
}

/*!
    Returns the \l{MIME type} for the file format used in this media format.
*/

QMimeType QMediaFormat::mimeType() const
{
    return QMimeDatabase().mimeTypeForName(QString::fromLatin1(mimeTypeForFormat[fmt + 1]));
}

static QPlatformMediaFormatInfo *formatInfo()
{
    QPlatformMediaFormatInfo *result = nullptr;
    if (auto *pi = QPlatformMediaIntegration::instance())
        result = pi->formatInfo();
    return result;
}

/*!
    The function returns a list of file formats for the audio and video
    codec indicated by \a{m}.

    To get all supported file formats, run this query on a default constructed
    QMediaFormat.
*/
QList<QMediaFormat::FileFormat> QMediaFormat::supportedFileFormats(QMediaFormat::ConversionMode m)
{
    auto *fi = formatInfo();
    return fi != nullptr ? fi->supportedFileFormats(*this, m) : QList<QMediaFormat::FileFormat>{};
}

/*!
    The function returns a list of video codecs for the chosen file format and
    audio codec (\a m).

    To get all supported video codecs, run this query on a default constructed
    QMediaFormat.
*/
QList<QMediaFormat::VideoCodec> QMediaFormat::supportedVideoCodecs(QMediaFormat::ConversionMode m)
{
    auto *fi = formatInfo();
    return fi != nullptr ? fi->supportedVideoCodecs(*this, m) : QList<QMediaFormat::VideoCodec>{};
}

/*!

    The function returns a list of audio codecs for the chosen file format and
    video codec (\a m).

    To get all supported audio codecs, run this query on a default constructed
    QMediaFormat.
*/
QList<QMediaFormat::AudioCodec> QMediaFormat::supportedAudioCodecs(QMediaFormat::ConversionMode m)
{
    auto *fi = formatInfo();
    return fi != nullptr ? fi->supportedAudioCodecs(*this, m) : QList<QMediaFormat::AudioCodec>{};
}

QString QMediaFormat::fileFormatName(QMediaFormat::FileFormat c)
{
    constexpr const char *descriptions[QMediaFormat::LastFileFormat + 2] = {
        "Unspecified",
        "WMV",
        "AVI",
        "Matroska",
        "MPEG-4",
        "Ogg",
        "QuickTime",
        "WebM",
        // Audio Formats
        "MPEG-4 Audio",
        "AAC",
        "WMA",
        "MP3",
        "FLAC",
        "Wave"
    };
    return QString::fromUtf8(descriptions[int(c) + 1]);
}

QString QMediaFormat::audioCodecName(QMediaFormat::AudioCodec c)
{
    constexpr const char *descriptions[] = {
        "Invalid",
        "MP3",
        "AAC",
        "AC3",
        "EAC3",
        "FLAC",
        "DolbyTrueHD",
        "Opus",
        "Vorbis",
        "Wave",
        "WMA",
        "ALAC",
    };
    return QString::fromUtf8(descriptions[int(c) + 1]);
}

QString QMediaFormat::videoCodecName(QMediaFormat::VideoCodec c)
{
    constexpr const char *descriptions[] = {
        "Invalid",
        "MPEG1",
        "MPEG2",
        "MPEG4",
        "H264",
        "H265",
        "VP8",
        "VP9",
        "AV1",
        "Theora",
        "WMV",
        "MotionJPEG"
    };
    return QString::fromUtf8(descriptions[int(c) + 1]);
}

QString QMediaFormat::fileFormatDescription(QMediaFormat::FileFormat c)
{
    constexpr const char *descriptions[QMediaFormat::LastFileFormat + 2] = {
        "Unspecified File Format",
        "Windows Media Video",
        "Audio Video Interleave",
        "Matroska Multimedia Container",
        "MPEG-4 Video Container",
        "Ogg",
        "QuickTime Container",
        "WebM",
        // Audio Formats
        "MPEG-4 Audio",
        "AAC",
        "Windows Media Audio",
        "MP3",
        "Free Lossless Audio Codec (FLAC)",
        "Wave File"
    };
    return QString::fromUtf8(descriptions[int(c) + 1]);
}

QString QMediaFormat::audioCodecDescription(QMediaFormat::AudioCodec c)
{
    constexpr const char *descriptions[] = {
        "Unspecified Audio Codec",
        "MP3",
        "Advanced Audio Codec (AAC)",
        "Dolby Digital (AC3)",
        "Dolby Digital Plus (E-AC3)",
        "Free Lossless Audio Codec (FLAC)",
        "Dolby True HD",
        "Opus",
        "Vorbis",
        "Wave",
        "Windows Media Audio",
        "Apple Lossless Audio Codec (ALAC)",
    };
    return QString::fromUtf8(descriptions[int(c) + 1]);
}

QString QMediaFormat::videoCodecDescription(QMediaFormat::VideoCodec c)
{
    constexpr const char *descriptions[] = {
        "Unspecified Video Codec",
        "MPEG-1 Video",
        "MPEG-2 Video",
        "MPEG-4 Video",
        "H.264",
        "H.265",
        "VP8",
        "VP9",
        "AV1",
        "Theora",
        "Windows Media Video",
        "MotionJPEG"
    };
    return QString::fromUtf8(descriptions[int(c) + 1]);
}

bool QMediaFormat::operator==(const QMediaFormat &other) const
{
    Q_ASSERT(!d);
    return fmt == other.fmt &&
            audio == other.audio &&
           video == other.video;
}

/*!
    Resolves the format to a format that is supported by QMediaRecorder.

    This method tries to find the best possible match for unspecified settings.
    Settings that are not supported by the recorder will be modified to the closest
    match that is supported.

    When resolving, priority is given in the following order:
    \list 1
    \li File format
    \li Video codec
    \li Audio codec
    \endlist
*/
void QMediaFormat::resolveForEncoding(ResolveFlags flags)
{
    const bool requiresVideo = (flags & ResolveFlags::RequiresVideo) != 0;

    if (!requiresVideo)
        video = VideoCodec::Unspecified;

    // need to adjust the format. Priority is given first to file format, then video codec, then audio codec

    QMediaFormat nullFormat;
    auto supportedFormats = nullFormat.supportedFileFormats(QMediaFormat::Encode);
    auto supportedAudioCodecs = nullFormat.supportedAudioCodecs(QMediaFormat::Encode);
    auto supportedVideoCodecs = nullFormat.supportedVideoCodecs(QMediaFormat::Encode);

    auto bestSupportedFileFormat = [&](QMediaFormat::AudioCodec audio = QMediaFormat::AudioCodec::Unspecified,
                                       QMediaFormat::VideoCodec video = QMediaFormat::VideoCodec::Unspecified)
    {
        QMediaFormat f;
        f.setAudioCodec(audio);
        f.setVideoCodec(video);
        auto supportedFormats = f.supportedFileFormats(QMediaFormat::Encode);
        auto *list = (flags == NoFlags) ? audioFormatPriorityList : videoFormatPriorityList;
        while (*list != QMediaFormat::UnspecifiedFormat) {
            if (supportedFormats.contains(*list))
                break;
            ++list;
        }
        return *list;
    };

    // reset format if it does not support video when video is required
    if (requiresVideo && this->supportedVideoCodecs(QMediaFormat::Encode).isEmpty())
        fmt = QMediaFormat::UnspecifiedFormat;

    // reset non supported formats and codecs
    if (!supportedFormats.contains(fmt))
        fmt = QMediaFormat::UnspecifiedFormat;
    if (!supportedAudioCodecs.contains(audio))
        audio = QMediaFormat::AudioCodec::Unspecified;
    if (!requiresVideo || !supportedVideoCodecs.contains(video))
        video = QMediaFormat::VideoCodec::Unspecified;

    if (requiresVideo) {
        // try finding a file format that is supported
        if (fmt == QMediaFormat::UnspecifiedFormat)
            fmt = bestSupportedFileFormat(audio, video);
        // try without the audio codec
        if (fmt == QMediaFormat::UnspecifiedFormat)
            fmt = bestSupportedFileFormat(QMediaFormat::AudioCodec::Unspecified, video);
    }
    // try without the video codec
    if (fmt == QMediaFormat::UnspecifiedFormat)
        fmt = bestSupportedFileFormat(audio);
    // give me a format that's supported
    if (fmt == QMediaFormat::UnspecifiedFormat)
        fmt = bestSupportedFileFormat();
    // still nothing? Give up
    if (fmt == QMediaFormat::UnspecifiedFormat) {
        *this = {};
        return;
    }

    // find a working video codec
    if (requiresVideo) {
        // reset the audio codec, so that we won't throw away the video codec
        // if it is supported (choosing the specified video codec has higher
        // priority than the specified audio codec)
        auto a = audio;
        audio = QMediaFormat::AudioCodec::Unspecified;
        auto videoCodecs = this->supportedVideoCodecs(QMediaFormat::Encode);
        if (!videoCodecs.contains(video)) {
            // not supported, try to find a replacement
            auto *list = videoPriorityList;
            while (*list != QMediaFormat::VideoCodec::Unspecified) {
                if (videoCodecs.contains(*list))
                    break;
                ++list;
            }
            video = *list;
        }
        audio = a;
    } else {
        video = QMediaFormat::VideoCodec::Unspecified;
    }

    // and a working audio codec
    auto audioCodecs = this->supportedAudioCodecs(QMediaFormat::Encode);
    if (!audioCodecs.contains(audio)) {
        auto *list = audioPriorityList;
        while (*list != QMediaFormat::AudioCodec::Unspecified) {
            if (audioCodecs.contains(*list))
                break;
            ++list;
        }
        audio = *list;
    }
}

QT_END_NAMESPACE
