#ifndef _MLX_BITOPS_H
#define _MLX_BITOPS_H

/*
 * Copyright (C) 2007 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

FILE_LICENCE ( GPL2_OR_LATER );

/**
 * @file
 *
 * Mellanox bit operations
 *
 */

/* Datatype used to represent a bit in the Mellanox autogenerated headers */
typedef unsigned char pseudo_bit_t;

/**
 * Wrapper structure for pseudo_bit_t structures
 *
 * This structure provides a wrapper around the autogenerated
 * pseudo_bit_t structures.  It has the correct size, and also
 * encapsulates type information about the underlying pseudo_bit_t
 * structure, which allows the MLX_FILL etc. macros to work without
 * requiring explicit type information.
 */
#define MLX_DECLARE_STRUCT( _structure )				     \
	_structure {							     \
	    union {							     \
		uint8_t bytes[ sizeof ( struct _structure ## _st ) / 8 ];    \
		uint32_t dwords[ sizeof ( struct _structure ## _st ) / 32 ]; \
		struct _structure ## _st *dummy[0];			     \
	    } __attribute__ (( packed )) u;				     \
	} __attribute__ (( packed ))

/** Get pseudo_bit_t structure type from wrapper structure pointer */
#define MLX_PSEUDO_STRUCT( _ptr )					     \
	typeof ( *((_ptr)->u.dummy[0]) )

/** Bit offset of a field within a pseudo_bit_t structure */
#define MLX_BIT_OFFSET( _structure_st, _field )				     \
	offsetof ( _structure_st, _field )

/** Dword offset of a field within a pseudo_bit_t structure */
#define MLX_DWORD_OFFSET( _structure_st, _field )			     \
	( MLX_BIT_OFFSET ( _structure_st, _field ) / 32 )

/** Dword bit offset of a field within a pseudo_bit_t structure
 *
 * Yes, using mod-32 would work, but would lose the check for the
 * error of specifying a mismatched field name and dword index.
 */
#define MLX_DWORD_BIT_OFFSET( _structure_st, _index, _field )		     \
	( MLX_BIT_OFFSET ( _structure_st, _field ) - ( 32 * (_index) ) )

/** Bit width of a field within a pseudo_bit_t structure */
#define MLX_BIT_WIDTH( _structure_st, _field )				     \
	sizeof ( ( ( _structure_st * ) NULL )->_field )

/** Bit mask for a field within a pseudo_bit_t structure */
#define MLX_BIT_MASK( _structure_st, _field )				     \
	( ( ~( ( uint32_t ) 0 ) ) >>					     \
	  ( 32 - MLX_BIT_WIDTH ( _structure_st, _field ) ) )

/*
 * Assemble native-endian dword from named fields and values
 *
 */

#define MLX_ASSEMBLE_1( _structure_st, _index, _field, _value )		     \
	( (_value) << MLX_DWORD_BIT_OFFSET ( _structure_st, _index, _field ) )

#define MLX_ASSEMBLE_2( _structure_st, _index, _field, _value, ... )	     \
	( MLX_ASSEMBLE_1 ( _structure_st, _index, _field, _value ) |	     \
	  MLX_ASSEMBLE_1 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_ASSEMBLE_3( _structure_st, _index, _field, _value, ... )	     \
	( MLX_ASSEMBLE_1 ( _structure_st, _index, _field, _value ) |	     \
	  MLX_ASSEMBLE_2 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_ASSEMBLE_4( _structure_st, _index, _field, _value, ... )	     \
	( MLX_ASSEMBLE_1 ( _structure_st, _index, _field, _value ) |	     \
	  MLX_ASSEMBLE_3 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_ASSEMBLE_5( _structure_st, _index, _field, _value, ... )	     \
	( MLX_ASSEMBLE_1 ( _structure_st, _index, _field, _value ) |	     \
	  MLX_ASSEMBLE_4 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_ASSEMBLE_6( _structure_st, _index, _field, _value, ... )	     \
	( MLX_ASSEMBLE_1 ( _structure_st, _index, _field, _value ) |	     \
	  MLX_ASSEMBLE_5 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_ASSEMBLE_7( _structure_st, _index, _field, _value, ... )	     \
	( MLX_ASSEMBLE_1 ( _structure_st, _index, _field, _value ) |	     \
	  MLX_ASSEMBLE_6 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_ASSEMBLE_8( _structure_st, _index, _field, _value, ... )	     \
	( MLX_ASSEMBLE_1 ( _structure_st, _index, _field, _value ) |	     \
	  MLX_ASSEMBLE_7 ( _structure_st, _index, __VA_ARGS__ ) )

/*
 * Build native-endian (positive) dword bitmasks from named fields
 *
 */

#define MLX_MASK_1( _structure_st, _index, _field )			     \
	( MLX_BIT_MASK ( _structure_st, _field ) <<			     \
	  MLX_DWORD_BIT_OFFSET ( _structure_st, _index, _field ) )

#define MLX_MASK_2( _structure_st, _index, _field, ... )		     \
	( MLX_MASK_1 ( _structure_st, _index, _field ) |		     \
	  MLX_MASK_1 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_MASK_3( _structure_st, _index, _field, ... )		     \
	( MLX_MASK_1 ( _structure_st, _index, _field ) |		     \
	  MLX_MASK_2 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_MASK_4( _structure_st, _index, _field, ... )		     \
	( MLX_MASK_1 ( _structure_st, _index, _field ) |		     \
	  MLX_MASK_3 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_MASK_5( _structure_st, _index, _field, ... )		     \
	( MLX_MASK_1 ( _structure_st, _index, _field ) |		     \
	  MLX_MASK_4 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_MASK_6( _structure_st, _index, _field, ... )		     \
	( MLX_MASK_1 ( _structure_st, _index, _field ) |		     \
	  MLX_MASK_5 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_MASK_7( _structure_st, _index, _field, ... )		     \
	( MLX_MASK_1 ( _structure_st, _index, _field ) |		     \
	  MLX_MASK_6 ( _structure_st, _index, __VA_ARGS__ ) )

#define MLX_MASK_8( _structure_st, _index, _field, ... )		     \
	( MLX_MASK_1 ( _structure_st, _index, _field ) |		     \
	  MLX_MASK_7 ( _structure_st, _index, __VA_ARGS__ ) )

/*
 * Populate big-endian dwords from named fields and values
 *
 */

#define MLX_FILL( _ptr, _index, _assembled )				     \
	do {								     \
		uint32_t *__ptr = &(_ptr)->u.dwords[(_index)];		     \
		uint32_t __assembled = (_assembled);			     \
		*__ptr = cpu_to_be32 ( __assembled );			     \
	} while ( 0 )

#define MLX_FILL_1( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_1 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

#define MLX_FILL_2( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_2 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

#define MLX_FILL_3( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_3 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

#define MLX_FILL_4( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_4 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

#define MLX_FILL_5( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_5 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

#define MLX_FILL_6( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_6 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

#define MLX_FILL_7( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_7 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

#define MLX_FILL_8( _ptr, _index, ... )					     \
	MLX_FILL ( _ptr, _index, MLX_ASSEMBLE_8 ( MLX_PSEUDO_STRUCT ( _ptr ),\
						  _index, __VA_ARGS__ ) )

/*
 * Modify big-endian dword using named field and value
 *
 */

#define MLX_SET( _ptr, _field, _value )					     \
	do {								     \
		unsigned int __index = 					     \
		    MLX_DWORD_OFFSET ( MLX_PSEUDO_STRUCT ( _ptr ), _field ); \
		uint32_t *__ptr = &(_ptr)->u.dwords[__index];		     \
		uint32_t __value = be32_to_cpu ( *__ptr );		     \
		__value &= ~( MLX_MASK_1 ( MLX_PSEUDO_STRUCT ( _ptr ),	     \
					   __index, _field ) );		     \
		__value |= MLX_ASSEMBLE_1 ( MLX_PSEUDO_STRUCT ( _ptr ),	     \
					    __index, _field, _value );	     \
		*__ptr = cpu_to_be32 ( __value );			     \
	} while ( 0 )

/*
 * Extract value of named field
 *
 */

#define MLX_GET( _ptr, _field )						     \
	( {								     \
		unsigned int __index = 					     \
		    MLX_DWORD_OFFSET ( MLX_PSEUDO_STRUCT ( _ptr ), _field ); \
		uint32_t *__ptr = &(_ptr)->u.dwords[__index];		     \
		uint32_t __value = be32_to_cpu ( *__ptr );		     \
		__value >>=						     \
		    MLX_DWORD_BIT_OFFSET ( MLX_PSEUDO_STRUCT ( _ptr ),	     \
					    __index, _field );		     \
		__value &=						     \
		    MLX_BIT_MASK ( MLX_PSEUDO_STRUCT ( _ptr ), _field );     \
		__value;						     \
	} )

/*
 * Fill high dword of physical address, if necessary
 *
 */
#define MLX_FILL_H( _structure_st, _index, _field, _address ) do {	     \
	if ( sizeof ( physaddr_t ) > sizeof ( uint32_t ) ) {		     \
		MLX_FILL_1 ( _structure_st, _index, _field,		     \
			     ( ( ( uint64_t ) (_address) ) >> 32 ) );	     \
	} } while ( 0 )

#endif /* _MLX_BITOPS_H */
