/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __QZIONTEXT_P_H__
#define __QZIONTEXT_P_H__

#include <QGraphicsItem>
#include <QAbstractGraphicsShapeItem>

#include "qziontext.h"
#include "qzionrectangle.h"
#include "qzioncommon.h"

class QGraphicsQZionTextItem : public QAbstractGraphicsShapeItem
{
public:
    enum TextStyle { NoStyle, OutLine, Shadow };

    QGraphicsQZionTextItem(QGraphicsItem *parent = 0);

    QRectF boundingRect () const;
    virtual void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                       QWidget *widget = 0);

    QString text() const;
    void setText(const QString &text);

    void setStyle(TextStyle style);
    TextStyle style() const;

    QColor textColor() const;
    void setTextColor(const QColor &color);
    QColor outlineColor() const;
    void setOutlineColor(const QColor &color);
    QColor shadowColor() const;
    void setShadowColor(const QColor &color);

    QFont font() const;
    void setFont(const QFont &font);

    QSize size() const;
    void setSize(const QSize &size);

    QPointF alignment() const;
    void setAlignment(const QPointF &alignment);

    qreal ellipsisAlignment() const;
    void setEllipsisAlignment(const qreal alignment);

    bool contains(const QPointF &point) const;

private:
    void updateBoundingRect();
    void drawTextWithOutline(QPainter *painter, const QRectF rect, int flags, QString text);

    void drawTextWithShadow(QPainter *painter, const QRectF rect, int flags, QString text);

    QString _text;
    QFont _font;
    QRectF _boundingRect;
    QColor _color;
    QColor _color2;
    QColor _color3;

    QSize _size;
    QPointF _alignment;
    qreal _ellipsisAlignment;
    TextStyle _style;
};


class QZionTextPrivate : public QGraphicsQZionTextItem
{
public:
    QZionText *owner;
    QColor baseColor;
    QColor myTextColor;
    QColor myOutlineColor;
    QColor myShadowColor;

    QZionTextPrivate(QGraphicsItem *parent);

    inline virtual void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                       QWidget *widget = 0);

    inline void updateColors();

    QGV_EFFECTIVE_SHAPE(QGraphicsQZionTextItem);
};

inline QZionTextPrivate::QZionTextPrivate(QGraphicsItem *parent = 0)
    : QGraphicsQZionTextItem(parent), baseColor(Qt::white),
      myTextColor(Qt::white), myOutlineColor(Qt::white),
      myShadowColor(Qt::white)
{
        setFlag(QGraphicsItem::ItemClipsToShape, true);
}


void QZionTextPrivate::paint(QPainter *painter,
                             const QStyleOptionGraphicsItem *option,
                             QWidget *widget)
{
    // we assume no one else has changed opacity
    int opacity = owner->effectiveColor().alpha();

    if (opacity >= 255) {
        QGraphicsQZionTextItem::paint(painter, option, widget);
    } else {
        painter->setOpacity(opacity / (float) 255);
        QGraphicsQZionTextItem::paint(painter, option, widget);
        painter->setOpacity(1.0);
    }
}

void QZionTextPrivate::updateColors()
{
    // alpha composition will be done later, in paint time
    QColor effectiveColor = QColor(owner->effectiveColor().rgb());

    // Call low level QGraphicsZionTextItem setColor methods
    setTextColor(composeColorClass(myTextColor, effectiveColor));
    setOutlineColor(composeColorClass(myOutlineColor, effectiveColor));
    setShadowColor(composeColorClass(myShadowColor, effectiveColor));
}

#endif
