#! /usr/bin/python3
# -*- coding: utf-8 -*-
import imp
import os
import shutil
import tempfile
from unittest import TestCase
from gi.repository import Unity


class ResultSet(Unity.ResultSet):
    def __init__(self):
        Unity.ResultSet.__init__(self)
        self.results = []

    def do_add_result(self, result):
        self.results.append({'uri':result.uri,
                             'title':result.title,
                             'comment':result.comment,
                             'icon':result.icon_hint})


class ScopeTestCase(TestCase):
    def init_scope(self, scope_path):
        self.scope_module = imp.load_source('scope', scope_path)
        self.scope = self.scope_module.load_scope()

    def perform_query(self, query, filter_set = Unity.FilterSet.new()):
        result_set = ResultSet()
        ctx = Unity.SearchContext.create(query, 0, filter_set,
                                         None, result_set, None)
        s = self.scope.create_search_for_query(ctx)
        s.run()
        return result_set

class WallpaperTestScope(Unity.AbstractScope):
    def set_as_wallpaper(pattern):
        return

class TestColourlovers(ScopeTestCase):
    def setUp(self):
        self.init_scope('src/unity_colourlovers_daemon.py')
        self.cache_dir = tempfile.mkdtemp()

    def tearDown(self):
        self.scope = None
        self.scope_module = None
        shutil.rmtree(self.cache_dir, ignore_errors=True)

    def test_valid_searches(self):
        self.scope_module.SEARCH_URI = 'file:tests/data/mock_colourlovers_pass#'
        expected_results = ['tiny paper umbrella',
                            'paparadsha',
                            'http://www.colourlovers.com/color/F8F671/tiny_paper_umbrella',
                            'http://www.colourlovers.com/img/F8F671/100/100/tiny_paper_umbrella.png']
        results = []
        for s in ['query']:
            result_set = self.perform_query(s)
            results.append(result_set.results[0]['title'])
            results.append(result_set.results[0]['comment'])
            results.append(result_set.results[0]['uri'])
            results.append(result_set.results[0]['icon'])
        self.assertEqual(results, expected_results)

    def test_preview(self):
        result = Unity.ScopeResult()
        result.title = 'Fancy Title'
        result.comment = 'author'
        result.icon_hint = 'http://www.colourlovers.com/img/fancy_pattern.png'
        scope = self.scope_module.Scope()
        previewer = scope.create_previewer(result, Unity.SearchMetadata())
        self.assertEqual(previewer.run().props.title, result.title)
        self.assertEqual(previewer.run().props.subtitle, 'By '+result.comment)
        self.assertEqual(previewer.run().props.image_source_uri, result.icon_hint)

    def test_wallpaper_caching(self):
        self.scope_module.CACHE = self.cache_dir
        patterns = []
        def set_as_wallpaper(pattern):
            patterns.append(pattern)
        self.scope.set_as_wallpaper = set_as_wallpaper
        self.scope.wallpaper_action('file:tests/data/ubuntu-raptor.png')
        local_path = os.path.join(self.cache_dir, 'ubuntu-raptor.png')
        self.assertEqual(os.path.isfile(local_path), True)
        self.assertEqual(patterns, [local_path])

    def test_failing_search(self):
        self.scope_module.SEARCH_URI = 'file:tests/data/mock_colourlovers_fail#'
        for s in ['query']:
            result_set = self.perform_query(s)
            self.assertEqual(len(result_set.results), 0)


if __name__ == '__main__':
    unittest.main()
