/*
 * PedroM - Operating System for Ti-89/Ti-92+/V200.
 * Copyright (C) 2003, 2004, 2005 Patrick Pelissier
 *
 * This program is free software ; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program;
 * if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "PedroM-Internal.h"

#define TAB_KEY		9

typedef enum {
  FLAG_LJUSTIFY=1, FLAG_PRECGIVEN=2, FLAG_BLANKER=4, FLAG_VARIANT=8, FLAG_PADZERO=0x10,
  FLAG_SIGNED=0x20, FLAG_LONG=0x40, FLAG_SHORT=0x80, FLAG_FPCONV=0x100
} flags_e;

typedef struct {
  flags_e flags;
  int precision, width;
  const char *prefix, *hextab;
  short (*putc)(short, FILE *);
  float f;
  unsigned long v;
} vcbprintf_display_rec;

#define PUTC(dr, ch, f) ((dr)->putc(ch, f))

#define PADDING(dr, ch, n, f)  \
  if (n > 0) {charcount +=n; while (--n>=0) PUTC(dr, ch, f);}

/* Display a float or a long */
/*static int inline
  vcbprintf_display_number (FILE *f, int ch, vcbprintf_display_rec *dr) __attribute__((always_inline)); */

static int
vcbprintf_display_number (FILE *f, int ch, vcbprintf_display_rec *dr)
{
  unsigned long v = dr->v;
  flags_e flags = dr->flags;
  int precision = dr->precision;
  int width = dr->width;
  bcd *b = (bcd*) (&(dr->f));
  const char *prefix = dr->prefix;
  const char *hextab = dr->hextab;

  int i, len, charcount;
  short expo;
  char sign, bzero;
  char c;
  char buffer[32];

  len = charcount = 0;
  if ((flags & (FLAG_FPCONV|FLAG_PRECGIVEN)) != 0)
    precision = 1;

  /* Parse the number and fill buffer with the reverse digits */
  switch (ch)
    {
    case 'p':
    case 'X':
    case 'x':
      while(v != 0)
	{
	  buffer[len++] = hextab[v & 0xf];
	  v = v >> 4;
	}
      break;

    case 'o':
      while(v != 0)
	{
	  buffer[len++] = '0' + (v & 7);
	  v = v >> 3;
	}
      if (flags & FLAG_VARIANT)
	dr->precision--;
      break;

    case 'b':
      while(v != 0)
	{
	  buffer[len++] = '0' + (v & 1);
	  v = v >> 1;
	}
      break;

    case 'u':
    case 'i':
    case 'd':
      while(v != 0)
	{
	  buffer[len++] = '0' + (char)(v % 10);
	  v = v / 10;
	}
      break;

    default:    /* Float */
      expo = (b->exponent & 0x7FFF) - 0x4000;
      if (expo == (0x7FFF- 0x4000)) {
        buffer[len++] = 'f';
        buffer[len++] = 'e';
        buffer[len++] = 'd';
        buffer[len++] = 'n';
        buffer[len++] = 'u';
      } else if (expo == (0x2000-0x4000))
        buffer[len++] = '0';
      else if (expo == (0x6000-0x4000)) {
        buffer[len++] = 190;
        if (b->exponent & 0x8000)
          buffer[len++] = '-';
        else
          buffer[len++] = '+';
      } else if (expo >=-1 && expo <5 && ch == 'f') {
        /* Normal print */
        bzero = 0;
        v = b->mantissa2;
        for(i = 0 ; i < 8 ; i++) {
          if (bzero || (v &0x0F)) {
            buffer[len++] = (v & 0xF) + '0';
            bzero = 1;
          }
          v >>= 4;
        }
        v = b->mantissa1;
        for(i = 0 ; i < (7-expo) ; i++) {
          if (bzero || (v &0x0F)) {
            buffer[len++] = (v & 0xF) + '0';
            bzero = 1;
          }
          v >>= 4;
        }
        if (bzero)
          buffer[len++] = '.';
        for( ; i < 8 ; i++) {
          buffer[len++] = (v & 0xF) + '0';
          v >>= 4;
        }
        if (b->exponent & 0x8000)
          buffer[len++] = 175;
      } else {
	/* Scientific print */
	sign = 0;
	if (expo < 0) {
          expo=-expo;
          sign = 1;
        }
	do  {
	  buffer[len++] = '0' + (char)(expo % 10);
	  expo = expo / 10;
	} while (expo != 0) ;
	if (sign)
	  buffer[len++] ='-';
	buffer[len++]=149;
	/* Mantisse */
	bzero = 0;
	v = b->mantissa2;
	for(i = 0 ; i < 8 ; i++) {
          if (bzero || (v &0x0F)) {
            buffer[len++] = (v & 0xF) + '0';
            bzero = 1;
          }
          v >>= 4;
        }
	v = b->mantissa1;
	for(i = 0 ; i < 7 ; i++) {
          if (bzero || (v &0x0F)) {
            buffer[len++] = (v & 0xF) + '0';
            bzero = 1;
          }
          v >>= 4;
        }
	if (bzero)
	  buffer[len++] = '.';
	buffer[len++] = (v & 0xF) + '0';
	if (b->exponent & 0x8000)
	  buffer[len++] = 175;
      }
      break;
    }

  if (v == 0 && len == 0)
    buffer[len++] = '0';

  if ((precision -= len)<0)
      precision = 0;
  width -= (precision + len + strlen (prefix));

  /* Print Number in FILE f, adding prefix and padding if necessary */
  if ((flags & (FLAG_PADZERO|FLAG_LJUSTIFY)) == 0) {
    c = flags & FLAG_PADZERO ? '0' : ' ';
    PADDING (dr, c, width, f);
  }
  for (; (c=*prefix++)!=0; charcount++)
    PUTC (dr, c, f);
  if ((flags&FLAG_LJUSTIFY) == 0) {
    c = flags & FLAG_PADZERO ? '0' : ' ';
    PADDING (dr, c, width, f);
  }
  PADDING (dr, '0', precision, f);
  charcount += len;
  while (len-- > 0)
    PUTC (dr, buffer[len], f);
  if (flags & FLAG_LJUSTIFY)
    PADDING (dr, ' ', width, f);
  return charcount;
}

int
vcbprintf (short (*callback)(short,FILE *), FILE *p, const char *fmt, va_list args)
{
  vcbprintf_display_rec dr;
  long w;
  int charcount;
  flags_e flags;
  char  ch;

  charcount = 0;
  dr.putc = callback;
  while ( (ch = *fmt++) != 0)
    {
      if (ch != '%') /* Normal char: print it */
	{
	  PUTC (&dr, ch, p);
	  charcount++;
	}
      else
        {
	  /* Read flags (Ti special flags not supported!) */
	  flags = 0;
	  for (;;) {
	    switch (ch = *fmt++)
	      {
	      case '+':
		flags |= FLAG_SIGNED;
		continue;
	      case '-':
		flags = FLAG_LJUSTIFY | (flags & ~FLAG_PADZERO);
		continue;
	      case '#':
		flags |= FLAG_VARIANT;
		continue;
	      case ' ':
		flags |= FLAG_BLANKER;
		continue;
	      case '0':
		flags |= FLAG_PADZERO;
		continue;
	      case 'z':
	      case '^':
	      case '|':	/* Ti special Format, not yet supported */
		continue;
	      default:
		break;
	      }
	    break;
	  } /* for(;;) */

	  /* Read width field */
	  dr.width = 0;
	  if (ch=='*')
	    {
	      dr.width = va_arg (args, int);
	      ch = *fmt++;
	    }
	  else while (isdigit (ch))
	    {
	      dr.width = dr.width*10 + ch - '0';
	      ch = *fmt++;
	    }
	  if (dr.width < 0)
	    {
	      dr.width = -dr.width;
	      flags ^= FLAG_LJUSTIFY;
	    }

	  /* Read precision */
	  dr.precision = 0;
	  if (ch == '.')
	    {
	      ch = *fmt++;
	      if (ch == '*')
		{
		  dr.precision = va_arg (args, int);
		  ch = *fmt++;
		}
	      else if (ch =='-')	/* -1 (Ti specific) */
		{
		  ch = *fmt++;
		  if (ch == '1')
		    {
		      dr.precision = 6;
		      ch = *fmt++;
		    }
		  else
		    dr.precision = -1;
		}
	      else while (isdigit(ch))
		{
		  dr.precision = dr.precision*10 + ch - '0';
		  ch = *fmt++;
		}
	      if (dr.precision >= 0)
		flags |= FLAG_PRECGIVEN;
	      else
		dr.precision = 0;
	    }

	  /* Read Short or Long? */
	  if (ch=='l' || ch=='L')
	    {
	      flags |= FLAG_LONG;
	      ch = *fmt++;
	    }
	  else if (ch=='h' || ch=='H')
	    {
	      flags |= FLAG_SHORT;
	      ch = *fmt++;
	    }
	  else
	    /* Default is short */
	    flags |= FLAG_SHORT;

	  /* Get the var */
	  dr.v = 0;
	  dr.prefix = "";
	  switch (ch)
            {
	    case 0:
	      fmt--;
	      continue;

	    case 's':
	      {
		const char *str = va_arg(args, const char *);
		long i, n;

		n = strlen (str);
		if (flags & FLAG_PRECGIVEN)
		  n = MIN (n, dr.precision);
		dr.width -= n;
		if ((flags&FLAG_LJUSTIFY) == 0)
		  {
		    ch = flags & FLAG_PADZERO ? '0' : ' ';
		    PADDING (&dr, ch, dr.width, p);
		  }
		charcount += n;
		for (i=0; i<n; i++)
		  PUTC (&dr, str[i], p);
		if (flags&FLAG_LJUSTIFY)
		  PADDING (&dr, ' ', dr.width, p);
		continue;
	      }

	    case 'p':
	      dr.v = (unsigned long) va_arg(args, const void *);
	      dr.hextab = "0123456789abcdef";
	      dr.prefix = (flags&FLAG_VARIANT) ? "@" : "";
	      dr.precision = 6;	/* Only 24 bits avialble */
	      flags |= FLAG_PRECGIVEN;
	      break;

	    case 'X':
	      dr.hextab = "0123456789ABCDEF";
	      dr.prefix = (flags&FLAG_VARIANT) ? "0X" : "";
	      goto ReadNumber;
	    case 'x':
	      dr.hextab = "0123456789abcdef";
	      dr.prefix = (flags&FLAG_VARIANT) ? "0x" : "";
	      goto ReadNumber;
	    case 'b':
	      dr.prefix = (flags&FLAG_VARIANT) ? "0b" : "";
	      goto ReadNumber;
	    case 'o':
	      dr.prefix = (flags&FLAG_VARIANT) ? "0" : "";
	      goto ReadNumber;
	    case 'u':
	      dr.prefix = "";
	    ReadNumber:
	      if (flags & FLAG_LONG)
		dr.v = va_arg (args, unsigned long);
	      else
		dr.v = va_arg (args, unsigned short);
	      if (flags & FLAG_PRECGIVEN)
		flags &= ~FLAG_PADZERO;
	      break;

	    case 'i':
	    case 'd':
	      if (flags & FLAG_LONG)
		w = va_arg(args, signed long);
	      else
		w = va_arg(args, signed short);
	      dr.v = (w < 0) ? -w : w;
	      dr.prefix = (w < 0) ? "-" : (flags & FLAG_SIGNED) ? "+" :
		(flags & FLAG_BLANKER) ? " " : "";
	      if (flags & FLAG_PRECGIVEN)
		flags &= ~FLAG_PADZERO;
	      break;

	    case 'f':
	    case 'F':
	    case 'e':
	    case 'E':
	    case 'g':
	    case 'G':
	    case 'r':
	    case 'R':
	    case 'y':
	    case 'Y':
	      flags |= FLAG_FPCONV;
	      if ((flags & FLAG_PRECGIVEN) == 0)
		dr.precision = 1;
	      dr.f = va_arg (args, double);
	      dr.prefix = (flags&FLAG_SIGNED) ? "+" : (flags&FLAG_BLANKER) ? " " : "";
	      break;

	    case 'c':
	      ch = va_arg(args, int);

	    default:
	      dr.width--;                        /* char width is 1 */
	      if ((flags & FLAG_LJUSTIFY) == 0)
		{
		  const char c = flags & FLAG_PADZERO ? '0' : ' ';
		  PADDING (&dr, c, dr.width, p);
		}
	      charcount++;
	      PUTC (&dr, ch, p);
	      if (flags&FLAG_LJUSTIFY)
		PADDING (&dr, ' ', dr.width, p);
	      continue;
            }
	  /* Print the number */
	  dr.flags = flags;
	  charcount += vcbprintf_display_number (p, ch, &dr);
	  continue;
        }
    }
  return charcount;
}

short
_sputc(short ch, FILE *fp)
{
  char **op = (char **) fp;
  *((*op)++) = ch;
  return ch;
}

int
sprintf(char *buff, const char *fmt, ...)
{
  char *sf = buff;
  va_list a;
  int length;
  va_start(a, fmt);
  asm ("nop\n nop\n nop");
  length = vcbprintf(_sputc, (FILE *)(((void*)(&sf))), fmt, a);
  *sf = 0;
  va_end(a);
  return length;
}

short
PrintChar (short ch)
{
  short	Size = FontCharWidth (ch);
  short	x = CURRENT_POINT_X, y = CURRENT_POINT_Y;

  if (ch == TAB_KEY)		// Tab code
    {
      CURRENT_POINT_X = ((x/TabSize)+1)*TabSize;
      return ch;
    }
  if (ch == '\r')
    ch = '\n';
  if ((ch == '\n') || (x + Size > (GET_XMAX+1)))
    {
      short Height = (CURRENT_FONT*2+6);	// NewLine
      y += Height;
      if (y + Height > GET_YMAX)
	{
	  ScrRectScroll (&ScrRect, &ScrRect, Height, 0);
	  y -= Height;
	}
      if (++PRINTF_LINE_COUNTER > 13)
	{
	  ST_busy (2);		// Display 'Pause'
	  while (!PID_CheckSwitch (GetKey(), ARGV[0]));
	  ST_busy (0);		// Return to normal mode
	  PRINTF_LINE_COUNTER = 0;
	}
      x = 0;
    }
  if (ch != '\n')
    {
      DrawChar(x, y, ch, 4);
      x+=Size;
    }
  CURRENT_POINT_X = x;
  CURRENT_POINT_Y = y;
  return ch;
}

